<?php

declare(strict_types=1);

namespace DataNormalizer\PHPStan;

use Dalten\DataNormalizer\DataNormalizer;
use PhpParser\Node\Expr\MethodCall;
use PHPStan\Analyser\Scope;
use PHPStan\Reflection\MethodReflection;
use PHPStan\Type\ArrayType;
use PHPStan\Type\BooleanType;
use PHPStan\Type\Constant\ConstantArrayTypeBuilder;
use PHPStan\Type\Constant\ConstantStringType;
use PHPStan\Type\FloatType;
use PHPStan\Type\IntegerType;
use PHPStan\Type\MixedType;
use PHPStan\Type\ObjectType;
use PHPStan\Type\StringType;
use PHPStan\Type\Type;
use PHPStan\Type\TypeCombinator;

class DataNormalizerDynamicReturnTypeExtension implements \PHPStan\Type\DynamicMethodReturnTypeExtension
{
    public function getClass(): string
    {
        return DataNormalizer::class;
    }

    public function isMethodSupported(MethodReflection $methodReflection): bool
    {
        return 'normalizeData' === $methodReflection->getName();
    }

    public function getTypeFromMethodCall(MethodReflection $methodReflection, MethodCall $methodCall, Scope $scope): Type
    {
        $structureArg = $methodCall->args[0]->value ?? null;
        if (null === $structureArg) {
            return new MixedType();
        }

        $varType = $scope->getType($structureArg);

        if (!$varType instanceof \PHPStan\Type\Constant\ConstantArrayType) {
            return new MixedType();
        }

        $builder = ConstantArrayTypeBuilder::createEmpty();

        foreach ($varType->getKeyTypes() as $i => $keyType) {
            if ($keyType instanceof ConstantStringType) {
                $key = $keyType->getValue();
                $valueType = $varType->getValueTypes()[$i];

                if ($valueType instanceof ConstantStringType) {
                    $builder->setOffsetValueType($keyType, $this->parseValueType($valueType->getValue()));
                } else {
                    return new ArrayType(new StringType(), new MixedType());
                }
            } else {
                return new ArrayType(new StringType(), new MixedType());
            }
        }

        return $builder->getArray();
    }

    private function parseValueType(string $value): Type
    {
        $type = $value;
        $isNullable = '?' === substr($type, 0, 1);
        if ($isNullable) {
            $type = substr($type, 1);
        }

        $isArray = '[]' === substr($type, -2, 2);
        if ($isArray) {
            $type = substr($type, 0, -2);
        }

        switch ($type) {
            case 'int':
                $result = new IntegerType();
                break;
            case 'string':
                $result = new StringType();
                break;
            case 'float':
                $result = new FloatType();
                break;
            case 'bool':
                $result = new BooleanType();
                break;
            case 'array':
            case 'json_array':
            case 'serialized_array':
                $result = new ArrayType(
                    TypeCombinator::union(new StringType(), new IntegerType()), new MixedType()
                );
                break;
            case 'date':
                $result = new ObjectType(\DateTimeImmutable::class);
                break;
            default:
                $result = new MixedType();
                break;
        }

        if ($isArray) {
            $result = new ArrayType(new IntegerType(), $result);
        }

        if ($isNullable) {
            $result = TypeCombinator::addNull($result);
        }

        return $result;
    }
}
