<?php

declare(strict_types=1);

namespace Dalten\DataNormalizer;

/**
 * Slouží k normalizaci předaných dat z DB nebo odjinud.
 *
 * Vynucuje existenci položek a normalizuje je na daný datový typ.
 */
class DataNormalizer
{
    private const SIMPLE_VALUE_TYPES = ['int', 'float', 'string', 'bool', 'array'];

    /**
     * Normalizuje data dle předané definice.
     *
     * @param array $propertyDefinition definice dat (pole: [název property] => [typ])
     *
     * @return array
     *
     * @throws \Exception výjimky z převodu stringu na \Datetime, či převodu array<>scalar
     */
    public function normalizeData(array $propertyDefinition, array $data)
    {
        $normalizedData = [];

        foreach ($propertyDefinition as $propertyName => $valueType) {
            $value = $data[$propertyName] ?? null;
            $nullable = 0 === strpos($valueType, '?');
            if ($nullable) {
                $valueType = substr($valueType, 1);
            }

            $normalizedData[$propertyName] = $this->normalizeValueToType($value, $valueType, $nullable);
        }

        return $normalizedData;
    }

    /**
     * @param mixed  $value     hodnota k normalizaci
     * @param string $valueType Typ hodnoty (int, string, date...).
     * @param bool   $nullable  Může nabývat hodnoty null?
     *
     * @return array|\DateTimeImmutable|mixed|null
     */
    private function normalizeValueToType($value, string $valueType, bool $nullable)
    {
        if ($nullable && null === $value) {
            return null;
        }

        if (in_array($valueType, self::SIMPLE_VALUE_TYPES)) {
            return $this->normalizeSimpleValue($value, $valueType);
        }

        switch ($valueType) {
            case 'serialized_array':
                return (array) ($value ? unserialize($value) : []);
            case 'json_array':
                return (array) ($value ? json_decode($value, true) : []);
            case 'int[]':
                return array_map('intval', (array) $value);
            case 'string[]':
                return array_map('strval', (array) $value);
            case 'date':
                return $this->normalizeDateValue($value, $nullable);
            default:
                throw new \InvalidArgumentException(sprintf('Unknown property type "%s"!', $valueType));
        }
    }

    /**
     * @param mixed $value    hodnota k normalizaci
     * @param bool  $nullable Může nabývat hodnoty null?
     */
    private function normalizeDateValue($value, bool $nullable): ?\DateTimeImmutable
    {
        if ($value instanceof \DateTimeImmutable) {
            return clone $value;
        }
        if ($value instanceof \DateTime) {
            return \DateTimeImmutable::createFromMutable($value);
        }

        try {
            if (is_string($value)) {
                $value = trim($value);
            }
            if (is_numeric($value)) {
                return new \DateTimeImmutable('@' . $value);
            } elseif (isset($value) && is_scalar($value) && $value) {
                return new \DateTimeImmutable((string) $value);
            }

            throw new \InvalidArgumentException(sprintf('Nelze převést hodnotu "%s" na \DateTime', print_r($value, true)));
        } catch (\Exception $e) {
            if ($nullable) {
                return null;
            } else {
                throw new \InvalidArgumentException(sprintf('Nelze převést hodnotu "%s" na \DateTime', print_r($value, true)), $e->getCode(), $e);
            }
        }
    }

    /**
     * @param mixed  $value     hodnota k normalizaci
     * @param string $valueType Typ hodnoty (int, string, ...).
     *
     * @return string|int|float|bool
     */
    private function normalizeSimpleValue($value, string $valueType)
    {
        if (is_object($value)) {
            throw new \InvalidArgumentException(sprintf('Není možné převést instanci %s na datový typ "%s".', get_class($value), $valueType));
        }

        if (is_array($value) && 'array' !== $valueType) {
            throw new \InvalidArgumentException(sprintf('Není možné převést pole na datový typ "%s".', $valueType));
        }

        if (!in_array($valueType, self::SIMPLE_VALUE_TYPES)) {
            throw new \InvalidArgumentException(sprintf('Neznámý jednoduchý typ "%s". (Známé: %s.)', $valueType, implode(', ', self::SIMPLE_VALUE_TYPES)));
        }
        $normalizedValue = $value;
        if (false === @settype($normalizedValue, $valueType)) {
            // tohle by nemělo nikdy nastat, dokud budeme za skalární používat pouze (int, bool, float, string)
            // tato výjimka je tč. nedosažitelná, ale v budoucnu může ohlídat regresi
            throw new \InvalidArgumentException(sprintf('Nelze převést hodnotu "%s" na "%s".', print_r($normalizedValue, true), $valueType));
        }

        return $normalizedValue;
    }
}
