<?php

declare(strict_types=1);

namespace Dalten\DataNormalizer\Tests;

use Dalten\DataNormalizer\DataNormalizer;
use PHPUnit\Framework\TestCase;

/**
 * @covers \Dalten\DataNormalizer\DataNormalizer
 * @group unit
 */
class DataNormalizerTest extends TestCase
{
    /**
     * @param string $dataType
     * @param mixed  $raw
     * @param mixed  $normalized
     *
     * @throws \Exception
     *
     * @dataProvider simpleConversionDataProvider
     */
    public function testCanNormalizeScalarTypes(string $dataType, $raw, $normalized): void
    {
        $this->assertConversionResult(['test' => $dataType], ['test' => $raw], ['test' => $normalized]);
    }

    public function testNullValuesAreCoercedToSpecifiedTypeUnlessThatTypeIsAlsoNullable(): void
    {
        $this->assertConversionResult(['test' => 'int'], ['test' => null], ['test' => 0]);
    }

    public function testTypesBeginningWithQuestionMarkAreNullable(): void
    {
        $this->assertConversionResult(['test' => '?int'], ['test' => null], ['test' => null]);
    }

    public function testThrowsExceptionWhenUnexpectedPropertyTypeIsSpecified(): void
    {
        $this->expectException(\InvalidArgumentException::class);
        $this->expectErrorMessageMatches('/"undefined"/');
        (new DataNormalizer())->normalizeData(['test' => 'undefined'], ['test' => null]);
    }

    /**
     * @param int|string|\DateTime $raw
     * @param \DateTimeImmutable   $expected
     *
     * @throws \Exception
     *
     * @dataProvider dateFormatsDataProvider
     */
    public function testCanNormalizeVariousFormatsOfDates($raw, \DateTimeImmutable $expected): void
    {
        /** @var \DateTimeImmutable $converted */
        $converted = (new DataNormalizer())->normalizeData(['test' => 'date'], ['test' => $raw])['test'];
        $this->assertInstanceOf(\DateTimeImmutable::class, $converted);
        $this->assertSame($expected->format('c'), $converted->format('c'));
    }

    /**
     * @param string $dataType
     * @param mixed  $raw
     * @param array  $expected
     *
     * @throws \Exception
     * @dataProvider arrayDataTypesDataProvider
     */
    public function testCanHandleSpecificArrayFormats(string $dataType, $raw, $expected): void
    {
        $normalizedArray = (new DataNormalizer())->normalizeData(['test' => $dataType], ['test' => $raw]);
        $this->assertIsArray($normalizedArray);
        $this->assertSame(['test' => $expected], $normalizedArray);
    }

    /**
     * @param mixed $invalidValue
     *
     * @throws \Exception
     *
     * @dataProvider invalidDateValuesDataProvider
     */
    public function testThrowsExceptionIfValueWithDateTypeCannotBeConvertedAndIsNotNullable($invalidValue): void
    {
        $this->expectException(\Exception::class);

        (new DataNormalizer())->normalizeData(['test' => 'date'], ['test' => $invalidValue]);
    }

    /**
     * @param mixed $invalidValue
     *
     * @throws \Exception
     *
     * @dataProvider invalidDateValuesDataProvider
     */
    public function testSetsValueToNullIfValueWithDateTypeCannotBeConvertedAndIsNullable($invalidValue): void
    {
        $this->assertNull((new DataNormalizer())->normalizeData(['test' => '?date'], ['test' => $invalidValue])['test']);
    }

    /**
     * @param mixed $invalidValue
     *
     * @throws \Exception
     * @dataProvider invalidValueTypesDataProvider
     */
    public function testThrowsExceptionWhenRawValueCannotBeConvertedToSpecifiedType($invalidValue): void
    {
        $this->expectException(\InvalidArgumentException::class);
        (new DataNormalizer())->normalizeData(['test' => 'string'], ['test' => $invalidValue]);
    }

    public static function simpleConversionDataProvider(): array
    {
        return [
            'int to int' => ['int', 7, 7],
            'string(number) to int' => ['int', '7', 7],
            'string(starting with number) to int' => ['int', '7cups', 7],
            'float(zero decimal) to int' => ['int', 7.0, 7],
            'float(non-zero decimal) to int (strips decimal part)' => ['int', 7.7, 7],
            'bool to int' => ['int', true, 1],
            'int to string' => ['string', 7, '7'],
            'string to string' => ['string', 'test', 'test'],
            'float to string' => ['string', 7.7, '7.7'],
            'bool to string' => ['string', true, '1'],
            'int to float' => ['float', 7, 7.0],
            'string(number) to float' => ['float', '7', 7.0],
            'string(starting with number) to float' => ['float', '7cups', 7.0],
            'float to float' => ['float', 7.7, 7.7],
            'bool to float' => ['float', true, 1.0],
            'int to bool' => ['bool', 7, true],
            'string to bool' => ['bool', 'test', true],
            'float to bool' => ['bool', 7.7, true],
            'bool to bool' => ['bool', false, false],
            'int to array' => ['array', 7, [7]],
            'string to array' => ['array', 'test', ['test']],
            'float to array' => ['array', 7.7, [7.7]],
            'bool to array' => ['array', false, [false]],
            'array to array' => ['array', [1, 2], [1, 2]],
        ];
    }

    public static function arrayDataTypesDataProvider(): array
    {
        return [
            'serialized array to array (serialized_array)' => ['serialized_array', serialize([1, 3]), [1, 3]],
            'json object to array (json_array)' => [
                'json_array', json_encode(['x' => 'test', 'y' => ['a' => 3]]), ['x' => 'test', 'y' => ['a' => 3]],
            ], // toto pole je kontrola bugu (podpole se převáděla na stdClass) – prosím nenahrazovat za jednodušší :)
            'mixed[] to int[] (int[])' => ['int[]', ['1', '5x', 7.0], [1, 5, 7]],
            'mixed[] to string[] (string[])' => ['string[]', ['1', '5x', 7.1], ['1', '5x', '7.1']],
        ];
    }

    public static function dateFormatsDataProvider(): array
    {
        $date = new \DateTimeImmutable('now');
        $dateInUtc = new \DateTimeImmutable('now', new \DateTimeZone('UTC'));
        $secondOfJanuary = new \DateTimeImmutable('2019-01-02');
        $mutableDate = new \DateTime('now');

        return [
            'unix timestamp (strips timezone)' => [$dateInUtc->format('U'), clone $dateInUtc],
            'instance of DateTime' => [$mutableDate, \DateTimeImmutable::createFromMutable($mutableDate)],
            'instance of DateTimeImmutable' => [clone $date, clone $date],
            'any format that DateTime can consume (now)' => [clone $date, clone $date],
            'any format that DateTime can consume (2019-01-02)' => [
                $secondOfJanuary->format('Y-m-d'), $secondOfJanuary,
            ],
        ];
    }

    public function invalidDateValuesDataProvider(): array
    {
        return [
            'empty string' => [''],
            'one space' => [' '],
            'nonsensical string' => ['ahoj'],
            'array' => [['ahoj']],
            'object' => [new \stdClass()],
        ];
    }

    public static function invalidValueTypesDataProvider(): array
    {
        return [
            'array to scalar conversion' => [['ahoj'], 'string'],
            'object to scalar conversion' => [new \stdClass(), 'string'],
        ];
    }

    private function assertConversionResult(array $spec, array $input, array $expected): void
    {
        $this->assertSame($expected, (new DataNormalizer())->normalizeData($spec, $input));
    }
}
