<?php

declare(strict_types=1);

namespace Dalten\DataNormalizer;

use Dalten\DataNormalizer\Error\ImpossibleValueConversion;
use Dalten\DataNormalizer\Error\UnknownDataType;

/**
 * Slouží k normalizaci předaných dat z DB nebo odjinud.
 *
 * Vynucuje existenci položek a normalizuje je na daný datový typ.
 */
class DataNormalizer
{
    private const SIMPLE_VALUE_TYPES = ['int', 'float', 'string', 'bool', 'array'];

    /**
     * Normalizuje data dle předané definice.
     *
     * @param array<string, string> $propertyDefinition definice dat (pole: [název property] => [typ])
     * @param array<string, mixed>  $data               data k převodu
     *
     * @return array<string, scalar|array<scalar,mixed>|\DateTimeImmutable|null>
     *
     * @throws ImpossibleValueConversion|UnknownDataType
     */
    public function normalizeData(array $propertyDefinition, array $data): array
    {
        $normalizedData = [];

        foreach ($propertyDefinition as $propertyName => $valueType) {
            $value = $data[$propertyName] ?? null;
            $nullable = str_starts_with($valueType, '?');
            if ($nullable) {
                $valueType = substr($valueType, 1);
            }

            try {
                $normalizedData[$propertyName] = $this->normalizeValueToType($value, $valueType, $nullable);
            } catch (ImpossibleValueConversion|UnknownDataType $exception) {
                $exception->withPropertyName($propertyName);
                throw $exception;
            }
        }

        return $normalizedData;
    }

    /**
     * @param mixed  $value     hodnota k normalizaci
     * @param string $valueType Typ hodnoty (int, string, date...).
     * @param bool   $nullable  Může nabývat hodnoty null?
     *
     * @return array<scalar,mixed>|scalar|\DateTimeImmutable|null
     */
    private function normalizeValueToType($value, string $valueType, bool $nullable)
    {
        if ($nullable && null === $value) {
            return null;
        }

        if (\in_array($valueType, self::SIMPLE_VALUE_TYPES)) {
            try {
                return $this->normalizeSimpleValue($value, $valueType);
            } catch (\InvalidArgumentException $exception) {
                throw ImpossibleValueConversion::becauseValueCannotBeConvertedToExpectedDataType($valueType);
            }
        }

        if (\in_array($valueType, ['serialized_array', 'json_array']) && (!\is_string($value) && !empty($value))) {
            throw ImpossibleValueConversion::becauseValueCannotBeConvertedToExpectedDataType($valueType);
        }

        switch ($valueType) {
            case 'serialized_array':
                if ($value) {
                    /** @phpstan-ignore-next-line Protože kontrola na string cast je už výše, ale phpstan ji nevidí */
                    $converted = @unserialize((string) $value);
                    if (false === $converted) {
                        throw ImpossibleValueConversion::becauseValueIsInvalidForChosenDataType($valueType);
                    }
                }

                return isset($converted) ? (array) $converted : [];
            case 'json_array':
                try {
                    /* @phpstan-ignore-next-line Protože kontrola na string cast je už výše, ale phpstan ji nevidí */
                    return $value ? (array) json_decode((string) $value, true, 512, \JSON_THROW_ON_ERROR) : [];
                } catch (\JsonException $exception) {
                    throw ImpossibleValueConversion::becauseValueIsInvalidForChosenDataType($valueType);
                }
            case 'int[]':
                return array_map(
                    function ($value) {
                        return (int) $this->normalizeSimpleValue($value, 'int');
                    },
                    (array) $value
                );
            case 'string[]':
                return array_map(
                    function ($value) {
                        return (string) $this->normalizeSimpleValue($value, 'string');
                    },
                    (array) $value
                );
            case 'date':
                try {
                    return $this->normalizeDateValue($value, $nullable);
                } catch (\InvalidArgumentException $exception) {
                    throw ImpossibleValueConversion::becauseValueCannotBeConvertedToExpectedDataType($valueType, $exception);
                }
            default:
                throw UnknownDataType::becauseDataTypeIsNotKnown($valueType);
        }
    }

    /**
     * @param mixed $value    hodnota k normalizaci
     * @param bool  $nullable Může nabývat hodnoty null?
     */
    private function normalizeDateValue($value, bool $nullable): ?\DateTimeImmutable
    {
        if ($value instanceof \DateTimeImmutable) {
            return clone $value;
        }
        if ($value instanceof \DateTime) {
            return \DateTimeImmutable::createFromMutable($value);
        }

        try {
            if (\is_string($value)) {
                $value = trim($value);
            }
            if (is_numeric($value)) {
                return new \DateTimeImmutable('@'.$value);
            } elseif (isset($value) && \is_scalar($value) && $value) {
                return new \DateTimeImmutable((string) $value);
            }

            throw new \InvalidArgumentException(sprintf('Nelze převést hodnotu "%s" na \DateTime', print_r($value, true)));
        } catch (\Exception $e) {
            if ($nullable) {
                return null;
            } else {
                throw new \InvalidArgumentException(sprintf('Nelze převést hodnotu "%s" na \DateTime', print_r($value, true)), $e->getCode(), $e);
            }
        }
    }

    /**
     * @param mixed  $value     hodnota k normalizaci
     * @param string $valueType Typ hodnoty (int, string, ...).
     *
     * @return ($valueType is 'array' ? array<scalar, mixed> : scalar)
     */
    private function normalizeSimpleValue($value, string $valueType)
    {
        if (\is_object($value)) {
            throw new \InvalidArgumentException(sprintf('Není možné převést instanci %s na datový typ "%s".', \get_class($value), $valueType));
        }

        if (\is_array($value) && 'array' !== $valueType) {
            throw new \InvalidArgumentException(sprintf('Není možné převést pole na datový typ "%s".', $valueType));
        }

        if (!\in_array($valueType, self::SIMPLE_VALUE_TYPES)) {
            throw new \InvalidArgumentException(sprintf('Neznámý jednoduchý typ "%s". (Známé: %s.)', $valueType, implode(', ', self::SIMPLE_VALUE_TYPES)));
        }
        $normalizedValue = $value;
        if (false === @settype($normalizedValue, $valueType)) {
            // tohle by nemělo nikdy nastat, dokud budeme za skalární používat pouze (int, bool, float, string)
            // tato výjimka je tč. nedosažitelná, ale v budoucnu může ohlídat regresi
            throw new \InvalidArgumentException(sprintf('Nelze převést hodnotu "%s" na "%s".', print_r($value, true), $valueType));
        }

        return $normalizedValue; /* @phpstan-ignore-line Pomocí in_array výše zajistíme že vracíme správný typ */
    }
}
