<?php

declare(strict_types=1);

namespace Dalten\Geomapi\Client\Method;

use Dalten\Geomapi\Client\Method\Exception\CommunicationErrorException;
use Dalten\Geomapi\Client\Method\Exception\InvalidRequestException;
use Dalten\Geomapi\Client\Method\Exception\InvalidResultFormatException;
use Dalten\Geomapi\Contracts\Api\Method\ApiMethodInterface;
use Dalten\Geomapi\Contracts\Api\Request\ApiRequestInterface;
use Dalten\Geomapi\Contracts\Api\Response\ApiResponse;
use Dalten\Geomapi\Contracts\Api\Response\ApiResponseInterface;
use Symfony\Contracts\HttpClient\Exception\HttpExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;
use Symfony\Contracts\HttpClient\HttpClientInterface;

class SearchMethod implements SearchMethodInterface
{
	private HttpClientInterface $client;
	private string $apiUri;

	public function __construct(HttpClientInterface $client, string $apiUri)
	{
		$this->client = $client;
		$this->apiUri = $apiUri;
	}

	public function accepts(ApiRequestInterface $apiRequest): bool
	{
		return ApiMethodInterface::SEARCH_METHOD_NAME === $apiRequest->getMethodName();
	}

	/**
	 * @throws CommunicationErrorException
	 * @throws InvalidRequestException
	 * @throws InvalidResultFormatException
	 */
	public function handleRequest(ApiRequestInterface $apiRequest): ApiResponseInterface
	{
		if (!$this->accepts($apiRequest)) {
			throw new InvalidRequestException();
		}

		$query = [
			'limit' => $apiRequest->getLimit(),
			'types' => $apiRequest->getTypes(),
			'filter' => $apiRequest->getFilter(),
		];
		$headers = ['Accept' => 'application/json'];
		$endpoint = $this->apiUri.'/'.ApiMethodInterface::SEARCH_METHOD_NAME;

		try {
			$response = $this->client->request('GET', $endpoint, ['query' => $query, 'headers' => $headers]);
			$content = json_decode($response->getContent(), true);
			if (!isset($content['status_code'], $content['status_message'], $content['result'])) {
				throw new InvalidResultFormatException('Neočekávaná odpověď od geomapi-server.
                    Odpověď musí být json s klíči status_code, status_message a result.');
			}

			return new ApiResponse($content['status_code'], $content['status_message'], $content['result']);
		} catch (TransportExceptionInterface|HttpExceptionInterface $e) {
			throw new CommunicationErrorException('Došlo k chybě při komunikaci s geomapi-server.', $e->getCode(), $e);
		}
	}
}
