<?php
namespace Dalten\Import;

use Composer\CaBundle\CaBundle;
use Dalten\Import\XmlRpcServer;
use Dalten\Import\PDOWrapper;

use BadMethodCallException;
use finfo;

/**
 * Implementace logiky importního rozhraní.
 *
 * Pokud chcete upravit chování exportu, nejlepší bude vytvořit rozšíření této třídy nebo její fork.
 */
class Logic
{
	const VERSION = '1.4.3';
	const PARENT_TYPE_NABIDKA = 1;
	const PARENT_TYPE_PROJEKT = 2;

	protected $_pdoWrapper;
	protected $_remotePath = 'https://www.irest3.cz/data/';
	protected $_localPath;
	protected $_useAutoincrementPriloha = true;

	/**
	 * Konstruktor.
	 *
	 * @param PDOWrapper $pdoWrapper Obálka nad databází.
	 * @param string     $localPath  Cesta do adresáře data (na ukládání dat)
	 *
	 * @throws BadMethodCallException Pokud se adresář na ukládání dat nejmenuje "data".
	 */
	function __construct(PDOWrapper $pdoWrapper, $localPath)
	{
		$this->_pdoWrapper = $pdoWrapper;
		$localPath = realpath($localPath);
		if (basename($localPath) != 'data') {
			throw new BadMethodCallException('Cesta na ukladani priloh musi existovat a vest do adresare "data"');
		}
		$this->_localPath = $localPath;
	}

	/**
	 * Nastaví cestu odkud se budou stahovat soubory.
	 *
	 * @param string $path Cesta na stahování souborů.
	 *
	 * @throws BadMethodCallException Pokud cesta nevede do adresáře data.
	 */
	public function setRemotePath($path)
	{
		if (!preg_match('~.+/data/$~', $path)) {
			throw new BadMethodCallException('Cesta na stahovani priloh musi koncit v adresari "/data/".');
		}
		$this->_remotePath = $path;
	}

	/**
	 * Nastaví, zda chceme IDčka příloh autoincrementovat.
	 *
	 * Starý (zpětně-kompatibilní) postup je autoincrementovat.
	 *
	 * Nový je kopírovat IDčka příloh z Irestu.
	 *
	 * Do budoucna to určitě nechcem, po větším množství obrázků totiž začne v databázi přetékat Int autoincrementu.
	 *
	 * @param bool $useIt Chceme autoincrementovat?
	 */
	public function useAutoincrementPriloha($useIt)
	{
		$this->_useAutoincrementPriloha = (bool) $useIt;
	}

	/**
	 * Zaregistruje všechny implementované metody do předaného XMLRPC serveru.
	 *
	 * @param XmlRpcServer $server Server, kam chceme metody zaregistrovat.
	 */
	public function registerMethods(XmlRpcServer $server)
	{
		$server->registerMethod('version', array($this, 'version'));

		$server->registerMethod('getHash', array($this, 'getHash'));
		$server->registerMethod('login', array($this, 'login'));
		$server->registerMethod('logout', array($this, 'logout'));

		$server->registerMethod('listUzivatel', array($this, 'listUzivatel'));
		$server->registerMethod('addUzivatel', array($this, 'addUzivatel'));
		$server->registerMethod('delUzivatel', array($this, 'delUzivatel'));

		$server->registerMethod('listProjekt', array($this, 'listProjekt'));
		$server->registerMethod('addProjekt', array($this, 'addProjekt'));
		$server->registerMethod('delProjekt', array($this, 'delProjekt'));

		$server->registerMethod('listNabidka', array($this, 'listNabidka'));
		$server->registerMethod('addNabidka', array($this, 'addNabidka'));
		$server->registerMethod('delNabidka', array($this, 'delNabidka'));

		$server->registerMethod('addAktuality', array($this, 'addAktuality'));
		$server->registerMethod('delAktuality', array($this, 'delAktuality'));

		$server->registerMethod('listFirma', array($this, 'listFirma'));
		$server->registerMethod('addFirma', array($this, 'addFirma'));
		$server->registerMethod('delFirma', array($this, 'delFirma'));

		$server->registerMethod('addReferenceUzivatele', array($this, 'addReferenceUzivatele'));
		$server->registerMethod('delReferenceUzivatele', array($this, 'delReferenceUzivatele'));
	}

	/**
	 * Stáhne a uloží na disk přílohu.
	 *
	 * Vytváří chybějící adresáře.
	 *
	 * Neukládá záznamy do databáze.
	 *
	 * @param string $fileName Relativní URL cesty k příloze (ve stylu "data/FXXXX/PXXXX/priloha/NXXXXX/img_XXX.jpg").
	 *
	 * @return bool|string Chybová hláška nebo FALSE, pokud stahování proběhlo úspěšně.
	 */
	protected function _downloadAndSaveFile($fileName)
	{
		$matches = array();
		if (!preg_match('~^/?data/(.+)$~', $fileName, $matches)) {
			return 'Spatny format cesty k souboru.';
		}
		$path = $matches[1];
		$saveFileName = $this->_localPath . '/' . $path;
		if (!file_exists(dirname($saveFileName))) {
			mkdir(dirname($saveFileName), 0777, true);
		}

		$downloadContext = stream_context_create([
			'ssl' => [
				'verify_peer' => true,
				'cafile' =>  CaBundle::getBundledCaBundlePath()
			]
		]);

		$fileContent = file_get_contents($this->_remotePath . $path, false, $downloadContext);
		if (!$fileContent) {
			return 'Nepodarilo se stahnout obrazek z URL ' . $this->_remotePath . $path;
		}
		if (file_put_contents($saveFileName, $fileContent) === FALSE) {
			return 'Nepodarilo se ulozit obrazek na filesystem.';
		}

		return false;
	}

	/**
	 * Smaže všechny obrázky nabídky/projektu ze souborového systému.
	 *
	 * @param int  $id           ID nabídky/projektu.
	 * @param int  $parentType   Jedna z konstant PARENT_TYPE_NABIDKA/PARENT_TYPE_PROJEKT.
	 * @param bool $removeFolder Smazat i složku, ve které obrázky byly?
	 */
	protected function _sweepAttachments($id, $parentType, $removeFolder = false)
	{
		$prilohy = $this->_pdoWrapper->fetchArray('SELECT * FROM priloha WHERE id_parent=? AND typ_parent=?', array($id, $parentType));
		$soubor = null;
		foreach ($prilohy as $priloha) {
			list($prefix, $path) = explode('/', $priloha['soubor'], 2);
			if ($prefix != 'data') {
				continue; // nevede do dat
			}
			$soubor = $this->_localPath . '/' . $path;
			unlink($soubor);
		}
		if ($soubor && $removeFolder) {
			$slozka = dirname($soubor);
			// když je složka prázdná
			if (count(scandir($slozka)) == 2) {
				rmdir($slozka);
			}
		}
	}

	/**
	 * Zkontroluje, zda mají předané parametry $params požadovanou strukturu odpovídají schématu $schema.
	 *
	 * Slouží ke kontrole správné struktury dat předaných skrz XMLRPC rozhraní.
	 *
	 * @param array $params Kontrolované parametry.
	 * @param array $schema Schéma požadované struktury parametrů.
	 *
	 *                      Schéma je pole, kde klíč je cesta k prvku v "adresářové" notaci a hodnota požadovaný
	 *                      typ prvku - jedna z "string", "numeric" nebo "array".
	 *
	 * @return array|bool Chybová hláška pro klienta nebo FALSE, pokud parametry odpovídají požadované struktuře.
	 */
	protected function _checkParameters($params, $schema)
	{
		foreach ($schema as $path => $type) {
			$walk = explode('/', $path);
			$walked = array();
			$node = $params;
			foreach ($walk as $step) {
				$walked[] = $step;
				if (array_key_exists($step, $node)) {
					$node = $node[$step];
					if (count($walk) == count($walked)) {
						if ($type == 'string') {
							if (!is_string($node)) {
								return array('status' => 500, 'statusMessage' => 'Chyba parametrů: Prvek ' . implode('/', $walked) . ' má špatný typ, má být string.');
							}
						} elseif ($type == 'numeric') {
							if (!is_numeric($node)) {
								return array('status' => 500, 'statusMessage' => 'Chyba parametrů: Prvek ' . implode('/', $walked) . ' má špatný typ, má být numeric.');
							}
						} elseif ($type == 'array') {
							if (!is_array($node)) {
								return array('status' => 500, 'statusMessage' => 'Chyba parametrů: Prvek ' . implode('/', $walked) . ' má špatný typ, má být array.');
							}
						}
					}
				} else {
					return array('status' => 500, 'statusMessage' => 'Chyba parametrů: Neexistuje povinný prvek ' . implode('/', $walked));
				}
			}
		}

		return false;
	}

	/**
	 * Zkontroluje, zda je importující klient přihlášen.
	 *
	 * Slouží ke kontrole přihlášení v metodách.
	 *
	 * @param array $params Parametry volané metody.
	 *
	 * @return array|bool Chybová hláška pro klienta, nebo FALSE pokud je klient správně přihlášen.
	 */
	protected function _checkSessionError($params)
	{
		if (empty($params['session_id'])) {
			return array('status' => 404, 'statusMessage' => "Nezadané session_id.");
		}
		$session = $this->_pdoWrapper->fetchOne('SELECT * FROM export_sess WHERE sessionid=?', array($params['session_id']));
		if (!$session) {
			return array('status' => 404, 'statusMessage' => "Neplatné session_id.");
		}
		if (!$session['prihlasen']) {
			return array('status' => 400, 'statusMessage' => "Nepřihlášen.");
		}

		return false;
	}

	/**
	 * Implementace metody getHash.
	 *
	 * @param array $params Parametry metody.
	 *
	 * @return array Odpověď metody.
	 */
	public function getHash($params)
	{
		$schemaError = $this->_checkParameters($params, array('name' => 'string'));
		if ($schemaError) {
			return $schemaError;
		}

		$sessionId = MD5(uniqid());
		$hash = MD5(uniqid());

		$login = $this->_pdoWrapper->fetchOne('SELECT * FROM export_login WHERE jmeno=?', array($params['name']));
		if (!$login) {
			return array('status' => 402, 'statusMessage' => "Neexistující uzivatel");
		}
		$this->_pdoWrapper->insertRow(
			'export_sess',
			array(
				'hash' => $hash,
				'sessionid' => $sessionId,
				'posledni_akce' => time(),
				'uzivatel' => $params['name'],
				'prihlasen' => 0
			)
		);

		return array('status' => 200, 'statusMessage' => "OK", 'output' => array("session_id" => $sessionId, "hash" => $hash));
	}

	/**
	 * Implementace metody login.
	 *
	 * @param array $params Parametry metody.
	 *
	 * @return array Odpověď metody.
	 */
	public function login($params)
	{
		$schemaError = $this->_checkParameters($params, array('session_id' => 'string', 'password' => 'string'));
		if ($schemaError) {
			return $schemaError;
		}

		$session = $this->_pdoWrapper->fetchOne('SELECT * FROM export_sess WHERE sessionid=?', array($params['session_id']));
		if (!$session) {
			return array('status' => 404, 'statusMessage' => "Neplatný session id.");
		}

		$uzivatel = $this->_pdoWrapper->fetchOne('SELECT * FROM export_login WHERE jmeno=?', array($session['uzivatel']));
		if (!$uzivatel) {
			return array('status' => 402, 'statusMessage' => "Neexistující uzivatel");
		}
		if (!$uzivatel['aktivni']) {
			return array('status' => 412, 'statusMessage' => "Účet není aktivní");
		}

		$spravneHeslo = md5($uzivatel['heslo'] . $session['hash']);
		if ($params['password'] != $spravneHeslo) {
			return array('status' => 403, 'statusMessage' => "Chybné heslo.");
		}

		$this->_pdoWrapper->perform('UPDATE export_sess SET prihlasen=? WHERE sessionid=?', array(1, $params['session_id']));

		return array('status' => 200, 'statusMessage' => 'OK');
	}

	/**
	 * Implementace metody logout.
	 *
	 * @param array $params Parametry metody.
	 *
	 * @return array Odpověď metody.
	 */
	public function logout($params)
	{
		$schemaError = $this->_checkParameters($params, array('session_id' => 'string'));
		if ($schemaError) {
			return $schemaError;
		}

		$session = $this->_pdoWrapper->fetchOne('SELECT * FROM export_sess WHERE sessionid=?', array($params['session_id']));
		if (!$session) {
			return array('status' => 404, 'statusMessage' => "Neplatný session id.");
		}
		if (!$session['prihlasen']) {
			return array('status' => 400, 'statusMessage' => "Nepřihlášen.");
		}
		$this->_pdoWrapper->perform('DELETE FROM export_sess WHERE sessionid=?', array($params['session_id']));

		return array('status' => 200, 'statusMessage' => 'OK');
	}

	/**
	 * Implementace metody version.
	 *
	 * @return array Odpověď metody.
	 */
	function version()
	{
		return array('status' => 200, 'statusMessage' => "OK", 'output' => array("version" => self::VERSION));
	}

	/**
	 * Implementace metody listUzivatel.
	 *
	 * Metoda vyžaduje přihlášení.
	 *
	 * @param array $params Parametry metody.
	 *
	 * @return array Odpověď metody.
	 */
	public function listUzivatel($params)
	{
		$sessionError = $this->_checkSessionError($params);
		if ($sessionError) {
			return $sessionError;
		}
		$uzivatele = $this->_pdoWrapper->fetchArray('SELECT id FROM uzivatel');
		$idUzivatelu = array();
		foreach ($uzivatele as $uzivatel) {
			$idUzivatelu[] = $uzivatel['id'];
		}

		return array(
			'status' => 200,
			'statusMessage' => "OK",
			'output' => $idUzivatelu,
		);
	}

	/**
	 * Implementace metody addUzivatel.
	 *
	 * Metoda vyžaduje přihlášení.
	 *
	 * @param array $params Parametry metody.
	 *
	 * @return array Odpověď metody.
	 */
	public function addUzivatel($params)
	{
		$sessionError = $this->_checkSessionError($params);
		if ($sessionError) {
			return $sessionError;
		}

		$schemaError = $this->_checkParameters($params, array('uzivatel/id' => 'numeric'));
		if ($schemaError) {
			return $schemaError;
		}

		$this->_pdoWrapper->perform('SET FOREIGN_KEY_CHECKS=0');
		$userId = (int) $params['uzivatel']['id'];
		$this->_pdoWrapper->perform('DELETE FROM uzivatel WHERE id = ?', array($userId));
		$awards = isset($params['uzivatel']['oceneni']) ? (array) $params['uzivatel']['oceneni'] : null;
		unset($params['uzivatel']['oceneni']);
		$otherPositions = isset($params['uzivatel']['dalsi_pozice']) ? (array) $params['uzivatel']['dalsi_pozice'] : null;
		unset($params['uzivatel']['dalsi_pozice']);
		$this->_pdoWrapper->insertRow('uzivatel', $params['uzivatel']);
		$this->_pdoWrapper->perform('SET FOREIGN_KEY_CHECKS=1');

		if (!empty($params['uzivatel']['foto'])) {
			$this->_downloadAndSaveFile($params['uzivatel']['foto']);
		}

		if (!empty($params['uzivatel']['foto_big'])) {
			$this->_downloadAndSaveFile($params['uzivatel']['foto_big']);
		}

		if (isset($awards)) {
			$this->_pdoWrapper->perform('DELETE FROM uzivatel_oceneni WHERE id_uzivatel = ?', array($userId));
			foreach ($awards as $award) {
				$award['id_uzivatel'] = $userId;
				$this->_pdoWrapper->insertRow('uzivatel_oceneni', $award);
			}
		}

		if (isset($otherPositions)) {
			$this->_pdoWrapper->perform('DELETE FROM uzivatel_dalsi_pozice WHERE id_uzivatel = ?', array($userId));
			if ($otherPositions) {
				foreach ($otherPositions as $position) {
					$this->_pdoWrapper->insertRow('uzivatel_dalsi_pozice', $position);
				}
			}
		}


		return array('status' => 200, 'statusMessage' => 'OK');
	}

	/**
	 * Implementace metody delUzivatel.
	 *
	 * Metoda vyžaduje přihlášení.
	 *
	 * @param array $params Parametry metody.
	 *
	 * @return array Odpověď metody.
	 */
	public function delUzivatel($params)
	{
		$sessionError = $this->_checkSessionError($params);
		if ($sessionError) {
			return $sessionError;
		}

		$schemaError = $this->_checkParameters($params, array('uzivatel_id' => 'numeric'));
		if ($schemaError) {
			return $schemaError;
		}

		$this->_pdoWrapper->perform('DELETE FROM uzivatel WHERE id=?', array($params['uzivatel_id']));



		return array('status' => 200, 'statusMessage' => 'OK');
	}

	/**
	 * Implementace metody listProjekt.
	 *
	 * Metoda vyžaduje přihlášení.
	 *
	 * @param array $params Parametry metody.
	 *
	 * @return array Odpověď metody.
	 */
	public function listProjekt($params)
	{
		$sessionError = $this->_checkSessionError($params);
		if ($sessionError) {
			return $sessionError;
		}

		$idProjektu = array();

		$projekty = $this->_pdoWrapper->fetchArray('SELECT id FROM projekt');
		foreach ($projekty as $projekt) {
			$idProjektu[] = $projekt['id'];
		}

		return array(
			'status' => 200,
			'statusMessage' => "OK",
			'output' => (implode("|", $idProjektu)),
			'count' => count($idProjektu)
		);
	}

	/**
	 * Implementace metody addProjekt.
	 *
	 * Metoda vyžaduje přihlášení.
	 *
	 * @param array $params Parametry metody.
	 *
	 * @return array Odpověď metody.
	 */
	public function addProjekt($params)
	{
		$sessionError = $this->_checkSessionError($params);
		if ($sessionError) {
			return $sessionError;
		}

		$schemaError = $this->_checkParameters($params, array('projekt/id' => 'numeric', 'projekt/priloha' => 'array'));
		if ($schemaError) {
			return $schemaError;
		}

		$projektData = $params['projekt'];
		$projektId = $projektData['id'];
		$prilohy = $projektData['priloha'];
		unset($projektData['priloha']);

		$this->_sweepAttachments($projektId, self::PARENT_TYPE_PROJEKT);
		$this->_pdoWrapper->perform('DELETE FROM priloha WHERE id_parent=? AND typ_parent=?', array($projektId, self::PARENT_TYPE_PROJEKT));

		foreach ($prilohy as $priloha) {
			if ($this->_useAutoincrementPriloha) {
				unset($priloha['id']);
			}
			$downloadError = $this->_downloadAndSaveFile($priloha['soubor']);
			if ($downloadError) {
				return array('status' => 500, 'statusMessage' => 'Chyba nahravani obrazku: ' . $downloadError);
			}
			$this->_pdoWrapper->insertRow('priloha', $priloha);
		}

		$projectPriceTemplate = 'SELECT
		IF(zakazka_typ = 1, nemovitost_cena_prodej, nemovitost_cena_pronajem) AS cena,
		IF(zakazka_typ = 1, nemovitost_cena_prodej_jednotka, nemovitost_cena_pronajem_jednotka) AS cena_jednotka,
		zakazka_typ, nemovitost_mena
		FROM nabidka
		WHERE id_projekt = ? AND nezobrazovat_cenu = 0
		ORDER BY cena %s LIMIT 1';
		$nabidkaMinCena = $this->_pdoWrapper->fetchOne(sprintf($projectPriceTemplate, 'ASC'), $projektData['id']);
		$nabidkaMaxCena = $this->_pdoWrapper->fetchOne(sprintf($projectPriceTemplate, 'DESC'), $projektData['id']);
		$projektData['projekt_cena_min'] = (int) isset($nabidkaMinCena['cena']) ? $nabidkaMinCena['cena'] : 0;
		$projektData['projekt_cena_max'] = (int) isset($nabidkaMaxCena['cena']) ? $nabidkaMaxCena['cena'] : 0;
		$projektData['projekt_cena_min_jednotka'] = (int) isset($nabidkaMinCena['cena_jednotka']) ? $nabidkaMinCena['cena_jednotka'] : 0;
		$projektData['projekt_cena_max_jednotka'] = (int) isset($nabidkaMaxCena['cena_jednotka']) ? $nabidkaMaxCena['cena_jednotka'] : 0;
		$projektData['projekt_cena_min_mena'] = (int) isset($nabidkaMinCena['nemovitost_mena']) ? $nabidkaMinCena['nemovitost_mena'] : 0;
		$projektData['projekt_cena_max_mena'] = (int) isset($nabidkaMaxCena['nemovitost_mena']) ? $nabidkaMaxCena['nemovitost_mena'] : 0;

		$this->_pdoWrapper->perform('SET FOREIGN_KEY_CHECKS=0');
		$this->_pdoWrapper->perform('DELETE FROM projekt WHERE id=?', array($projektId));
		$this->_pdoWrapper->insertRow('projekt', $projektData);
		$this->_pdoWrapper->perform('SET FOREIGN_KEY_CHECKS=1');

		return array('status' => 200, 'statusMessage' => 'OK');
	}

	/**
	 * Implementace metody delProjekt.
	 *
	 * Metoda vyžaduje přihlášení.
	 *
	 * @param array $params Parametry metody.
	 *
	 * @return array Odpověď metody.
	 */
	public function delProjekt($params)
	{
		$sessionError = $this->_checkSessionError($params);
		if ($sessionError) {
			return $sessionError;
		}

		$schemaError = $this->_checkParameters($params, array('projekt_id' => 'numeric'));
		if ($schemaError) {
			return $schemaError;
		}

		$this->_sweepAttachments($params['projekt_id'], self::PARENT_TYPE_PROJEKT, true);
		$this->_pdoWrapper->perform('DELETE FROM priloha WHERE id_parent=? AND typ_parent=?', array($params['projekt_id'], self::PARENT_TYPE_PROJEKT));

		$this->_pdoWrapper->perform('DELETE FROM projekt WHERE id=?', array($params['projekt_id']));

		return array('status' => 200, 'statusMessage' => 'OK');
	}

	/**
	 * Implementace metody listNabidka.
	 *
	 * Metoda vyžaduje přihlášení.
	 *
	 * @param array $params Parametry metody.
	 *
	 * @return array Odpověď metody.
	 */
	public function listNabidka($params)
	{
		$sessionError = $this->_checkSessionError($params);
		if ($sessionError) {
			return $sessionError;
		}

		$idNabidky = array();

		$nabidky = $this->_pdoWrapper->fetchArray('SELECT id FROM nabidka');
		foreach ($nabidky as $nabidka) {
			$idNabidky[] = $nabidka['id'];
		}

		return array(
			'status' => 200,
			'statusMessage' => "OK",
			'output' => (implode("|", $idNabidky)),
			'count' => count($idNabidky)
		);
	}

	/**
	 * Implementace metody addNabidka.
	 *
	 * Metoda vyžaduje přihlášení.
	 *
	 * @param array $params Parametry metody.
	 *
	 * @return array Odpověď metody.
	 */
	public function addNabidka($params)
	{
		$sessionError = $this->_checkSessionError($params);
		if ($sessionError) {
			return $sessionError;
		}

		$schemaError = $this->_checkParameters($params, array('nabidka/id' => 'numeric', 'nabidka/priloha' => 'array'));
		if ($schemaError) {
			return $schemaError;
		}

		$nabidkaData = $params['nabidka'];
		$nabidkaId = $nabidkaData['id'];
		$prilohy = $nabidkaData['priloha'];
		unset($nabidkaData['priloha']);

		$this->_sweepAttachments($nabidkaId, self::PARENT_TYPE_NABIDKA);
		$this->_pdoWrapper->perform('DELETE FROM priloha WHERE id_parent=? AND typ_parent=?', array($nabidkaId, self::PARENT_TYPE_NABIDKA));

		foreach ($prilohy as $priloha) {
			if ($this->_useAutoincrementPriloha) {
				unset($priloha['id']);
			}
			$downloadError = $this->_downloadAndSaveFile($priloha['soubor']);
			if ($downloadError) {
				return array('status' => 500, 'statusMessage' => 'Chyba nahravani obrazku: ' . $downloadError);
			}
			$this->_pdoWrapper->insertRow('priloha', $priloha);
		}
		if (isset($nabidkaData['dokumenty']) && is_array($nabidkaData['dokumenty'])) {
			// stahujeme dokumenty
			foreach ($nabidkaData['dokumenty'] as $dokument) {
				if ($this->_useAutoincrementPriloha) {
					unset($dokument['id']);
				}
				$dokument['typ_prilohy'] = 2; // pro jistotu
				$downloadError = $this->_downloadAndSaveFile($dokument['soubor']);
				if ($downloadError) {
					return array('status' => 500, 'statusMessage' => 'Chyba nahravani obrazku: ' . $downloadError);
				}
				$this->_pdoWrapper->insertRow('priloha', $dokument);
			}
		}

		if (!empty($nabidkaData['nemovitost_pudorys_priloha']) && $this->_savePudorysOrStitek($nabidkaData['nemovitost_pudorys_priloha'], 'pudorys', $nabidkaData['kod'])) {
            $this->_pdoWrapper->insertRow('priloha', array(
                'id_parent'=>$nabidkaId,
                'typ_parent'=>self::PARENT_TYPE_NABIDKA,
                'typ_prilohy'=>2,
                'soubor'=>sprintf('data/pudorys/%s.pdf', $nabidkaData['kod'])
            ));
        }

        if (
            !empty($nabidkaData['nemovitost_energeticky_stitek_priloha']) &&
            ($stitekExt = $this->_guessExtension($nabidkaData['nemovitost_energeticky_stitek_priloha'])) &&
            $this->_savePudorysOrStitek($nabidkaData['nemovitost_energeticky_stitek_priloha'], 'pudorys', $nabidkaData['kod'], $stitekExt)
        ) {
            $this->_pdoWrapper->insertRow('priloha', array(
                'id_parent'=>$nabidkaId,
                'typ_parent'=>self::PARENT_TYPE_NABIDKA,
                'typ_prilohy'=>2,
                'soubor'=>sprintf('data/pudorys/%s.%s', $nabidkaData['kod'], $stitekExt)
            )); // ukládáme mezi přílohy, aby bylo smazáno s nemovitostí (nebo při reexportu)
            $nabidkaData['nemovitost_energeticky_stitek_priloha'] = sprintf('data/stitky/%s.%s', $nabidkaData['kod'], $stitekExt);
        }

		$this->_pdoWrapper->perform('SET FOREIGN_KEY_CHECKS=0');
		$this->_pdoWrapper->perform('DELETE FROM nabidka WHERE id=?', array($nabidkaId));
		$this->_pdoWrapper->insertRow('nabidka', $nabidkaData);
		$this->_pdoWrapper->perform('SET FOREIGN_KEY_CHECKS=1');

		return array('status' => 200, 'statusMessage' => 'OK');
	}

	/**
	 * Implementace metody delNabidka.
	 *
	 * Metoda vyžaduje přihlášení.
	 *
	 * @param array $params Parametry metody.
	 *
	 * @return array Odpověď metody.
	 */
	public function delNabidka($params)
	{
		$sessionError = $this->_checkSessionError($params);
		if ($sessionError) {
			return $sessionError;
		}

		$schemaError = $this->_checkParameters($params, array('nabidka_id' => 'numeric'));
		if ($schemaError) {
			return $schemaError;
		}

		$this->_sweepAttachments($params['nabidka_id'], self::PARENT_TYPE_NABIDKA, true);
		$this->_pdoWrapper->perform('DELETE FROM priloha WHERE id_parent=? AND typ_parent=?', array($params['nabidka_id'], self::PARENT_TYPE_NABIDKA));

		$this->_pdoWrapper->perform('DELETE FROM nabidka WHERE id=?', array($params['nabidka_id']));

		return array('status' => 200, 'statusMessage' => 'OK');
	}

	/**
	 * Implementace metody addAktuality.
	 *
	 * Metoda vyžaduje přihlášení.
	 *
	 * @param array $params Parametry metody.
	 *
	 * @return array Odpověď metody.
	 */
	public function addAktuality($params)
	{
		$sessionError = $this->_checkSessionError($params);
		if ($sessionError) {
			return $sessionError;
		}

		$schemaError = $this->_checkParameters($params, array('aktualita/id' => 'numeric'));
		if ($schemaError) {
			return $schemaError;
		}

		$this->_pdoWrapper->perform('DELETE FROM www_aktualita WHERE id=?', array($params['aktualita']['id']));
		$this->_pdoWrapper->insertRow('www_aktualita', $params['aktualita']);

		return array('status' => 200, 'statusMessage' => 'OK');
	}

	/**
	 * Implementace metody delAktuality.
	 *
	 * Metoda vyžaduje přihlášení.
	 *
	 * @param array $params Parametry metody.
	 *
	 * @return array Odpověď metody.
	 */
	public function delAktuality($params)
	{
		$sessionError = $this->_checkSessionError($params);
		if ($sessionError) {
			return $sessionError;
		}

		$schemaError = $this->_checkParameters($params, array('aktualita_id' => 'numeric'));
		if ($schemaError) {
			return $schemaError;
		}

		$this->_pdoWrapper->perform('DELETE FROM www_aktualita WHERE id=?', array($params['aktualita_id']));

		return array('status' => 200, 'statusMessage' => 'OK');
	}

	/**
	 * Implementace metody addFirma
	 *
	 * Metoda vyžaduje přihlášení.
	 *
	 * @param array $params Parametry metody.
	 *
	 * @return array Odpověď metody.
	 */
	public function addFirma($params)
	{
		$sessionError = $this->_checkSessionError($params);
		if ($sessionError) {
			return $sessionError;
		}

		$schemaError = $this->_checkParameters($params, array('firma' => 'array'));
		if ($schemaError) {
			return $schemaError;
		}

		$this->_pdoWrapper->perform('SET FOREIGN_KEY_CHECKS=0');
		$this->_pdoWrapper->perform('DELETE FROM firma WHERE id=?', array($params['firma']['id']));

		$this->_pdoWrapper->insertRow('firma', $params['firma']);
		$this->_pdoWrapper->perform('SET FOREIGN_KEY_CHECKS=1');

		if (!empty($params['firma']['obrazky']) && json_decode($params['firma']['obrazky'])) {
			$obrazky = json_decode($params['firma']['obrazky']);
			foreach ($obrazky as $obrazek) {
				$this->_downloadAndSaveFile($obrazek);
			}
		}

		return array('status' => 200, 'statusMessage' => 'OK');
	}

	/**
	 * Implementace metody delFirma.
	 *
	 * Metoda vyžaduje přihlášení.
	 *
	 * @param array $params Parametry metody.
	 *
	 * @return array Odpověď metody.
	 */
	public function delFirma($params)
	{
		$sessionError = $this->_checkSessionError($params);
		if ($sessionError) {
			return $sessionError;
		}

		$schemaError = $this->_checkParameters($params, array('firma_id' => 'numeric'));
		if ($schemaError) {
			return $schemaError;
		}

		$this->_pdoWrapper->perform('DELETE FROM firma WHERE id=?', array($params['firma_id']));

		return array('status' => 200, 'statusMessage' => 'OK');
	}

	/**
	 * Implementace metody listFirma.
	 *
	 * Metoda vyžaduje přihlášení.
	 *
	 * @param array $params Parametry metody.
	 *
	 * @return array Odpověď metody.
	 */
	public function listFirma($params)
	{
		$sessionError = $this->_checkSessionError($params);
		if ($sessionError) {
			return $sessionError;
		}

		$idFirem = array();

		$firmy = $this->_pdoWrapper->fetchArray('SELECT * FROM firma');
		foreach ($firmy as $firma) {
			$idFirem[] = $firma['id'];
		}

		return array('status' => 200, 'statusMessage' => 'OK', 'output'=>$idFirem);
	}

	/**
	 * Implementace metody addReferenceUzivatele.
	 *
	 * Metoda vyžaduje přihlášení.
	 *
	 * @param array $params Parametry metody.
	 *
	 * @return array Odpověď metody.
	 */
	public function addReferenceUzivatele($params)
	{
		$sessionError = $this->_checkSessionError($params);
		if ($sessionError) {
			return $sessionError;
		}

		$this->_pdoWrapper->perform('DELETE FROM uzivatel_reference WHERE id=?', array($params['reference']['id']));

		$this->_pdoWrapper->insertRow('uzivatel_reference', $params['reference']);
		if (!empty($params['reference']['priloha'])) {
			$downloadError = $this->_downloadAndSaveFile($params['reference']['priloha']);
			if ($downloadError) {
				return array('status' => 500, 'statusMessage' => $downloadError);
			}
		}

		return array('status' => 200, 'statusMessage' => 'OK');
	}

	/**
	 * Implementace metody delReferenceUzivatele.
	 *
	 * Metoda vyžaduje přihlášení.
	 *
	 * @param array $params Parametry metody.
	 *
	 * @return array Odpověď metody.
	 */
	public function delReferenceUzivatele($params)
	{
		$sessionError = $this->_checkSessionError($params);
		if ($sessionError) {
			return $sessionError;
		}

		$schemaError = $this->_checkParameters($params, array('reference_id' => 'numeric'));
		if ($schemaError) {
			return $schemaError;
		}

		$this->_pdoWrapper->perform('DELETE FROM uzivatel_reference WHERE id=?', array($params['reference_id']));

		return array('status' => 200, 'statusMessage' => 'OK');
	}

    /**
     * Uloží půdorys nebo štítek na disk.
     *
     * @param mixed  $val    Vstupní data.
     * @param string $prefix Prefix cesty.
     * @param string $kod    Kód nabídky/projektu.
     * @param string $ext    Přípona souboru (výchozí "pdf").
     *
     * @return bool Podařilo se uložit štítek?
     */
	protected function _savePudorysOrStitek($val, $prefix, $kod, $ext='pdf')
    {
        if (is_object($val) && get_class($val)=='stdClass' && $val->xmlrpc_type=='base64') {
            $data = $val->scalar;

            $path = sprintf('%s/%s.%s', $prefix, $kod, $ext);
            $saveFileName = $this->_localPath . '/' . $path;
            if (!file_exists(dirname($saveFileName))) {
                mkdir(dirname($saveFileName), 0777, true);
            }

            file_put_contents($saveFileName, $data);

            return true;
        }
        return false;
    }

    /**
     * Uhodneme příponu souboru na uložení.
     *
     * @param mixed $val Obsah souboru.
     * @return string Přípona souboru.
     */
    protected function _guessExtension($val)
    {
        if (is_object($val) && get_class($val)=='stdClass' && $val->xmlrpc_type=='base64') {
            $content = $val->scalar;

            // hádání MIME typu souboru z obsahu
            // vyzobáno z https://github.com/flourishlib/flourish-classes/blob/master/fFile.php

            $length = strlen($content);
            $_0_8   = substr($content, 0, 8);
            $_0_6   = substr($content, 0, 6);
            $_0_5   = substr($content, 0, 5);
            $_0_4   = substr($content, 0, 4);
            $_0_3   = substr($content, 0, 3);
            $_0_2   = substr($content, 0, 2);
            $_8_4   = substr($content, 8, 4);

            if ($_0_5 == '%PDF-') {
                return 'pdf';
            }

            $normal_jpeg    = $length > 10 && in_array(substr($content, 6, 4), array('JFIF', 'Exif'));
            $photoshop_jpeg = $length > 24 && $_0_4 == "\xFF\xD8\xFF\xED" && substr($content, 20, 4) == '8BIM';
            $jpeg_prefix = $_0_3 == "\xFF\xD8\xFF";
            if ($normal_jpeg || $photoshop_jpeg || $jpeg_prefix) {
                return 'jpg';
            }

            if ($_0_8 == "\x89PNG\x0D\x0A\x1A\x0A") {
                return 'png';
            }

            return 'dat'; // neuhodli jsme obsah - jsou to generická data
        }

        return null;
    }
}




