<?php
namespace Dalten\Import;

use Exception;
use PhpXmlRpc\Encoder;
use PhpXmlRpc\Request;
use PhpXmlRpc\Response;

/**
 * XMLRPC server - překládá XMLRPC volání na volání callbacku v PHP a zpět.
 */
class XmlRpcServer
{
	protected $_server;
	protected $_methods = array();
	protected $_encodeOptions = array();
	protected $_verboseExceptions = false;

	/**
	 * Konstruktor.
	 */
	function __construct()
	{
		$this->_encodeOptions = array('encoding' => 'UTF-8', 'escaping' => 'markup');
	}

	/**
	 * Vypne nebo zapne detailní vyjímky.
	 *
	 * @param bool $verbose Chceme detailní vyjímky?
	 */
	public function setVerboseException($verbose)
	{
		$this->_verboseExceptions = (bool)$verbose;
	}

	/**
	 * Zaregistruje XMLRPC metodu.
	 *
	 * @param string $methodName Název metody.
	 * @param mixed  $callback   Callback, který se zavolá při volání této metody.
	 */
	public function registerMethod($methodName, $callback)
	{
		$this->_methods[$methodName] = $callback;
	}

	/**
	 * Zakóduje vyjímku pro odpověď v XMLRPC.
	 *
	 * @param Exception $exception
	 *
	 * @return string Zakódovaná odpověď vyjímky.
	 */
	public function exception(Exception $exception)
	{
		return $this->fail(get_class($exception) . ': ' . $exception->getMessage() . PHP_EOL . $exception->getTraceAsString());
	}

	/**
	 * Zakóduje chybu volání.
	 *
	 * @param string $statusMessage Chybová hláška, výchozí je "Error!".
	 * @param int    $status        Kód chyby, výchozí je 500.
	 *
	 * @return string Zakódovaná odpověď chyby.
	 */
	public function fail($statusMessage = 'Error!', $status = 500)
	{
	    $response = new Response(null, $status, $statusMessage);
	    return $response->serialize();
	}

	/**
	 * Obslouží XMLRPC požadavek.
	 *
	 * Zavolá zaregistrovanou metodu, předá jí parametry z požadavku a zakóduje její odpověď. V případě chyby
	 * zakóduje chybu.
	 *
	 * @param string $xmlRpcCall Tělo požadavku (XMLRPC požadavek).
	 *
	 * @return string Odpověď serveru.
	 */
	public function serve($xmlRpcCall)
	{
        $encoder = new Encoder();

        $request= $encoder->decodeXml($xmlRpcCall);
        if (!Request instanceof $request) {
            return $this->fail('Unable to parse XMLRPC request.');
        }

		$methodName = $request->method();
		if (isset($this->_methods[$methodName])) {
			$params = $encoder->decode($request);
			try {
				$result = call_user_func_array($this->_methods[$methodName], $params);
				if (!$result) {
					return $this->fail('Method call failed.');
				}
			} catch (Exception $exception) {
				if ($this->_verboseExceptions) {
					return $this->exception($exception);
				} else {
					return $this->fail('Method call thrown exception.');
				}
			}

            $response = new Response($encoder->encode($result));
			try {
                return $response->serialize();
            } catch (Exception $exception) {
			    return $this->exception($exception); // data se nepodařilo správně zakódovat
            }
		}

		return $this->fail(sprintf("Method '%s' not found!", $methodName));
	}
}
