<?php
namespace Dalten\Import;

/**
 * Implementace spouštěče rozhraní.
 */
class Runner
{
	protected $_dbServer;
	protected $_dbName;
	protected $_dbUserName;
	protected $_dbPassword;
	protected $_dataPath;
	protected $_downloadPath;
	protected $_debugMode = false;

	/**
	 * Konstruktor.
	 *
	 * @param string $dbServer     Adresa databázového stroje.
	 * @param string $dbName       Název databáze.
	 * @param string $dbUserName   Uživatel do databáze.
	 * @param string $dbPassword   Heslo k databázi.
	 * @param string $dataPath     Cesta k ukládání obrázků. Musí končit v adresáři nazvaném data.
	 * @param string $downloadPath Cesta odkud se obrázky stahují. Musí končit v adresáři nazvaném data.
	 *                             Výchozí vede na Irest 3.
	 */
	function __construct($dbServer, $dbName, $dbUserName, $dbPassword, $dataPath, $downloadPath = 'https://www.irest3.cz/data/')
	{
		$this->_dbServer = $dbServer;
		$this->_dbName = $dbName;
		$this->_dbUserName = $dbUserName;
		$this->_dbPassword = $dbPassword;
		$this->_dataPath = $dataPath;
		$this->_downloadPath = $downloadPath;
	}

	/**
	 * Zapne debug mód.
	 */
	public function enableDebugMode()
	{
		$this->_debugMode = true;
	}

	/**
	 * Obslouží požadavek a vrátí odpověď.
	 */
	public function run()
	{
		$server = new XmlRpcServer();

		if ($this->_debugMode) {
			$server->setVerboseException(true);
		}

		try {
			$wrap = PDOWrapper::connectToMySQL(
				$this->_dbServer, $this->_dbName, $this->_dbUserName, $this->_dbPassword
			);
			$import = new Logic($wrap, $this->_dataPath);
			$import->registerMethods($server);
			if ($this->_downloadPath) {
				$import->setRemotePath($this->_downloadPath);
			}
		} catch (\PDOException $exception) {
			if ($this->_debugMode) {
				return $server->exception($exception);
			}
			// nedaří se připojit do databáze - pravděpodobně dočasný výpadek
			return $server->fail('Chyba připojení k databázi. Zkuste prosím export později.');
		} catch (\Exception $exception) {
			if ($this->_debugMode) {
				return $server->exception($exception);
			}
			// jiná chyba nastavení
			return $server->fail('Chyba nastavení exportního serveru. Kontaktujte technickou podporu.');
		}

		$request = HttpRequest::createFromGlobals();
		if ($request->isPost()) {
			// vypínáme notice a warningy
			error_reporting(E_ALL & ~E_NOTICE & ~E_WARNING);
			// zde obsluhujeme XMLRPC požadavky
			return $server->serve($request->getBody());
		} else {
			// a zde může být například dokumentace
			return 'Importní rozhraní programu iRest v ' . Logic::VERSION;
		}
	}
}
