<?php
namespace Dalten\Import;

use PDO;
use Exception;

/**
 * Jednoduchá obálka nad PDO ulehčující vybírání a ukládání dat.
 *
 * V současnosti podporuje pouze MySQL, funkčnost pod jinou databází není zaručena.
 */
class PDOWrapper
{
	/** @var \PDO */
	protected $_pdo;
	protected $_columnsByTable = array();

	/**
	 * Vytvoří novou instanci databázové obálky.
	 *
	 * @param string $host          Databázový server.
	 * @param string $database_name Název databáze.
	 * @param string $username      Uživatel databáze.
	 * @param string $password      Heslo k databázi.
	 *
	 * @return \Dalten\Import\PDOWrapper Instanci databázové obálky.
	 */
	static function connectToMySQL($host, $database_name, $username, $password)
	{
		if (version_compare(PHP_VERSION, '5.3.6') <= 0) {
			throw new Exception('Verze php nizsi nez 5.3.6 nejsou podporovany kvuli chybě ve funkci PDO::quote.');
		}
		$dsn = sprintf('mysql:host=%s;dbname=%s;charset=utf8', $host, $database_name);
		$pdo = new PDO($dsn, $username, $password);
		$pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
		return new self($pdo);
	}

	/**
	 * Konstruktor.
	 *
	 * (Chráněn, aby uživatel nemohl vložit jinou databázi než MySQL)
	 *
	 * @param PDO $pdo Instance PDO.
	 *
	 * @throws Exception Pokud předáme špatnou instanci PDO.
	 */
	protected function __construct(PDO $pdo)
	{
		if ($pdo->getAttribute(PDO::ATTR_ERRMODE)!=PDO::ERRMODE_EXCEPTION) {
			throw new Exception('PDOWrapper nepodporuje jiny ERRORMODE nez EXCEPTION.');
		}
		$this->_pdo = $pdo;
	}

	/**
	 * Vrací instanci PDO.
	 *
	 * @return PDO Obalená instance.
	 */
	public function getPDO()
	{
		return $this->_pdo;
	}

	/**
	 * Vykoná SQL dotaz a vrátí výsledky jako PDOStatement.
	 *
	 * @param string $sql    SQL dotaz k vykonání.
	 * @param array  $params Parametry dotazu.
	 *
	 * @return PDOStatement Výsledek dotazu.
	 */
	public function perform($sql, $params = array())
	{
		$statement = $this->_pdo->prepare($sql);
		$statement->execute($params);

		return $statement;
	}

	/**
	 * Vykoná dotaz a vrátí všechny řádky výsledku.
	 *
	 * @param string $sql    SQL dotaz k vykonání.
	 * @param array  $params Parametry dotazu.
	 *
	 * @return array Všechny řádky výsledku.
	 */
	public function fetchArray($sql, $params = array())
	{
		$statement = $this->_pdo->prepare($sql);
		$statement->execute($params);

		return $statement->fetchAll(PDO::FETCH_ASSOC);
	}

	/**
	 * Vykoná dotaz a vrátí první řádek výsledku.
	 *
	 * @param string $sql    SQL dotaz k vykonání.
	 * @param array  $params Parametry dotazu.
	 *
	 * @return array První řádek výsledku.
	 */
	public function fetchOne($sql, $params = array())
	{
		$statement = $this->_pdo->prepare($sql);
		$statement->execute($params);

		return $statement->fetch(PDO::FETCH_ASSOC);
	}

	/**
	 * Vyčistí identifikátor (vyházením nealfanumerických znaků).
	 *
	 * @param string $identifier Identifikátor k vyčištění.
	 *
	 * @return string Čistý identifikátor.
	 */
	protected function _quoteIdentifier($identifier)
	{
		return preg_replace('/[^a-z0-9_-]/i', '', $identifier);
	}

	/**
	 * Získá seznam sloupců tabulky.
	 *
	 * @param string $table Název tabulky.
	 *
	 * @return array Seznam sloupců.
	 */
	protected function _getColumns($table)
	{
		if (isset($this->_columnsByTable[$table])) {
			return $this->_columnsByTable[$table];
		}
		$columns = array();
		$fields = $this->fetchArray('SHOW COLUMNS FROM ' . $this->_quoteIdentifier($table));
		foreach ($fields as $field) {
			$columns[] = $field['Field'];
		}
		$this->_columnsByTable[$table] = $columns;

		return $columns;
	}

	/**
	 * Vloží data do tabulky.
	 *
	 * @param string $table Název tabulky, do které budeme data vkládat.
	 * @param array  $data  Řádek dat jako pole.
	 *
	 * @return string ID vloženého záznamu.
	 */
	public function insertRow($table, $data)
	{
		$validColumns = $this->_getColumns($table);
		$keys = array();
		$values = array();
		foreach ($data as $key => $val) {
            if ($val==='' || $val==='0000-00-00') {
                continue; // přeskakujeme prázdné hodnoty
            }

			if (in_array($key, $validColumns)) {
				$keys[] = $this->_quoteIdentifier($key);

				// umíme ukládat base64 do DB
				if (is_object($val) && get_class($val)=='stdClass' && $val->xmlrpc_type=='base64') {
					$val = $val->scalar;
				}

				$values[] = $this->_pdo->quote($val);
			}
		}

		$sql = 'INSERT INTO ' . $this->_quoteIdentifier($table) . ' (' . implode(',', $keys) . ') VALUES (' . implode(',', $values) . ')';
		try {
			$this->_pdo->exec($sql);
		} catch (\PDOException $exception) {
			throw new DatabaseException($exception->getMessage() . PHP_EOL . $sql, 0, $exception);
		}

		return $this->_pdo->lastInsertId();
	}
}
