<?php

class Dalten_Export_Api_Idnes implements Dalten_Export_Api_LoggableApiInterface, Dalten_Export_Api_TopListingInterface, Dalten_Export_Api_ListListingsInterface
{
	/**
	 * @var Dalten_Export_Api_Backend_Idnes|null
	 */
	private $_backend = null;

	public function __construct(
		Dalten_Export_Idnes $export,
		Serenity_Config_Config $config,
		Dalten_Export_Api_Backend_Idnes $backend)
	{
		$this->_export = $export;
		$this->_config = $config;
		$this->_backend = $backend;
	}

	/**
	 * Naváže spojení se vzdáleným serverem.
	 *
	 * @param string $login			Přihlašovací jméno.
	 * @param string $password		 Heslo.
	 * @param string $softwareKey	  Softwarový klíč.
	 * @param array  $additionalParams Pole dalších parametrů.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function openConnection($login, $password, $softwareKey = '', array $additionalParams = array())
	{
		$this->_backend->setUsername($login)
			->setPassword($password);
		return $this->_backend->connect();
	}

	/**
	 * Vyexportuje nabídku.
	 *
	 * @param array $listingData      Data nabídky ve formátu iRest 1.
	 * @param array $userData         Data uživatele (makléře, kterému nabídka patří) ve formátu iRest 1.
	 * @param array $images           Pole fotografií nabídky ve formátu iRest 1.
	 * @param array $additionalParams Pro tento export, nejsou přídavná data potřebná.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru, jestli se export podařil nebo co selhalo.
	 */
	public function addListing(array $listingData, array $userData, array $images = array(), array $additionalParams = array())
	{
		$this->_export->setAdditionalParams($additionalParams);
		$data = $this->_export->convertEntityValues('listing', $listingData, $this->_config->listing);

		// Přidáme příznak, že se jedná o update nabídky.
		// Idnes má příznaky pouze update a delete. Takže pokud se jedná o novou nabídku je to boj Idnesu.
		$data['akce'] = 'update';

		if (isset($additionalParams['listing']['hide_price'])) {
			$data['b_cena_hide'] = (boolean) $additionalParams['listing']['hide_price'];
		}
		$response = $this->_backend->addListing($data, $userData, $images);

		return $response;
	}

	/**
	 * Odstraní nabídku ze vzdáleného serveru.
	 *
	 * @param array $listingData      Data nabídky ve formátu iRest 1.
	 * @param array $additionalParams Specifická data pro daný export (zde prázdné pole).
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function deleteListing(array $listingData, array $additionalParams = array())
	{
		$this->_export->setAdditionalParams($additionalParams);
		return $this->_backend->deleteListing($listingData['id']);
	}

	/**
	 * Vytopuje nemovitost.
	 *
	 * @param array $listingData      Data nemovitosti.
     * @param array $additionalParams Další parametry (nepoužívá se).
	 *
	 * @return Dalten_Export_Api_ServerResponseVerifiable Odpověď serveru.
	 */
	public function topListing(array $listingData, array $additionalParams = array())
	{
		return $this->_backend->hopListing($listingData['id']);
	}


	/**
	 * Vrátí seznam inzerátů na serveru.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function getListingsList()
	{
		return $this->_backend->getListingsList();
	}

    /**
     * Získá dotazy k nemovitosti za datum.
     *
     * @param DateTime $dateTime Datum ke kterému se vztahujeme.
     * @return Dalten_Export_Api_ServerResponse Dotazy k nemovitosti.
     */
	public function getResponses(DateTime $dateTime)
    {
        return $this->_backend->getResponses($dateTime);
    }

    /**
     * Získá statistiky nemovitostí pro datum.
     *
     * @param DateTime $dateTime Datum ke kterému se vztahujeme.
     * @return Dalten_Export_Api_ServerResponse Statistiky nemovitostí.
     */
    public function getStats(DateTime $dateTime)
    {
        return $this->_backend->getStats($dateTime);
    }

    /**
     * {@inheritDoc}
     */
    public function listListings()
    {
        return $this->_backend->listListings();
    }

	/**
	 * Vždy uspěje.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function closeConnection(){
		return new Dalten_Export_Api_ServerResponse(true);
	}

	/**
	 * Nastaví logger pro backend.
	 *
	 * Logger bude použit pouze pokud to backend dovoluje.
	 *
	 * @param Dalten_Export_Api_Backend_Logger_LoggerInterface $logger Instance loggeru.
	 *
	 * @return bool Podařilo se přiřadit backendu logger?
	 */
	public function setLogger(Dalten_Export_Api_Backend_Logger_LoggerInterface $logger)
	{
		if ($this->_backend instanceof Dalten_Export_Api_Backend_LoggableBackendInterface) {
			$this->_backend->setLogger($logger);
		}
		return false;
	}

	/**
	 * Odstraní nastavený logger pro backend.
	 *
	 * @return Dalten_Export_Api_LoggableApiInterface Fluent interface.
	 */
	public function removeLogger()
	{
		$this->_backend->removeLogger();
	}
}
