<?php
/**
 * Exportní API serveru Reality.sk.
 *
 * @category   Dalten
 * @package    Export
 * @subpackage Api
 */
class Dalten_Export_Api_RealityCz implements Dalten_Export_Api_ApiInterface, Dalten_Export_Api_VerifiableApiInterface,
	Dalten_Export_Api_LoggableApiInterface, Dalten_Export_Api_TopListingInterface
{

	/**
	 * Backend ktery se stara o exportovani dat.
	 *
	 * @var \Dalten_Export_Api_Backend_RealityCz|null
	 */
	private $_backend = null;

	/**
	 * Data expotu.
	 *
	 * @var \Dalten_Export_RealityCz|null
	 */
	private $_export = null;

	/**
	 * Konfigurace exportu.
	 *
	 * @var null|\Serenity_Config_Config
	 */
	private $_config = null;

	private $_adresar  =null;

	/**
	 * Nastavi vlastnosti objektu.
	 *
	 * @param Dalten_Export_RealityCz                     $export  Data exportu.
	 * @param Serenity_Config_Config                      $config  Nastaveni exportu.
	 * @param Dalten_Export_Api_Backend_RealityCz         $backend Rozhrani pro komunikaci
	 *                                                             s exportnim serverem.
	 * @param Dalten_AddressConverter_RealityCz_Interface $adresar Převodník adres.
	 */
	public function __construct(Dalten_Export_RealityCz $export,
		Serenity_Config_Config $config,
		Dalten_Export_Api_Backend_RealityCz $backend,
		Dalten_AddressConverter_RealityCz_Interface $adresar
	)
	{
		$this->_export = $export;
		$this->_config = $config;
		$this->_backend = $backend;
		$this->_adresar = $adresar;
	}

	/**
	 * Naváže spojení se vzdáleným serverem.
	 *
	 * @param string $login            Přihlašovací jméno.
	 * @param string $password         Heslo.
	 * @param string $softwareKey      Softwarový klíč.
	 * @param array  $additionalParams Pole dalších parametrů (musí obsahovat klíč "salt").
	 *
	 * @return \Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function openConnection($login, $password, $softwareKey = '', array $additionalParams = array())
	{
		return $this->_backend->login($login, $password);
	}

	/**
	 * Vyexportuje nabídku.
	 *
	 * @param array $listingData      Data nabídky ve formátu iRest 1.
	 * @param array $userData         Data uživatele (makléře, kterému nabídka patří) ve formátu iRest 1.
	 * @param array $images           Pole fotografií nabídky ve formátu iRest 1.
	 * @param array $additionalParams Parametry potrebne pro editaci.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru, jestli se export podařil nebo co selhalo.
	 */
	public function addListing(
		array $listingData,
		array $userData,
		array $images = array(),
		array $additionalParams = array()
	)
	{
        if (!empty($additionalParams['listing']['hide_price'])) {
            // Cena dohodou
            $listingData['nemovitost_cena_prodej'] = 0;
            $listingData['nemovitost_cena_pronajem'] = 0;
            $listingData['nemovitost_cena_poznamka'] = 'Informace v RK';
        }

		$listingData = $this->_adresar->convertAddress($listingData);
		$listingData['kod'] = $this->_getKod($listingData, $additionalParams);
		$data = $this->_export->convertEntityValues('listing', $listingData, $this->_config->listing);
		return $this->_backend->sendEstate($data, $userData, $images, $additionalParams);
	}

	/**
	 * Trvale odstraní nabídku ze vzdáleného serveru.
	 *
	 * @param array $listingData      Data nabídky ve formátu iRest 1.
	 * @param array $additionalParams Další parametry exportu.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpoveď serveru.
	 */
	public function deleteListing(array $listingData, array $additionalParams = array())
	{
		// staci nam jen kod
		$data = array(
			'vsCisloZakazky' => $this->_getKod($listingData, $additionalParams)
		);

		return $this->_backend->deleteEstate($data);
	}

	/**
	 * Implementace interface.
	 *
	 * @throws BadMethodCallException
	 */
	public function getListingsList()
	{
		throw new BadMethodCallException("Method 'getListingsList' IS NOT implemented yet.");
	}

	/**
	 * Vždy uspěje.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function closeConnection()
	{
		return new Dalten_Export_Api_ServerResponse(true);
	}

	/**
	 * Ověří si v jakém stavu je export nabídky nemovitosti.
	 *
	 * @param array $listingData      Data nemovitosti.
	 * @param array $additionalParams (Nepoužívá se)
	 *
	 * @return Dalten_Export_Api_ServerResponseVerifiable Odpověď serveru obsahující informaci o ověření.
	 */
	public function verifyListing(array $listingData, array $additionalParams)
	{
		try {
			$listingData = $this->_adresar->convertAddress($listingData);
			$listingData['kod'] = $this->_getKod($listingData, $additionalParams);
			$data = $this->_export->convertEntityValues('listing', $listingData, $this->_config->listing);
			return $this->_backend->checkEstate($data, $additionalParams);
		} catch (Exception $e) {
			return new Dalten_Export_Api_ServerResponseVerifiable(false);
		}
	}

	/** {@inheritDoc} */
    public function topListing(array $listingData, array $additionalParams = array())
    {
        // staci nam jen kod
        $data = array(
            'vsCisloZakazky' => $this->_getKod($listingData, [])
        );

        return $this->_backend->topListing($data);
    }

    /**
	 * Najde kód nemovitosti. Nejprve zkusí vlastní kód, potom kód z nabídky.
	 *
	 * @param array $listingData      Data nemovitosti.
	 * @param array $additionalParams Pole dalších parametrů.
	 *
	 * @return string Nalezený kód nemovitosti.
	 */
	protected function _getKod(array $listingData, array $additionalParams)
	{
		if (!empty($additionalParams['listing']['custom_id'])) {
			return $additionalParams['listing']['custom_id']; // vlastní kód z dvojky
		}
		if (!empty($additionalParams['cislo'])) {
			return $additionalParams['cislo']; // vlastní kód z jedničky
		}
		return $listingData['kod'];
	}

	/**
	 * Nastaví logger pro backend.
	 *
	 * Logger bude použit pouze pokud to backend dovoluje.
	 *
	 * @param Dalten_Export_Api_Backend_Logger_LoggerInterface $logger Instance loggeru.
	 *
	 * @return bool Podařilo se přiřadit backendu logger?
	 */
	public function setLogger(Dalten_Export_Api_Backend_Logger_LoggerInterface $logger)
	{
		$this->_backend->setLogger($logger);
		return true;
	}

	/**
	 * Odstraní nastavený logger pro backend.
	 *
	 * @return Dalten_Export_Api_LoggableApiInterface Fluent interface.
	 */
	public function removeLogger()
	{
		$this->_backend->removeLogger();
		return true;
	}

    public function ListFeedbacks(DateTime $dateTime)
    {
        if ($this->_backend instanceof Dalten_Export_Api_Backend_RealityCzNew) {
            return $this->_backend->ListFeedbacks($dateTime);
        }
        throw new BadMethodCallException('Puvodni backend nepodporuje metodu ListFeedbacks.');
    }
}
