<?php
/**
 * Export do formátu Českých realit.
 *
 * @category Dalten
 * @package  Export
 */
class Dalten_Export_Ceskereality extends Dalten_Export_AbstractExport
{
	/**
	 * Zformátuje a podstrčí některé informace před exportem skrz Newspeak.
	 *
	 * @param array  $values     Pole s daty (POZOR! po zákroku hyperoptimalizátoru).
	 * @param string $config     Definice entity.
	 * @param string $entityName Jméno entity.
	 *
	 * @return array Zformátovaná data.
	 */
	protected function _format(array $values, $config, $entityName)
	{
		if ($entityName == 'listing') {
			if (!empty($values['nemovitost_podlazi_cislo'])) {
				$values['_patro'] = $this->_podlaziCisloNaText($values['nemovitost_podlazi_cislo']);
			}

			$values['typceny'] = $this->_tranformPriceUnit($values);

			$values['vybaveni'] = $this->_constructVybaveni($values);
		}
		return $values;
	}

	/**
	 * Provede závěrečné úpravy po převodu Newspeakem.
	 *
	 * @param array  $convertedValues Převedené hodnoty.
	 * @param array  $originalValues  Originální údaje o nemovitosti.
	 * @param string $entityName      Jméno entity.
	 *
	 * @return array Definitivně převedená data entity.
	 */
	protected function _convert(array $convertedValues, array $originalValues, $entityName)
	{
		if ($entityName == 'listing') {
			$cis = $this->_getOldCiselnik();
			$r = $originalValues;

			switch ($originalValues['nemovitost_typ']) {
				case 2:
					$podtyp = $originalValues['nemovitost_ucel_budovy'];
					break;
				case 3:
					$podtyp = $originalValues['nemovitost_druh_pozemku'];
					break;
				case 4:
				    $podtyp = $originalValues['nemovitost_dispozice'];
				    break;
				case 5:
				    $podtyp = $originalValues['nemovitost_druh_objektu'];
				    break;
				case 7:
				    $podtyp = $originalValues['nemovitost_typ_zarizeni'];
				    break;
				case 9:
				    $podtyp = $originalValues['nemovitost_druh_prostor'];
				    break;
				case 6:
					$podtyp = $originalValues['nemovitost_poloha_objektu'];
					break;
				case 10:
					$podtyp = $originalValues['nemovitost_typ_rekreacniho_objektu'];
					break;
				case 11:
					$podtyp = $originalValues['nemovitost_typ_maleho_objektu'];
					break;
				default:
					$podtyp = 0;
					break;
			}

			$convertedValues['typ'] = $cis['typ'][$r['nemovitost_typ']][$podtyp][0];
			$convertedValues['subtyp'] = $cis['typ'][$r['nemovitost_typ']][$podtyp][1];

			$plochy = $this->_tranformAreas($originalValues);

			if (!empty($originalValues['nemovitost_popis_zakladni'])) {
				$convertedValues['popis'] =
					$originalValues['nemovitost_popis_zakladni'] . ' ' . $originalValues['nemovitost_popis_detailni'];
			}

			$convertedValues = $convertedValues + $plochy;

		}

		if ($entityName=='broker') {
			$jmenoCele = $originalValues['uzivatel_os_jmeno'] . (!empty($originalvalues['uzivatel_os_prostredni_jmeno']) ?  ' ' . $originalvalues['uzivatel_os_prostredni_jmeno'] : '') . ' ' . $originalValues['uzivatel_os_prijmeni'];

			if (!empty($originalValues['uzivatel_tituly_pred_jmenem'])) {
				$jmenoCele = $originalValues['uzivatel_tituly_pred_jmenem'] . ' ' . $jmenoCele;
			}

			if (!empty($originalValues['uzivatel_tituly_za_jmenem'])) {
				$jmenoCele = $jmenoCele . ' ' . $originalValues['uzivatel_tituly_za_jmenem'];
			}

			$convertedValues['makler_jmeno'] = $jmenoCele;
		}

		return $convertedValues;
	}



	/**
	 * Pomocná funkce. Obsahuje převodník typů
	 *
	 * @return array Převodník typů a podtypů vykopírovaný ze starého exportu.
	 */
	private function _getOldCiselnik()
	{
		$cis['nemovitost_typ'][4] = 2;	//Byty
		$cis['nemovitost_typ'][6] = 0;	//	Domy a vily
		$cis['nemovitost_typ'][10] = 1;	//	Chaty a rekreační objekty
		$cis['nemovitost_typ'][1] = 11;	//	Zemědělské objekty
		$cis['nemovitost_typ'][3] = 5;	//	Pozemky
		$cis['nemovitost_typ'][11] = 11;	//	Malé objekty, garáže
		$cis['nemovitost_typ'][9] = 8;	//	Komerční prostory
		$cis['nemovitost_typ'][2] = 4;	//	Komerční objekty
		$cis['nemovitost_typ'][5] = 10;	//	Historické objekty
		$cis['nemovitost_typ'][7] = 6;	//	Hotely, penziony a restaurace
		$cis['nemovitost_typ'][8] = 3;	//	Nájemní domy
		$cis['typ'][1][0]= array( 4,3); //Zemědělské objekty
		$cis['typ'][2][1]= array( 9,4); //Komerční objekty Sklad
		$cis['typ'][2][2]= array( 4,1); //Komerční objekty Výroba
		$cis['typ'][2][3]= array( 9,3); //Komerční objekty Obchodní centrum
		$cis['typ'][2][4]= array( 9,3); //Komerční objekty Nákupní středisko
		$cis['typ'][2][5]= array( 4,5); //Komerční objekty Průmyslový areál
		$cis['typ'][2][6]= array( 4,5); //Komerční objekty Jiný
		$cis['typ'][2][7]= array( 8,1); //Komerční objekty Administrativní budova
		$cis['typ'][3][1]= array( 5,14) ; //Pozemky Pro komerční výstavbu
		$cis['typ'][3][2]= array( 5,2); //Pozemky Pro bydlení
		$cis['typ'][3][3]= array( 5,1); //Pozemky Zemědělská
		$cis['typ'][3][4]= array( 5,5); //Pozemky Les
		$cis['typ'][3][5]= array( 5,13); //Pozemky Trvalý travní porost
		$cis['typ'][3][6]= array( 5,4); //Pozemky Zahrada
		$cis['typ'][3][7]= array( 5,13); //Pozemky Ostatní
		$cis['typ'][4][1]= array( 2,0); //Byty Garsoniéra
		$cis['typ'][4][2]= array( 2,1); //Byty 1+kk
		$cis['typ'][4][3]= array( 2,10); //Byty 2+kk
		$cis['typ'][4][4]= array( 2,12); //Byty 3+kk
		$cis['typ'][4][5]= array( 2,13); //Byty 4+kk
		$cis['typ'][4][6]= array( 2,14); //Byty 5+kk
		$cis['typ'][4][7]= array( 2,6); //Byty 6+kk
		$cis['typ'][4][8]= array( 2,6); //Byty 7+kk
		$cis['typ'][4][9]= array( 2,2); //Byty 1+1
		$cis['typ'][4][10]= array( 2,3); //Byty 2+1
		$cis['typ'][4][11]= array( 2,4); //Byty 3+1
		$cis['typ'][4][12]= array( 2,5); //Byty 4+1
		$cis['typ'][4][13]= array( 2,6); //Byty 5+1
		$cis['typ'][4][14]= array( 2,6); //Byty 6+1
		$cis['typ'][4][15]= array( 2,6); //Byty 7+1
		$cis['typ'][4][16]= array( 2,16); //Byty Atypický
		$cis['typ'][4][17]= array( 2,8); //Byty Jiný
		$cis['typ'][5][1]= array( 10,5); //Historické objekty Dřevěná
		$cis['typ'][5][2]= array( 10,5); //Historické objekty Cihlová
		$cis['typ'][5][3]= array( 10,5); //Historické objekty Kamenná
		$cis['typ'][5][4]= array( 10,5); //Historické objekty Montovaná
		$cis['typ'][5][5]= array( 10,5); //Historické objekty Panelová
		$cis['typ'][5][6]= array( 10,5); //Historické objekty Skeletová
		$cis['typ'][5][7]= array( 10,5); //Historické objekty Smíšená
		$cis['typ'][6][1]= array( 0,1); //Domy a vily Řadový
		$cis['typ'][6][2]= array( 0,1); //Domy a vily Rohový
		$cis['typ'][6][3]= array( 0,1); //Domy a vily V bloku
		$cis['typ'][6][4]= array( 0,1); //Domy a vily Samostatný
		$cis['typ'][7][1]= array( 6,0); //Hotely, penziony a restaurace Hotel
		$cis['typ'][7][2]= array( 6,2); //Hotely, penziony a restaurace Penzion
		$cis['typ'][7][3]= array( 7,1); //Hotely, penziony a restaurace Restaurace
		$cis['typ'][8][0]= array( 3,0); //Nájemní domy
		$cis['typ'][9][1]= array( 8,0); //Komerční prostory Kanceláře
		$cis['typ'][9][2]= array( 9,1); //Komerční prostory Obchodní
		$cis['typ'][9][3]= array( 9,4); //Komerční prostory Skladovací
		$cis['typ'][9][4]= array( 9,6); //Komerční prostory Jiné
		$cis['typ'][9][5]= array( 4,1); //Komerční prostory Výrobní
		$cis['typ'][10][1]= array( 1,0); //Chaty a rekreační objekty - Chata
		$cis['typ'][10][2]= array( 1,1); //Chaty a rekreační objekty - Chalupa
		$cis['typ'][11][1]= array( 11,0); //Malé objekty, garáže - Garáže
		$cis['typ'][11][2]= array( 11,6); //Malé objekty, garáže - Ostatní
		$cis['typ'][11][3]= array( 11,0); //Malé objekty, garáže - Garážové stání
		$cis['typ'][11][4]= array( 1,4); //Malé objekty, garáže - Vinný sklep


		return $cis;
	}

	/**
	 * Převede naše plochy na plochy Českých realit. Filozofie převodu převzaná z původního exportu.
	 *
	 * @param array $originalValues Pole obsahující originální informace o nemovitosti.
	 *
	 * @return array Pole jejich ploch.
	 */
	private function _tranformAreas(array $originalValues)
	{
		$plocha['1']  = $originalValues['nemovitost_plocha_parcely'];
		$plocha['2']  = $originalValues['nemovitost_plocha_celkova'];
		$plocha['3']  = $originalValues['nemovitost_plocha_celkova'];
		$plocha['4']  = $originalValues['nemovitost_plocha_celkova'];
		$plocha['5']  = $originalValues['nemovitost_plocha_celkova'];
		$plocha['6']  = $originalValues['nemovitost_plocha_parcely'];
		$plocha['7']  = $originalValues['nemovitost_plocha_celkova'];
		$plocha['8']  = $originalValues['nemovitost_plocha_uzitna'];
		$plocha['9']  = $originalValues['nemovitost_plocha_uzitna'];
		$plocha['10'] = $originalValues['nemovitost_plocha_parcely'];
		$plocha['11'] = $originalValues['nemovitost_plocha_zastavena'];

		$areas = array();
		$areas['pl_celkova'] = $plocha[ $originalValues['nemovitost_typ'] ];
		if ($originalValues['nemovitost_plocha_zastavena']) {
			$areas['pl_zastavena'] = $originalValues['nemovitost_plocha_zastavena'];
		}
		if ($originalValues['nemovitost_plocha_uzitna']) {
			$areas['pl_uzitna'] = $originalValues['nemovitost_plocha_uzitna'];
		}

		return $areas;
	}

	/**
	 * Převede číslo podlaží na text.
	 *
	 * @param int $cisloPodlazi Číslo podlaží
	 *                          (viz http://upload.wikimedia.org/wikipedia/commons/2/24/Podlazi_vs_patro.svg)
	 *
	 * @return null|string Textové vyjádření v patrech.
	 */
	private function _podlaziCisloNaText($cisloPodlazi)
	{
		if ($cisloPodlazi == 1) {
			return 'přízemí';
		} elseif ($cisloPodlazi>1) {
			return ($cisloPodlazi-1) . '. patro';
		} elseif ($cisloPodlazi<0) {
			return abs($cisloPodlazi) . '. podzemní podlaží';
		}
		return null;
	}

	/**
	 * Převede jednotku ceny.
	 *
	 * @param array $originalData Původní informace o nemovitosti.
	 *
	 * @return int|null Jejich výsledná jednotka ceny (měna+za co).
	 */
	private function _tranformPriceUnit(array $originalData)
	{
		//CZK
		$jednotky['1_1'] = 0; //	za nemovitost
		$jednotky['1_3'] = 2; //	za m2
		$jednotky['1_2'] = 1; //	za měsíc
		$jednotky['1_4'] = 3; //	za m2/měsíc
		$jednotky['1_5'] = 16; //	za m2/rok
		$jednotky['1_6'] = 19; //	za rok
		$jednotky['1_7'] = 22; //	za den
		//USD
		$jednotky['2_1'] = 8; //	za nemovitost
		$jednotky['2_3'] = 10; //	za m2
		$jednotky['2_2'] = 9; //	za měsíc
		$jednotky['2_4'] = 11; //	za m2/měsíc
		$jednotky['2_5'] = 17; //	za m2/rok
		$jednotky['2_6'] = 20; //	za rok
		$jednotky['2_7'] = 23; //	za den
		//EUR
		$jednotky['3_1'] = 12; //	za nemovitost
		$jednotky['3_3'] = 14; //	za m2
		$jednotky['3_2'] = 13; //	za měsíc
		$jednotky['3_4'] = 15; //	za m2/měsíc
		$jednotky['3_5'] = 18; //	za m2/rok
		$jednotky['3_6'] = 21; //	za rok
		$jednotky['3_7'] = 24; //	za den

		$jednotkaCeny = (
			$originalData['zakazka_typ']==1 ?
				$originalData['nemovitost_cena_prodej_jednotka'] :
				$originalData['nemovitost_cena_pronajem_jednotka']
		);
		if (isset($jednotky[$originalData['nemovitost_mena'] . '_' . $jednotkaCeny ])) {
			return $jednotky[$originalData['nemovitost_mena'] . '_' . $jednotkaCeny ];
		}
		return null;
	}

	/**
	 * Projde data nemovitosti a pokusí se vyzobat co nejvíc údajů o vybavení nemovitosti.
	 *
	 * Jejich vybavení  je položka ve stylu <tt>01100 000 1001 00011</tt>.
	 *
	 * @param array $originalData Původní informace o nemovitosti.
	 *
	 * @return string Vybavení jak ho definují České reality.
	 */
	private function _constructVybaveni(array $originalData)
	{
		$groupSet = array();

		// Tady si definuju jak jsou jednotlivé skupiny dlouhé (a v komentářích, co to je).
		$groupLen = array(
			0 => 11, // Dopravní spojení
			1 => 10, // Parkování
			2 => 7, // Příjezd
			3 => 7, // Elektrický proud
			4 => 9, // Kanalizace
			5 => 3, // Kvalita vody
			6 => 7, // Rozvod plynu
			7 => 9, // Vytápění
			8 => 7, // Využívané palivo
			9 => 10, // Zavedené inž. sítě
			10 => 6, // Zdroj teplé vody
			11 => 8, // Zdroj vody
			12 => 6, // Způsob vytápění
			13 => 12, // Druh krytiny
			14 => 9, // Druh omítky
			15 => 4, // Izolace proti vlhku
			16 => 15, // Podlahy
			17 => 3, // Stav vlhkosti
			18 => 5, // Stěny
			19 => 5, // Stropy
			20 => 6, // Tvar krovu
			21 => 11, // Typ konstrukce
			22 => 9, // WC
			23 => 7, // Zahrada
			24 => 12, // Nebytové prostory
			25 => 8, // Okolní prostranství
			26 => 10, // Vedlejší stavby
			27 => 5, // Okna
			28 => 2, // Balkony a Lodžie
		);

		//definice jsem ponechal pro případné snazší dopisování

		// Definice jsou ve tvarech
		// array("kod_checkoboxu");
		// array("kod_selectu_nebo_multiselectu", $index);
		// Samotný string v definici je de facto komentář.

		//Dopravní spojení
		$groupDef[0][0] = array('nemovitost_doprava', 4);
		$groupDef[0][1] = "ČSAD";
		$groupDef[0][2] = array('nemovitost_doprava', 0);
		$groupDef[0][3] = "Jiná";
		$groupDef[0][4] = "Letecká doprava";
		$groupDef[0][5] = "Lodní doprava";
		$groupDef[0][6] = "Metro";
		$groupDef[0][7] = array('nemovitost_doprava', 3);
		$groupDef[0][8] = "Mimo dosah";
		$groupDef[0][9] = "Příjezd nákladním autem";
		$groupDef[0][10] = array('nemovitost_doprava', 2);
		//Parkování
		$groupDef[1][0] = "Garáž - 1 auto";
		$groupDef[1][1] = "Garáž - 2 a více aut";
		$groupDef[1][2] = "Není";
		$groupDef[1][3] = "Parkování i pro autobusy";
		$groupDef[1][4] = "Parkování i pro kamiony";
		$groupDef[1][5] = "Parkování na ulici";
		$groupDef[1][6] = "Parkoviště poblíž";            // napsat ify
		$groupDef[1][7] = "Podzemní pakoviště";
		$groupDef[1][8] = array('nemovitost_ostatni', 5);
		$groupDef[1][9] = "Ano";
		if (
			(isset($originalData['nemovitost_pocet_mist']) && $originalData['nemovitost_pocet_mist'] > 0) ||
			(isset($originalData['nemovitost_parkovaci_stani']) && $originalData['nemovitost_parkovaci_stani']==1) ||
			(isset($originalData['nemovitost_garaz']) && $originalData['nemovitost_garaz']==1) ||
			(isset($originalData['nemovitost_ostatni']) && substr($originalData['nemovitost_ostatni'], 3, 1)==1)
		) {
			// máme nějaké parkovací místo.
			$groupSet[1][9] = 1;
		}
		//Příjezd
		$groupDef[2][0] = "Není";
		$groupDef[2][1] = "Nezpevněný";
		$groupDef[2][2] = "Pro kamiony";
		$groupDef[2][3] = "Pro nákladní auta";
		$groupDef[2][4] = "Rampa";
		$groupDef[2][5] = "Zpevněný";
		$groupDef[2][6] = "Ano";
		//Elektrický proud
		$groupDef[3][0] = array('nemovitost_elektrina', 0);
		$groupDef[3][1] = array('nemovitost_elektrina', 1);
		$groupDef[3][2] = array('nemovitost_elektrina', 3);
		$groupDef[3][3] = "Na hranici pozemku";
		$groupDef[3][4] = "Není";
		$groupDef[3][5] = "Vlastní zdroj";
		$groupDef[3][6] = "Ano";
		//Kanalizace
		$groupDef[4][0] = "Dešťová";
		$groupDef[4][1] = "Není";
		$groupDef[4][2] = array('nemovitost_odpad', 1);
		$groupDef[4][3] = array('nemovitost_odpad', 0);
		$groupDef[4][4] = "Vlastní ČOV";
		$groupDef[4][5] = "Žumpa, septik";
		$groupDef[4][6] = array('nemovitost_odpad', 2);
		$groupDef[4][7] = "Septik";
		$groupDef[4][8] = "Ano";
		//Kvalita vody
		$groupDef[5][0] = "Pitná, upravená";
		$groupDef[5][1] = "Studniční";
		$groupDef[5][2] = "Užitková";
		//Rozvod plynu
		$groupDef[6][0] = "Bioplyn";
		$groupDef[6][1] = "Na hranici pozemku";
		$groupDef[6][2] = "Není";
		$groupDef[6][3] = "Propanbutan";
		$groupDef[6][4] = "Svítiplyn";
		$groupDef[6][5] = "Zemní plyn";
		$groupDef[6][6] = "Ano"; // napsat ifánek - jak při indiv tak při plynov
		if (isset($originalData['nemovitost_plyn'])) {
			$plyn = $originalData['nemovitost_plyn'];
			if (substr($plyn, 0, 1)==1 || substr($plyn, 1, 1)) {
				$groupSet[6][6] = 1; //máme plyn
			}
		}
		//Vytápění
		$groupDef[7][0] = "Bez vytápění";
		$groupDef[7][1] = "Elektrická topidla";
		$groupDef[7][2] = "Infrazářiče";
		$groupDef[7][3] = "Jiné";
		$groupDef[7][4] = "Klimatizace";
		$groupDef[7][5] = "Plynová topidla";
		$groupDef[7][6] = "Radiátory";
		$groupDef[7][7] = "Topení podlahou";
		$groupDef[7][8] = "Ano";
		//Využívané palivo
		$groupDef[8][0] = "Elektřina";
		$groupDef[8][1] = "Jiný zdroj";
		$groupDef[8][2] = "Kapalná paliva";
		$groupDef[8][3] = "Není";                 // napsat si ifánky
		$groupDef[8][4] = "Pevná paliva";
		$groupDef[8][5] = "Plyn";
		$groupDef[8][6] = "Solární energie";
		//Zavedené inž. sítě
		$groupDef[9][0] = "Dosud nezavedeny";
		$groupDef[9][1] = "Elektrický proud";
		$groupDef[9][2] = array('nemovitost_telekomunikace', 1);
		$groupDef[9][3] = array('nemovitost_ostatni_rozvody', 1);
		$groupDef[9][4] = "Kanalizace";
		$groupDef[9][5] = "Na hranici pozemku";
		$groupDef[9][6] = "Plyn";
		$groupDef[9][7] = array('nemovitost_telekomunikace', 0);
		$groupDef[9][8] = "Vodovod";
		$groupDef[9][9] = "Ano";
		if (isset($originalData['nemovitost_elektrina']) && strpos($originalData['nemovitost_elektrina'], '1')!==false) {
			$groupSet[9][1] = 1; //máme elektřinu
		}
		//Zdroj teplé vody
		$groupDef[10][0] = "Bojler";
		$groupDef[10][1] = "Dálkový";
		$groupDef[10][2] = "Kotel UT";
		$groupDef[10][3] = "Není";
		$groupDef[10][4] = "Průtokový ohřívač";
		$groupDef[10][5] = "Ano";
		//Zdroj vody
		$groupDef[11][0] = "Domácí vodárna";
		$groupDef[11][1] = "Na hranici pozemku";
		$groupDef[11][2] = "Není";
		$groupDef[11][3] = "Pramen";
		$groupDef[11][4] = "Studna kopaná";
		$groupDef[11][5] = "Studna vrtaná";
		$groupDef[11][6] = "Veřejný vodovod";
		$groupDef[11][7] = "Ano";

		if (isset($originalData['nemovitost_voda'])) {
			$voda = $originalData['nemovitost_voda'];
			if (substr($voda, 0, 1)==1 || substr($voda, 1, 1) || substr($voda, 2, 1)) {
				$groupSet[11][6] = 1; // máme nějakou vodu
			}
			if (substr($voda, 2, 1)==1) { $groupSet[10][5] = 1; } // je dokonce teplá
		}
		//Způsob vytápění
		$groupDef[12][0] = "Dalkové";
		$groupDef[12][1] = "Etážové";
		$groupDef[12][2] = "Lokální";
		$groupDef[12][3] = "Není";
		$groupDef[12][4] = "Tuhá paliva";
		$groupDef[12][5] = "Wavky";

		if (isset($originalData['nemovitost_topeni'])) {
			$topeni = $originalData['nemovitost_topeni'];
			if (substr($topeni, 0, 1)==1 || substr($topeni, 1, 1)==1 || substr($topeni, 2, 1)==1) {
				// máme lokální topení
				$groupSet[12][2] = 1;
			}
			if (substr($topeni, 3, 1)==1 || substr($topeni, 4, 1)==1 || substr($topeni, 5, 1)==1) {
				// máme ústřední (etážové) topení
				$groupSet[12][1] = 1;
			}
			if (substr($topeni, 6, 1)==1) {
				// máme dálkové topení
				$groupSet[12][0] = 1;
			}
			if (substr($topeni, 0, 1)==1 || substr($topeni, 3, 1)==1) {
				// topíme plynem
				$groupSet[8][5] = 1;
			}
			if (substr($topeni, 1, 1)==1 || substr($topeni, 4, 1)==1) {
				// topíme tuhými palivy
				$groupSet[8][4] = 1;
			}
			if (substr($topeni, 2, 1)==1 || substr($topeni, 5, 1)==1) {
				// topíme elektricky
				$groupSet[8][0] = 1;
			}
		}

		//Druh krytiny
		$groupDef[13][0] = "Asfaltová krytina";
		$groupDef[13][1] = "Bramac";
		$groupDef[13][2] = "Břidlice";
		$groupDef[13][3] = "Eternit";
		$groupDef[13][4] = "Hliníkový plech";
		$groupDef[13][5] = "Měděný plech";
		$groupDef[13][6] = "Pálené tašky";
		$groupDef[13][7] = "Plech pozinkovaný";
		$groupDef[13][8] = "Pochozí střecha";
		$groupDef[13][9] = "Syntetické materiály";
		$groupDef[13][10] = "Šindele";
		$groupDef[13][11] = "Tegola";
		//Druh omítky
		$groupDef[14][0] = "Bez omítky";
		$groupDef[14][1] = "Brizolit";
		$groupDef[14][2] = "Obkládaná";
		$groupDef[14][3] = "Spárované zdivo";
		$groupDef[14][4] = "Vápenná drásaná";
		$groupDef[14][5] = "Vápenná hladká";
		$groupDef[14][6] = "Vápenná odtrh.";
		$groupDef[14][7] = "Silikátová";
		$groupDef[14][8] = "Jiná";
		//Izolace proti vlhku
		$groupDef[15][0] = "Bez izolace";
		$groupDef[15][1] = "Částečně izolovaný";
		$groupDef[15][2] = "Izolovaný";
		$groupDef[15][3] = "Poškozená izolace";
		//Podlahy
		$groupDef[16][0] = "Antistatická podlaha";
		$groupDef[16][1] = "Beton";
		$groupDef[16][2] = "Dlažba keramická";
		$groupDef[16][3] = "Dlažba korková";
		$groupDef[16][4] = "Dřevěná podlaha";
		$groupDef[16][5] = "Jiná, atypická";
		$groupDef[16][6] = "Koberec";
		$groupDef[16][7] = "Mramor";
		$groupDef[16][8] = "Natíraný beton";
		$groupDef[16][9] = "Parkety";
		$groupDef[16][10] = "Průmyslová podlaha";
		$groupDef[16][11] = "PVC";
		$groupDef[16][12] = "Teraco";
		$groupDef[16][13] = "Žula";
		$groupDef[16][14] = "Plovoucí podlaha";
		//Stav vlhkosti
		$groupDef[17][0] = "Částečně zvlhlý";
		$groupDef[17][1] = "Suchý";
		$groupDef[17][2] = "Vlhký";
		//Stěny
		$groupDef[18][0] = "Dřevěné obložení";
		$groupDef[18][1] = "Omítka";
		$groupDef[18][2] = "Štuk";
		$groupDef[18][3] = "Tapety";
		$groupDef[18][4] = "Sádrokarton";
		//Stropy
		$groupDef[19][0] = "Keramický";
		$groupDef[19][1] = "Klasický dřevěný";
		$groupDef[19][2] = "Klenba";
		$groupDef[19][3] = "Panel";
		$groupDef[19][4] = "Železobeton";
		//Tvar krovu
		$groupDef[20][0] = "Hambalkový";
		$groupDef[20][1] = "Kombinovaný";
		$groupDef[20][2] = "Příhradový";
		$groupDef[20][3] = "Pultový";
		$groupDef[20][4] = "Sedlový";
		$groupDef[20][5] = "Valbový";
		//Typ konstrukce
		$groupDef[21][0] = array('nemovitost_druh_objektu', 1);
		$groupDef[21][1] = array('nemovitost_druh_objektu', 7);
		$groupDef[21][2] = "Ocelová";
		$groupDef[21][3] = "Roubená";
		$groupDef[21][4] = array('nemovitost_druh_objektu', 6);
		$groupDef[21][5] = "Z prefabrikátu";
		$groupDef[21][6] = array('nemovitost_druh_objektu', 2);
		$groupDef[21][7] = array('nemovitost_druh_objektu', 3);
		$groupDef[21][8] = "Zděná z netradičního materiálu";
		$groupDef[21][9] = "Zděná z tvárnic";
		$groupDef[21][10] = array('nemovitost_druh_objektu', 5);
		//WC
		$groupDef[22][0] = "Dva a více";
		$groupDef[22][1] = "Chemický";
		$groupDef[22][2] = "Jeden";
		$groupDef[22][3] = "Mimo budovu (byt)";
		$groupDef[22][4] = "Splachovací";
		$groupDef[22][5] = "Společný";
		$groupDef[22][6] = "Suchý";
		$groupDef[22][7] = "Žádný";
		$groupDef[22][8] = "Ano";
		//Zahrada
		$groupDef[23][0] = "Není";
		$groupDef[23][1] = "Okrasná";
		$groupDef[23][2] = "Ovocný sad";
		$groupDef[23][3] = "Park";
		$groupDef[23][4] = "Předzahrádka";
		$groupDef[23][5] = "Zeleninová";
		$groupDef[23][6] = "Ano";
		//Nebytové prostory
		$groupDef[24][0] = "Dílny";
		$groupDef[24][1] = "Garáže";
		$groupDef[24][2] = "Jiné";
		$groupDef[24][3] = "Kanceláře";
		$groupDef[24][4] = "Klub, kino";
		$groupDef[24][5] = "Nevyužité prostory";
		$groupDef[24][6] = "Obchod";
		$groupDef[24][7] = "Ordinace";
		$groupDef[24][8] = "Restaurace";
		$groupDef[24][9] = "Sklady";
		$groupDef[24][10] = "Výrobny";
		$groupDef[24][11] = "Ano";
		//Okolní prostranství
		$groupDef[25][0] = "Hory";
		$groupDef[25][1] = "Lesy";
		$groupDef[25][2] = "Městská zástavba";
		$groupDef[25][3] = "Pole";
		$groupDef[25][4] = "Rybník, jezero";
		$groupDef[25][5] = "Řeka";
		$groupDef[25][6] = "Sídliště";
		$groupDef[25][7] = array('nemovitost_charakter_zastavby', 6);
		//Vedlejší stavby
		$groupDef[26][0] = "Dílna";
		$groupDef[26][1] = "Garáž";
		$groupDef[26][2] = "Chlévy, stáje";
		$groupDef[26][3] = "Chovatelská zař.";
		$groupDef[26][4] = "Kůlna";
		$groupDef[26][5] = "Skleník";
		$groupDef[26][6] = "Stodola";
		$groupDef[26][7] = "Studna";
		$groupDef[26][8] = "Zahradní domek";
		$groupDef[26][9] = "Rodinný dům";
		//Okna
		$groupDef[27][0] = "Dřevěná";
		$groupDef[27][1] = "Plastová";
		$groupDef[27][2] = "Euro-okna";
		$groupDef[27][3] = "Francouzská";
		$groupDef[27][4] = "Střešní";
		//Balkony a Lodžie
		$groupDef[28][0] = array('nemovitost_balkon');
		$groupDef[28][1] = array('nemovitost_lodzie');

		$vybaveni = array();
		// vytvoříme prázdné vybavení
		foreach ($groupLen as $groupIdx=>$len) {
			$vybaveni[$groupIdx] = str_repeat('0', $len);
		}

		// vygenerujeme změny nadefinováné pomocí $groupDef
		foreach ($groupDef as $r=>$tmp) {
			foreach ($tmp as $c=>$def) {
				// definice musí být array, jinak je to jen komentář
				if (is_array($def)) {
					$variableName = $def[0];
					if (isset($originalData[$variableName])) {
						if (count($def)<2) {
							// když je definice jednočlenná (checkbox), zkontrolujeme zda proměná je a platí
							$groupSet[$r][$c] = ($originalData[$variableName]==1 ? 1 : 0);
						} else {
							// když je definice dvoučlená (select, multichekobox)
							$tatoHodnota = $originalData[ $variableName ];
							if ($tatoHodnota === $def[1]) {
								// zkontrolujeme nejdříb zda platí jako select
								$groupSet[$r][$c] = 1;
							} elseif (substr($tatoHodnota, $def[1], 1)) {
								// a potom zda platí jako multicheckbox
								$groupSet[$r][$c] = 1;
							} else {
								// zůstává nula, nic nezaškrtnuto
								$groupSet[$r][$c] = 0;
							}
						}
					}
				}
			}
		}
		// vyplníme změny do vybavení
		foreach ($groupSet as $groupIdx=>$chars) {
			foreach ($chars as $charIdx=>$char) {
				$vybaveni[$groupIdx] = substr_replace($vybaveni[$groupIdx], $char, $charIdx, 1);
			}
		}

		return join(' ', $vybaveni);
	}
}
