<?php
/**
 * Exportní API na globální REMAX.
 *
 * @category   Dalten
 * @package    Export
 * @subpackage Api
 */
class Dalten_Export_Api_Rex implements
	Dalten_Export_Api_ApiInterface,
	Dalten_Export_Api_TransactionApiInterface,
	Dalten_Export_Api_OfficeApiInterface,
	Dalten_Export_Api_LoggableApiInterface,
	Dalten_Export_Api_CustomLimitInterface,
	Dalten_Export_Api_VerifiableApiInterface//,
	//Dalten_Export_Api_ExportTranslationInterface
{
	private $_backend = null;

	protected $_offices = array();

	/** @var array Seznam zapnutých hacků. */
	protected $_hacks = array();

	/**
	 * Konstruktor. Nastavuje závislosti.
	 *
	 * @param Dalten_Export_Api_Backend_Rex $backend Backend exportu.
	 */
	function __construct(Dalten_Export_Api_Backend_Rex $backend)
	{
		$this->_backend=$backend;
	}

	/**
	 * Naváže spojení se vzdáleným serverem.
	 *
	 * @param string $login            Přihlašovací jméno.
	 * @param string $password         Heslo.
	 * @param string $softwareKey      Softwarový klíč.
	 * @param array  $additionalParams Pole dalších parametrů.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function openConnection($login, $password, $softwareKey = '', array $additionalParams = array())
	{
		try {
			if ($this->_backend->testConnection()) {
				return new Dalten_Export_Api_ServerResponse(true);
			} else {
				return new Dalten_Export_Api_ServerResponse(false);
			}
		} catch (Dalten_Export_Exception_HumanReadable $e) {
			return new Dalten_Export_Api_ServerResponse(false, 500, $e->getMessage());
		} catch (Exception $e) {
			return new Dalten_Export_Api_ServerResponse(false, 500, 'Nastala neočekavaná chyba.');
		}
	}

	/**
	 * Začíná transakci.
	 */
	public function beginTransaction()
	{
		$this->_backend->updateFromRexapi();
		$this->_backend->openBatch();
	}


	/**
	 * Vyexportuje nabídku.
	 *
	 * @param array $listingData      Data nabídky ve formátu iRest 1.
	 * @param array $userData         Data uživatele (makléře, kterému nabídka patří) ve formátu iRest 1.
	 * @param array $images           Pole fotografií nabídky ve formátu iRest 1.
	 * @param array $additionalParams Údaje které vrací server při exportu a jsou
	 *                                potřeba pro další práci s nabídkou.
	 *                                Data můžou sloužit pro editaci apod...
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru, jestli se export podařil nebo co selhalo.
	 */
	public function addListing(
		array $listingData,
		array $userData,
		array $images = array(),
		array $additionalParams = array()
	)
	{
		$idFirma = $listingData['id_firma'];
		$firma = $this->_offices[$idFirma];

		if (!empty($additionalParams['listing']['hide_price'])) {
			$listingData['HidePricePublic'] = 'True';
		}

		if (!$this->_backend->officePresent($firma)) {
			$officeErrors = $this->_backend->officeErrors($firma);
			if (count($officeErrors)==0) {
				$this->_backend->officeAdd($firma);
			} else {
				return new Dalten_Export_Api_ServerResponseVerifiable(false, 500, 'Nepodařilo se vyexportovat kancelář!' . PHP_EOL . implode(PHP_EOL, $officeErrors));
			}
		}

		if (!$this->_backend->agentPresent($userData)) {
			$agentErrors = $this->_backend->agentErrors($userData);
			if (count($agentErrors)==0) {
				$this->_backend->agentAdd($userData);
			} else {
				return new Dalten_Export_Api_ServerResponseVerifiable(false, 500, 'Nepodařilo se vyexportovat makléře!' . PHP_EOL . implode(PHP_EOL, $agentErrors));
			}
		}

		$listingErrors =$this->_backend->listingErrors($listingData);
		if (count($listingErrors)>0) {
			return new Dalten_Export_Api_ServerResponseVerifiable(false, 500, 'Nepodařilo se vyexportovat nemovitost!' . PHP_EOL . implode(PHP_EOL, $listingErrors));
		}
		$remoteImages = array();
		if (isset($additionalParams['result']['images'])) {
			$remoteImages = $additionalParams['result']['images'];
		}
		$this->_backend->listingAdd($listingData, $images, $remoteImages);
		$uploadedImages = $this->_backend->getImagesForListing($listingData);
		$response = new Dalten_Export_Api_ServerResponseVerifiable(true);
		$response->setVerificationStatus(false);
		$response->setData(array('images'=>$uploadedImages));
		return $response;
	}

	/**
	 * Ověří si v jakém stavu je export nabídky nemovitosti.
	 *
	 * @param array $listingData      Data nemovitosti.
	 * @param array $additionalParams Údaje které vrací server při exportu a jsou potřeba pro další práci s nabídkou.
	 *                                Data můžou sloužit pro editaci apod...
	 *
	 * @return Dalten_Export_Api_ServerResponseVerifiable Odpověď serveru obsahující informaci o ověření.
	 */
	public function verifyListing(array $listingData, array $additionalParams)
	{
		$smazat = array(20,21);
		$isOnline = $this->_backend->listingPresent($listingData);

		if (isset($additionalParams['exp_pozadavek']) && in_array($additionalParams['exp_pozadavek'], $smazat)) {
			// pokud byl požadavek smazat, nesmí tam být
			$isVerified = !$isOnline;
		} else {
			// jinak tam být musí
			$isVerified = $isOnline;
		}

		$message = 'OK';
		if (!$isVerified) {
			$message = 'Nepodařilo se ověřit export nabídky.';
		}

		$remoteImages = array();
		if (isset($additionalParams['result']['images'])) {
			$remoteImages = $additionalParams['result']['images'];
		}
		$response = new Dalten_Export_Api_ServerResponseVerifiable(true, 202, $message);
		$response->setVerificationStatus(true, $isVerified);
		if ($isOnline) {
			$response->setData(array('images'=>$remoteImages));
		}
		return $response;
	}


	/**
	 * Pošle informace o kanceláři.
	 *
	 * @param array $officeData Informace o kanceláři.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function addOffice(array $officeData)
	{
		$id = $officeData['id_firma'];
		$this->_offices[$id] = $officeData;
		return new Dalten_Export_Api_ServerResponse(true);
	}

	/**
	 * Smaže kancelář z Globálu.
	 *
	 * @param array $officeData Informace o kanceláři.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function deleteOffice(array $officeData) {
		return new Dalten_Export_Api_ServerResponse(false, 500, 'Kanceláře se mažou synchronizačním skriptem.');
	}


	/**
	 * Odstraní nabídku ze vzdáleného serveru.
	 *
	 * @param array $listingData      Data nabídky ve formátu iRest 1.
	 * @param array $additionalParams Specifická data pro daný export.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function deleteListing(array $listingData, array $additionalParams = array())
	{
		$listingErrors =$this->_backend->listingErrors($listingData);
		if (count($listingErrors)>0) {
			return new Dalten_Export_Api_ServerResponseVerifiable(false, 500, 'Nepodařilo se smazat nemovitost!' . PHP_EOL . implode(PHP_EOL, $listingErrors));
		}
		$this->_backend->listingAdd($listingData, array(), array(), true);
		return new Dalten_Export_Api_ServerResponseVerifiable(true);
	}

	/**
	 * Vrátí seznam nabídek, které jsou na vzdáleném serveru.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 *
	 * @throws BadMethodCallException Jako obvykle tuto metodu nepodporujeme.
	 */
	public function getListingsList()
	{
		throw new BadMethodCallException('Function not yet implemented.');
	}

	/**
	 * Zavírá spojení.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď zda se podařilo zavřít spojení.
	 */
	public function commitTransaction()
	{
		try {
			$this->_backend->sendBatch();
			return new Dalten_Export_Api_ServerResponse(true);
		} catch (Dalten_Export_Exception_HumanReadable $exception) {
			return new Dalten_Export_Api_ServerResponse(false, 500, $exception->getMessage());
		} catch (Exception $exception) {
			//
		}
		return new Dalten_Export_Api_ServerResponse(false, 500, 'Nastala nečekaná chyba při posílání na FTP.');
	}

	/**
	 * Nastaví logger pro backend.
	 * Logger bude použit pouze pokud to backend dovoluje.
	 *
	 * @param Dalten_Export_Api_Backend_Logger_LoggerInterface $logger Instance loggeru.
	 *
	 * @return bool Podařilo se přiřadit backendu logger?
	 */
	public function setLogger(Dalten_Export_Api_Backend_Logger_LoggerInterface $logger)
	{
		if ($this->_backend instanceof Dalten_Export_Api_Backend_LoggableBackendInterface) {
			return $this->_backend->setLogger($logger);
		}
		return false;
	}

	/**
	 * Odstraní nastavený logger pro backend.
	 *
	 * @return Dalten_Export_Api_LoggableApiInterface Fluent interface.
	 */
	public function removeLogger()
	{
		if ($this->_backend instanceof Dalten_Export_Api_Backend_LoggableBackendInterface) {
			return $this->_backend->removeLogger();
		}
		return $this;
	}

	/**
	 * Vrací limit specifický pro tento export.
	 *
	 * @return int Limit.
	 */
	public function getLimit()
	{
		return 15; // to je až až
	}

	/**
	 * Vrací backend pro servisní skripty.
	 *
	 * @return Dalten_Export_Api_Backend_Rex|null Backend exportu na globál.
	 */
	public function getBackend()
	{
		return $this->_backend;
	}

	/**
	 * Nastaví seznam zaplých hacků.
	 *
	 * @param array $hacks Pole názvů hacků.
	 */
	public function setHacks(array $hacks)
	{
		$this->_hacks = $hacks;
		$this->_backend->setHacks($hacks);
	}
}
