<?php

/**
 * Export do formátu Srealit.
 *
 * @category   Dalten
 * @package    Export
 */
class Dalten_Export_Sreality extends Dalten_Export_AbstractExport
{
	/**
	 * Vrací je-li v poli $data zaškrtnuté pole $fieldName jako multicheckbox na hodnotě $value.
	 *
	 * @param array  $data      Pole s daty.
	 * @param string $fieldName Název políčka.
	 * @param int    $value     Pořadí zašktávátka.
	 *
	 * @return bool Je to tak?
	 */
	private function _isMultiChecked(array $data, $fieldName, $value) {
		if (isset($data[$fieldName])) {
			if (substr($data[$fieldName], $value, 1)==1) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Zjistí, zda máme parkovací místa.
	 *
	 * @param array $originalValues Pole původních (nepřevedených) hodnot entity.
	 *
	 * @return bool|null NULL - není vyplněno, TRUE - ano, FALSE - ne.
	 */
	protected function _convertParkingLots(array $originalValues)
	{
		$parkingLots = false;
		if (isset($originalValues['pocet_mist_k_parkovani'])) {
			$parkingLots = $originalValues['pocet_mist_k_parkovani'] > 0;
		} elseif (isset($originalValues['nemovitost_pocet_mist'])) {
			$parkingLots = $originalValues['nemovitost_pocet_mist'] > 0;
		} elseif (isset($originalValues['nemovitost_parkovaci_stani'])) {
			$parkingLots = $originalValues['nemovitost_parkovaci_stani'] > 0;
		}
		return $parkingLots;
	}

	/**
	 * Poskládá položku telekomunikace z položek telekomunikace a ostatní rozvody.
	 *
	 * @param array $originalvalues Pole původních (nepřevedených) hodnot entity.
	 *
	 * @return array Položka telekomunikace ve formátu multicheckboxu Srealit.
	 */
	protected function _convertTelekomunikace(array $originalvalues)
	{
		$telecommunication = array();
		if ($this->_isMultiChecked($originalvalues, 'nemovitost_telekomunikace', 0)) {
			$telecommunication[] = 1;  // Telefon
		}
		if ($this->_isMultiChecked($originalvalues, 'nemovitost_telekomunikace', 1)) {
			$telecommunication[] = 2; // Internet
		}
		if ($this->_isMultiChecked($originalvalues, 'nemovitost_ostatni_rozvody', 0)) {
			$telecommunication[] = 3; // Satelit
		}
		if ($this->_isMultiChecked($originalvalues, 'nemovitost_ostatni_rozvody', 1)) {
			$telecommunication[] = 4; // Kabelová televize
		}
		if ($this->_isMultiChecked($originalvalues, 'nemovitost_ostatni_rozvody', 2)) {
			$telecommunication[] = 5; // Kabelové rozvody
		}
		if ($this->_isMultiChecked($originalvalues, 'nemovitost_ostatni_rozvody', 3)) {
			$telecommunication[] = 6; // Ostatní
		}
		return $telecommunication;
	}

	/**
	 * Převede položku "Ochrana".
	 *
	 * Převádí z multicheckboxu na codebook, takže záleží na pořadí.
	 *
	 * @param array $originalvalues Pole původních (nepřevedených) hodnot entity.
	 *
	 * @return int Odpovídající hodnota položky protection.
	 */
	protected function _convertOchrana(array $originalvalues)
	{
		if ($this->_isMultiChecked($originalvalues, 'nemovitost_umisteni_v_chranenych_lokalitach', 1)) {
			return 2; // Národní park
		}
		if ($this->_isMultiChecked($originalvalues, 'nemovitost_umisteni_v_chranenych_lokalitach', 2)) {
			return 3; // CHKO
		}
		if ($this->_isMultiChecked($originalvalues, 'nemovitost_umisteni_v_chranenych_lokalitach', 0)) {
			return 1; // Ochranné pásmo
		}
		return 0;
	}

	/**
	 * Normalizuje položku výtah. Převede ji na číselník Srealit.
	 *
	 * @param array  $originalvalues  Pole původních (nepřevedených) hodnot entity.
	 *
	 * @return int Výsledná hodnota položky elevator (ano=1, ne=2, ?=0).
	 */
	protected function _normalizeVytah(array $originalvalues)
	{
		$AnoNeNevim = array(
			-1 => 0,
			1 => 1,
			0 => 2
		);

		$vytah = -1;

		if (isset($originalvalues['nemovitost_vytah']) && $originalvalues['nemovitost_vytah']!=='') {
			$vytah = (int) $originalvalues['nemovitost_vytah'];
		}
		if (isset($originalvalues['nemovitost_ostatni'][4])) {
			// nezaškrtnutý checkbox znamená že nevíme, zda výtah je
			$vytah = $originalvalues['nemovitost_ostatni'][4]==1 ? 1 : -1;
		}

		return $AnoNeNevim[$vytah];
	}

    /**
     * Normalizuje URL na Matterport. Pokud URL nevede na Matterport, vrací false.
     *
     * @param string $url Odkaz na Matterport.
     *
     * @return string Odkaz na Matterport nebo FALSE.
     */
    protected function _getMatterportUrl($url) {
        // https://my.matterport.com/show/?m=uRGXgoiYk9f
        if (strpos($url, 'http') === false) {
            $url = sprintf('http://%s', $url);
        }

        $urls = parse_url($url);

        if ($urls['host']=='my.matterport.com' && $urls['path']=='/show/') {
            $query = array();
            parse_str($urls['query'], $query);

            return sprintf('https://my.matterport.com/show/?m=%s', $query['m']);
        }

        return false;
    }

	/**
	 * Doplní do převedených hodnot položky locality_uir a locality_uir_level.
	 *
	 * @param array  $convertedValues Pole převedených hodnot entity.
     * @param array  $originalvalues  Pole původních (nepřevedených) hodnot entity.
     * @param string $entityName      Název entity.
     *
     * @return array Výsledné pole hodnot.
	 */
	protected function _convert(array $convertedValues, array $originalvalues, $entityName)
	{
		if (($entityName === 'listing') or ($entityName == 'project')) {

			if (!empty($originalvalues['ruian_adresa_kod'])) {
				$convertedValues['locality_uir_level'] = 11;
				$convertedValues['locality_uir'] = (int)$originalvalues['ruian_adresa_kod'];
			} elseif (!empty($originalvalues['nemovitost_uir_ulice_kod'])) {
				$convertedValues['locality_uir_level'] = 7;
				$convertedValues['locality_uir'] = (int)$originalvalues['nemovitost_uir_ulice_kod'];
			} elseif (!empty($originalvalues['nemovitost_uir_mcast_kod'])) {
				$convertedValues['locality_uir_level'] = 17;
				$convertedValues['locality_uir'] = (int)$originalvalues['nemovitost_uir_mcast_kod'];
			} elseif (!empty($originalvalues['nemovitost_uir_cobce_kod'])) {
				$convertedValues['locality_uir_level'] = 5;
				$convertedValues['locality_uir'] = (int)$originalvalues['nemovitost_uir_cobce_kod'];
			} elseif (!empty($originalvalues['nemovitost_uir_obec_kod'])) {
				$convertedValues['locality_uir_level'] = 3;
				$convertedValues['locality_uir'] = (int)$originalvalues['nemovitost_uir_obec_kod'];
			} elseif (!empty($originalvalues['nemovitost_uir_okres_kod'])) {
				$convertedValues['locality_uir_level'] = 1;
				$convertedValues['locality_uir'] = (int)$originalvalues['nemovitost_uir_okres_kod'];
			} elseif (!empty($originalvalues['nemovitost_uir_obec_text'])) {
			    // je-li vyplněno jenom textově, jde o zahraniční nemovitost
                $convertedValues['locality_city'] = $originalvalues['nemovitost_uir_obec_text'] . ', ' . $originalvalues['nemovitost_uir_stat'];
                unset($convertedValues['locality_uir_level'], $convertedValues['locality_uir'], $convertedValues['locality_cp'], $convertedValues['locality_co']);
            }
		}
		if ($entityName === 'listing') {
			$convertedValues['elevator'] = $this->_normalizeVytah($originalvalues);
			$convertedValues['telecommunication'] = $this->_convertTelekomunikace($originalvalues);
			$convertedValues['protection'] = $this->_convertOchrana($originalvalues);
			$parkingLots = $this->_convertParkingLots($originalvalues);
			if (isset($parkingLots)) {
				$convertedValues['parking_lots'] = $parkingLots;
			} else {
				unset($convertedValues['parking_lots']);
			}
			if (!empty($originalvalues['nemovitost_na_klic']) && $originalvalues['nemovitost_stav_objektu']==5 && $originalvalues['nemovitost_typ']==6) {
				$convertedValues['advert_subtype'] = 40;
			}

            if (!empty($originalvalues['nemovitost_virtualni_prohlidka']) && $this->_getMatterportUrl($originalvalues['nemovitost_virtualni_prohlidka'])) {
                $convertedValues['matterport_url'] = $this->_getMatterportUrl($originalvalues['nemovitost_virtualni_prohlidka']);
            }

            if (!empty($originalvalues['share_numerator']) && !empty($originalvalues['share_denominator'])) {
                $convertedValues['advert_function'] = 4; // Podíl
                $convertedValues['share_numerator'] = (int) $originalvalues['share_numerator'];
                $convertedValues['share_denominator'] = (int) $originalvalues['share_denominator'];

                if (!empty($originalvalues['share_common_area_numerator']) && !empty($originalvalues['share_common_area_denominator'])) {
                    $convertedValues['share_common_area_numerator'] = (int) $originalvalues['share_common_area_numerator'];
                    $convertedValues['share_common_area_denominator'] = (int) $originalvalues['share_common_area_denominator'];
                }
            }


		}

		if ($entityName == 'project') {
			// slučujeme popisy projektu - viz https://projekty.dalten.cz/issues/3849
			$convertedValues['description'] = $originalvalues['projekt_zakladni_popis'] . ' ' . $originalvalues['projekt_popis'];
			$convertedValues['annotation'] = mb_substr($convertedValues['description'], 0, 247, 'UTF-8')  . '...';
		}

		return parent::_convert($convertedValues, $originalvalues, $entityName);
	}
}
