<?php
/**
 * Export na Annonce.cz.
 *
 * @category   Dalten
 * @package    Export
 * @subpackage Api
 */
class Dalten_Export_Api_Annonce implements Dalten_Export_Api_LoggableApiInterface, Dalten_Export_Api_VerifiableApiInterface,
    Dalten_Export_Api_ListListingsInterface
{

	protected $_export = null;
	protected $_entities = null;
	protected $_backend = null;
	protected $_addressConverter = null;

	/**
	 * Kostruktor. Nastavuje závislosti třídy.
	 *
	 * @param Dalten_Export_Annonce                     $export           Převaděč do formátu Annonce.
	 * @param Serenity_Config_Config                    $entities         Definice převodních entit pro převaděč.
	 * @param Dalten_Export_Api_Backend_Annonce         $backend          Backend exportu na Annonci.
	 * @param Dalten_AddressConverter_Annonce_Interface $addressConverter Převaděč adres.
	 */
	function __construct(
		Dalten_Export_Annonce $export,
		Serenity_Config_Config $entities,
		Dalten_Export_Api_Backend_Annonce $backend,
		Dalten_AddressConverter_Annonce_Interface $addressConverter
	)
	{
		$this->_export = $export;
		$this->_entities = $entities;
		$this->_backend = $backend;
		$this->_addressConverter = $addressConverter;
	}

	/**
	 * Vždy uspěje, export na Annonci nemá přihlašovací fázi.
	 *
	 * @param string $login            Přihlašovací jméno.
	 * @param string $password         Heslo.
	 * @param string $softwareKey      Softwarový klíč.
	 * @param array  $additionalParams Pole dalších parametrů.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function openConnection($login, $password, $softwareKey = '', array $additionalParams = array())
	{
		$this->_backend->setLogin($login, $password);
		return new Dalten_Export_Api_ServerResponse(true);
	}

	/**
	 * Vyexportuje nabídku.
	 *
	 * @param array $listingData      Data nabídky ve formátu iRest 1.
	 * @param array $userData         Data uživatele (makléře, kterému nabídka patří) ve formátu iRest 1.
	 * @param array $images           Pole fotografií nabídky ve formátu iRest 1.
	 * @param array $additionalParams Údaje které vrací server při exportu a jsou potřeba pro další práci s nabídkou.
	 *                                Data můžou sloužit pro editaci apod...
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru, jestli se export podařil nebo co selhalo.
	 */
	public function addListing(array $listingData, array $userData, array $images = array(), array $additionalParams = array())
	{
		try {
			$listingData = $this->_addressConverter->convertAddress($listingData);
			$data = $this->_export->convertEntityValues('listing', $listingData, $this->_entities->listing);

			$schovatCenu = false;
			if (isset($additionalParams['listing']['hide_price'])) {
				$schovatCenu = $additionalParams['listing']['hide_price'];
			}
			$data['hide-price'] = $schovatCenu;
			$agent = $this->_export->convertEntityValues('agent',$userData,$this->_entities->agent);
			$data = array_merge($data, $agent);
			$response = $this->_backend->addListing($data, $images);

			return $this->_handleServerResponse($response);

		} catch (Dalten_Export_Exception_HumanReadable $e) {
			return new Dalten_Export_Api_ServerResponse(false, 500, $e->getMessage());
		} catch (Exception $e) {
			return new Dalten_Export_Api_ServerResponse(false, 500 );
		}
	}

	/**
	 * {@inheritdoc}
	 */
	public function verifyListing(array $listingData, array $additionalParams)
	{
		try {
			if (isset($additionalParams['result']['statusPageURL']) && $additionalParams['result']['statusPageURL']) {
				$statusPageURL = $additionalParams['result']['statusPageURL'];
				$response = $this->_backend->verifyListing($listingData, $statusPageURL);
				return $this->_handleServerResponse($response);
			}
			return new Dalten_Export_Api_ServerResponse(false);
		} catch (Dalten_Export_Exception_HumanReadable $e) {
			return new Dalten_Export_Api_ServerResponse(false, 500, $e->getMessage());
		} catch (Exception $e) {
			return new Dalten_Export_Api_ServerResponse(false, 500 );
		}
	}

	/**
	 * Přeloží strukturu vrácenou backendem do ServerResponseVerifiable objektu.
	 *
	 * @param StdClass $response Struktura vrácená backendem.
	 *
	 * @return Dalten_Export_Api_ServerResponseVerifiable Odpověď serveru.
	 */
	protected function _handleServerResponse($response)
	{
	    if (!is_object($response)) {
	        $ret = new Dalten_Export_Api_ServerResponseVerifiable(false);
	        $ret->setVerificationStatus(true, false);
	        return $ret;
        }
		if ($response->waiting) {
			$ret = new Dalten_Export_Api_ServerResponseVerifiable(true);
			$ret->setVerificationStatus(false);
		} else {
			if ($response->success) {
				$ret = new Dalten_Export_Api_ServerResponseVerifiable(true);
				$ret->setVerificationStatus(true, true);
			} else {
				$ret = new Dalten_Export_Api_ServerResponseVerifiable(false);
				$ret->setVerificationStatus(true, false);
			}
		}
		if (isset($response->data)) {
			$ret->setData($response->data);
		}
		return $ret;
	}


	/**
	 * Odstraní nabídku ze vzdáleného serveru.
	 *
	 * @param array $listingData      Data nabídky ve formátu iRest 1.
	 * @param array $additionalParams Specifická data pro daný export.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function deleteListing(array $listingData, array $additionalParams = array())
	{
		try {
		    $response = $this->_backend->deleteListing($listingData);
			return $this->_handleServerResponse($response);
		} catch (Dalten_Export_Exception_HumanReadable $e) {
			return new Dalten_Export_Api_ServerResponse(false, 500, $e->getMessage());
		} catch (Exception $e) {
			return new Dalten_Export_Api_ServerResponse(false, 500 );
		}
	}

	/**
	 * Vrátí seznam nabídek, které jsou na vzdáleném serveru.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 *
	 * @throws BadMethodCallException Protože není jako tradičně implementovaná.
	 */
	public function getListingsList()
	{
		throw new BadMethodCallException('Not yet implemented!');
	}

	/** {@inheritDoc} */
    public function listListings()
    {
        return $this->_backend->listListings();
    }


    /**
	 * Nastaví logger pro backend.
	 * Logger bude použit pouze pokud to backend dovoluje.
	 *
	 * @param Dalten_Export_Api_Backend_Logger_LoggerInterface $logger Instance loggeru.
	 *
	 * @return bool Podařilo se přiřadit backendu logger?
	 */
	public function setLogger(Dalten_Export_Api_Backend_Logger_LoggerInterface $logger)
	{
		$this->_backend->setLogger($logger);
		return true;
	}

	/**
	 * Odstraní nastavený logger pro backend.
	 *
	 * @return Dalten_Export_Api_LoggableApiInterface Fluent interface.
	 */
	public function removeLogger()
	{
		$this->_backend->removeLogger();
		return $this;
	}


}
