<?php

/**
 * Exportní API serveru RealityMorava.
 *
 * @category   Dalten
 * @package    Export
 * @subpackage Api
 */
class Dalten_Export_Api_RealityMorava implements Dalten_Export_Api_ApiInterface, Dalten_Export_Api_LoggableApiInterface, Dalten_Export_Api_ListListingsInterface
{
	private $_backend = null;
	private $_config = null;
	private $_export = null;
	private $_adressConverter = null;
	private $_login = '';
	private $_password = '';

	/**
	 * Kontruktor.
	 *
	 * @param Dalten_Export_RealityMorava                     $export  Exportní třída.
	 * @param Serenity_Config_Config                          $config  Entity exportu.
	 * @param Dalten_Export_Api_Backend_RealityMorava         $backend Backend exportu.
	 * @param Dalten_AddressConverter_RealityMorava_Interface $adresar Převodník adres.
	 */
	public function __construct(
		Dalten_Export_RealityMorava $export,
		Serenity_Config_Config $config,
		Dalten_Export_Api_Backend_RealityMorava $backend,
		Dalten_AddressConverter_RealityMorava_Interface $adresar
	)
	{
		$this->_export = $export;
		$this->_config = $config;
		$this->_backend = $backend;
		$this->_adressConverter = $adresar;
	}

	/**
	 * Naváže spojení se vzdáleným serverem.
	 *
	 * @param string $login            Přihlašovací jméno.
	 * @param string $password         Heslo.
	 * @param string $softwareKey      Softwarový klíč.
	 * @param array  $additionalParams Pole dalších parametrů.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function openConnection($login, $password, $softwareKey = '', array $additionalParams = array())
	{
		try {
			$this->_login = $login;
			$this->_password = $password;
			return $this->_backend->checkLogin($login, $password);
		} catch (Dalten_Export_Exception_HumanReadable $e) {
			return new Dalten_Export_Api_ServerResponse(false, 500, $e->getMessage());
		} catch (Exception $e) {
			return new Dalten_Export_Api_ServerResponse(false, 500);
		}
	}

	/**
	 * Vyexportuje nabídku.
	 *
	 * @param array $listingData      Data nabídky ve formátu iRest 1.
	 * @param array $userData         Data uživatele (makléře, kterému nabídka patří) ve formátu iRest 1.
	 * @param array $images           Pole fotografií nabídky ve formátu iRest 1.
	 * @param array $additionalParams Údaje které vrací server při exportu a jsou potřeba pro další práci
	 *                                s nabídkou.
	 *                                Data můžou sloužit pro editaci apod...
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru, jestli se export podařil nebo co selhalo.
	 */
	public function addListing(
		array $listingData,
		array $userData,
		array $images = array(),
		array $additionalParams = array()
	)
	{
		try {
			$listingData['akce_exportu'] = 'update';
            if (!empty($additionalParams['listing']['hide_price'])) {
                // Cena dohodou
                $listingData['nemovitost_cena_prodej'] = 0;
                $listingData['nemovitost_cena_pronajem'] = 0;
                $listingData['nemovitost_cena_poznamka'] = 'Informace v RK';
            }
			$listingData = $this->_adressConverter->convertAddress($listingData);
			$listingData = $this->_export->convertEntityValues('listing', $listingData, $this->_config->listing);
			$userData = $this->_export->convertEntityValues('broker', $userData, $this->_config->broker);
			return $this->_backend->addAdvert(
				$this->_login,
				$this->_password,
				$listingData,
				$userData,
				$images,
				$additionalParams
			);
		} catch (Dalten_Export_Exception_HumanReadable $e) {
			return new Dalten_Export_Api_ServerResponse(false, 500, $e->getMessage());
		} catch (Exception $e) {
			return new Dalten_Export_Api_ServerResponse(false, 500);
		}
	}

	/**
	 * Odstraní nabídku ze vzdáleného serveru.
	 *
	 * @param array $listingData      Data nabídky ve formátu iRest 1.
	 * @param array $additionalParams Specifická data pro daný export.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function deleteListing(array $listingData, array $additionalParams = array())
	{
		try {
			$listingData['akce_exportu'] = 'delete';
			$listingData = $this->_export->convertEntityValues('listing', $listingData, $this->_config->listing);
			return $this->_backend->addAdvert(
				$this->_login,
				$this->_password,
				$listingData,
				array(),
				array(),
				$additionalParams
			);
		} catch (Dalten_Export_Exception_HumanReadable $e) {
			return new Dalten_Export_Api_ServerResponse(false, 500, $e->getMessage());
		} catch (Exception $e) {
			return new Dalten_Export_Api_ServerResponse(false, 500);
		}
	}

	/**
	 * Vrátí seznam nabídek, které jsou na vzdáleném serveru.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function getListingsList()
	{
		throw new BadMethodCallException('Not yet implemented!');
	}

	/**
	 * Nastaví logger pro backend.
	 * Logger bude použit pouze pokud to backend dovoluje.
	 *
	 * @param Dalten_Export_Api_Backend_Logger_LoggerInterface $logger Instance loggeru.
	 *
	 * @return bool Podařilo se přiřadit backendu logger?
	 */
	public function setLogger(Dalten_Export_Api_Backend_Logger_LoggerInterface $logger)
	{
		if ($this->_backend instanceof Dalten_Export_Api_Backend_LoggableBackendInterface) {
			return $this->_backend->setLogger($logger);
		}
		return false;
	}

	/**
	 * Odstraní nastavený logger pro backend.
	 *
	 * @return Dalten_Export_Api_LoggableApiInterface Fluent interface.
	 */
	public function removeLogger()
	{
		if ($this->_backend instanceof Dalten_Export_Api_Backend_LoggableBackendInterface) {
			$this->_backend->removeLogger();
		}
		return $this;
	}

    /**
     * {@inheritDoc}
     */
    public function listListings()
    {
        return $this->_backend->listListings($this->_login, $this->_password);
    }


}
