#!/usr/bin/env php
<?php

if (PHP_SAPI !== 'cli') {
	echo 'This should only be run from console; exiting';
	exit(1);
}

if (count($argv) !== 2) {
	echo 'Usage: correct-tag-ordering.php <path/to/original.xml>' . PHP_EOL;
	exit(1);
}

if (!is_readable($argv[1])) {
	echo 'The file \'' . $argv[1] . '\' is not readable' . PHP_EOL;
	exit(1);
}

libxml_use_internal_errors(true);

$dom = DOMDocument::load($argv[1]);
if ($dom === false) {
	file_put_contents('php://stderr', 'The file is not an xml file, or contains xml syntax errors; errors:' . PHP_EOL);
	libxml_display_errors();
	exit(1);
}

regenerate_xml($dom->xmlVersion, $dom->xmlEncoding, simplexml_import_dom($dom));

// function listing

function libxml_format_error($error) {
	$dict = array(
		LIBXML_ERR_WARNING => 'Warning',
		LIBXML_ERR_ERROR => 'Error',
		LIBXML_ERR_FATAL => 'Fatal error',
	);

	return sprintf(
		"%s: %s; on line %d in %s\n",
		$dict[$error->level],
		trim($error->message),
		$error->line,
		$error->file
	);
}

function libxml_display_errors() {
	$errors = libxml_get_errors();
	foreach ($errors as $error) {
		file_put_contents('php://stderr', libxml_format_error($error));
	}
	libxml_clear_errors();
}

function regenerate_xml($version, $encoding, SimpleXMLElement $sxml) {
	$nsPrefix = null;
	$nsURI = 'http://schemas.ingatlanbazar.hu/Import/';
	$xsdLoc = 'http://schemas.ingatlanbazar.hu/Import/ http://schemas.ingatlanbazar.hu/Import/xsd/import.xsd';
	
	$xw = new XMLWriter();
	$xw->openURI('php://stdout');
	$xw->setIndent(true);
	$xw->setIndentString("\t");
	$xw->startDocument($version, $encoding);
	$xw->startElementNs($nsPrefix, 'import', $nsURI);
	$xw->writeAttributeNs('xsi', 'schemaLocation', 'http://www.w3.org/2001/XMLSchema-instance', $xsdLoc);
	
	$xw->startElementNs($nsPrefix, 'officeList', null);
	foreach ($sxml->officeList->office as $office) {
		$xw->startElementNs($nsPrefix, 'office', null);
		$xw->writeAttribute('ingatlanbazarId', (int) $office['ingatlanbazarId']);
		$xw->startElementNs($nsPrefix, 'information', null);
		$xw->startElementNs($nsPrefix, 'notificationmailList', null);
		foreach ($office->information->notificationmailList->notificationmailText as $nmText) {
			$xw->writeElementNs($nsPrefix, 'notificationmailText', null, (string) $nmText);
		}
		$xw->endElement();
		$xw->endElement();
		$xw->startElementNs($nsPrefix, 'agentList', null);
		if (isset($office->agentList->agent)) {
			foreach ($office->agentList->agent as $agent) {
				$xw->startElementNs($nsPrefix, 'agent', null);
				$xw->writeAttribute('foreignId', (string) $agent['foreignId']);
				$xw->writeElementNs($nsPrefix, 'nameText', null, (string) $agent->nameText);
				if (isset($agent->addressText)) {
					$xw->writeElementNs($nsPrefix, 'addressText', null, (string) $agent->addressText);
				}
				if (isset($agent->phonenumberText)) {
					$xw->writeElementNs($nsPrefix, 'phonenumberText', null, (string) $agent->phonenumberText);
				}
				if (isset($agent->mobilenumberText)) {
					$xw->writeElementNs($nsPrefix, 'mobilenumberText', null, (string) $agent->mobilenumberText);
				}
				$xw->writeElementNs($nsPrefix, 'emailText', null, (string) $agent->emailText);
				$xw->endElement();
			}
		}
		$xw->endElement();
		$xw->flush();
		$xw->startElementNs($nsPrefix, 'adList', null);
		if (isset($office->adList->ad)) {
			foreach ($office->adList->ad as $ad) {
				$xw->startElementNs($nsPrefix, 'ad', null);
				$xw->writeAttribute('foreignId', (string) $ad['foreignId']);
				$xw->writeAttribute('agentId', (string) $ad['agentId']);
				if (isset($ad['projectId'])) {
					$xw->writeAttribute('projectId', (int) $ad['projectId']);
				}
				$xw->startElementNs($nsPrefix, 'type', null);
				$xw->writeAttribute('option', (int) $ad->type['option']);
				$xw->endElement();
				$xw->startElementNs($nsPrefix, 'agreement', null);
				$xw->writeAttribute('option', (int) $ad->agreement['option']);
				$xw->endElement();
				if (isset($ad->coordinates)) {
					$xw->startElementNs($nsPrefix, 'coordinates', null);
					$xw->writeAttribute('latitude', (string) $ad->coordinates['latitude']);
					$xw->writeAttribute('longitude', (string) $ad->coordinates['longitude']);
					$xw->endElement();
				} else {
					if (isset($ad->regionText)) {
						$xw->writeElementNs($nsPrefix, 'regionText', null, (string) $ad->regionText);
					} else if (isset($ad->region)) {
						$msg = sprintf('Region node is deprecated; use regionText instead (ad \'%s\')', $ad['foreignId']);
						trigger_error($msg, E_USER_WARNING);
						$xw->startElementNs($nsPrefix, 'region', null);
						$xw->writeAttribute('option', (int) $ad->region['option']);
						$xw->endElement();
					} else {
						$msg = sprintf('Region definition is missing, but required (ad \'%s\')', $ad['foreignId']);
						trigger_error($msg, E_USER_WARNING);
					}
					if (isset($ad->cityText)) {
						$xw->writeElementNs($nsPrefix, 'cityText', null, (string) $ad->cityText);
					} else if (isset($ad->cityt)) {
						$msg = sprintf('City node is deprecated; use cityText instead (ad \'%s\')', $ad['foreignId']);
						trigger_error($msg, E_USER_WARNING);
						$xw->startElementNs($nsPrefix, 'city', null);
						$xw->writeAttribute('option', (int) $ad->city['option']);
						$xw->endElement();
					} else {
						$msg = sprintf('City definition is missing, but required (ad \'%s\')', $ad['foreignId']);
						trigger_error($msg, E_USER_WARNING);
					}
					if (isset($ad->suburbText)) {
						$xw->writeElementNs($nsPrefix, 'suburbText', null, (string) $ad->suburbText);
					} else if (isset($ad->suburb)) {
						$msg = sprintf('Suburb node is deprecated; use suburbText instead (ad \'%s\')', $ad['foreignId']);
						trigger_error($msg, E_USER_WARNING);
						$xw->startElementNs($nsPrefix, 'suburb', null);
						$xw->writeAttribute('option', (int) $ad->suburb['option']);
						$xw->endElement();
					}
					if (isset($ad->address)) {
						$xw->startElementNs($nsPrefix, 'address', null);
						$xw->writeAttribute('visible', (string) $ad->address['visible']);
						$xw->writeElementNs($nsPrefix, 'street', null, (string) $ad->address->street);
						$xw->writeElementNs($nsPrefix, 'number', null, (int) $ad->address->number);
						if (isset($ad->address->building)) {
							$xw->writeElementNs($nsPrefix, 'building', null, (string) $ad->address->building);
						}
						if (isset($ad->address->floor)) {
							$xw->writeElementNs($nsPrefix, 'floor', null, (int) $ad->address->floor);
						}
						if (isset($ad->address->roomNumber)) {
							$xw->writeElementNs($nsPrefix, 'roomNumber', null, (int) $ad->address->roomNumber);
						}
						$xw->endElement();
					}
				}
				$xw->startElementNs($nsPrefix, 'price', null);
				$xw->writeAttribute('intval', (int) $ad->price['intval']);
				$xw->endElement();
				if (isset($ad->currency)) {
					$xw->startElementNs($nsPrefix, 'currency', null);
					$xw->writeAttribute('option', (int) $ad->currency['option']);
					$xw->endElement();
				}
				if (isset($ad->payingperiod)) {
					$xw->startElementNs($nsPrefix, 'payingperiod', null);
					$xw->writeAttribute('option', (int) $ad->payingperiod['option']);
					$xw->endElement();
				}
				$xw->startElementNs($nsPrefix, 'condition', null);
				$xw->writeAttribute('option', (int) $ad->condition['option']);
				$xw->endElement();
				if (isset($ad->heating)) {
					$xw->startElementNs($nsPrefix, 'heating', null);
					$xw->writeAttribute('option', (int) $ad->heating['option']);
					$xw->endElement();
				}
				if (isset($ad->floors)) {
					$xw->startElementNs($nsPrefix, 'floors', null);
					$xw->writeAttribute('option', (int) $ad->floors['option']);
					$xw->endElement();
				}
				if (isset($ad->rooms)) {
					$xw->startElementNs($nsPrefix, 'rooms', null);
					$xw->writeAttribute('intval', (int) $ad->rooms['intval']);
					$xw->endElement();
				}
				if (isset($ad->halfrooms)) {
					$xw->startElementNs($nsPrefix, 'halfrooms', null);
					$xw->writeAttribute('intval', (int) $ad->halfrooms['intval']);
					$xw->endElement();
				}
				$xw->startElementNs($nsPrefix, 'floorspace', null);
				$xw->writeAttribute('intval', (int) $ad->floorspace['intval']);
				$xw->endElement();
				if (isset($ad->propertyspace)) {
					$xw->startElementNs($nsPrefix, 'propertyspace', null);
					$xw->writeAttribute('intval', (int) $ad->propertyspace['intval']);
					$xw->endElement();
				}
				$xw->writeElementNs($nsPrefix, 'descriptionText', null, (string) $ad->descriptionText);
				if (isset($ad->tagsText)) {
					$xw->writeElementNs($nsPrefix, 'tagsText', null, (string) $ad->tagsText);
				}
				if (isset($ad->imageList)) {
					$xw->startElementNs($nsPrefix, 'imageList', null);
					foreach ($ad->imageList->image as $image) {
						$xw->startElementNs($nsPrefix, 'image', null);
						if (isset($image['foreignId'])) {
							$xw->writeAttribute('foreignId', (string) $image['foreignId']);
						}
						$xw->writeAttribute('href', (string) $image['href']);
						$xw->endElement();
					}
					$xw->endElement();
				}
				if (isset($ad->highlightingList)) {
					$xw->startElementNs($nsPrefix, 'highlightingList', null);
					foreach ($ad->highlightingList->highlighting as $highlighting) {
						$xw->startElementNs($nsPrefix, 'highlighting', null);
						$xw->writeAttribute('id', (int) $highlighting['id']);
						$xw->writeAttribute('start', (string) $highlighting['start']);
						$xw->writeAttribute('duration', (int) $highlighting['duration']);
						$xw->endElement();
					}
					$xw->endElement();
				}
				$xw->endElement();
				$xw->flush();
			}
		}
		$xw->endElement();
		$xw->endElement();
	}
	$xw->endElement();
	$xw->endElement();
	$xw->flush();
}
