<?php
/**
 * Export na Ingatlannet.hu
 *
 * @category   Dalten
 * @package    Export
 * @subpackage Api
 */
class Dalten_Export_Api_Ingatlannethu implements Dalten_Export_Api_CloseableApiInterface
{
	/**
	 * @var null|Dalten_Export_Api_Backend_Ingatlannethu
	 */
	private $_backend = null;
	/**
	 * @var null|Dalten_Export_Ingatlannethu
	 */
	private $_export = null;

	/**
	 * @var null|Dalten_Xml_SimpleXml
	 */
	private $_xml = null;

	/**
	 * @var string|null Název cílového souboru
	 */
	private $_fileName = null;

	/**
	 * Konstruktor
	 *
	 * @param Dalten_Export_Api_Backend_Ingatlannethu $backend  Backend
	 * @param Dalten_Export_Ingatlannethu             $export   Převaděč exportu
	 * @param string                                  $fileName Cesta k cílovému souboru
	 */
	function __construct(Dalten_Export_Api_Backend_Ingatlannethu $backend, Dalten_Export_Ingatlannethu $export, $fileName)
	{
		$this->_backend = $backend;
		$this->_export = $export;
		$this->_fileName = $fileName;
	}

	/**
	 * Naváže spojení se vzdáleným serverem.
	 *
	 * @param string $login            Přihlašovací jméno.
	 * @param string $password         Heslo.
	 * @param string $softwareKey      Softwarový klíč.
	 * @param array  $additionalParams Pole dalších parametrů.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function openConnection($login, $password, $softwareKey = '', array $additionalParams = array())
	{
		$this->_xml = $this->_backend->prepareXml();
		return new Dalten_Export_Api_ServerResponse(true);
	}

	/**
	 * Vyexportuje nabídku.
	 *
	 * @param array $listingData      Data nabídky ve formátu iRest 1.
	 * @param array $userData         Data uživatele (makléře, kterému nabídka patří) ve formátu iRest 1.
	 * @param array $images           Pole fotografií nabídky ve formátu iRest 1.
	 * @param array $additionalParams Údaje které vrací server při exportu a jsou potřeba pro další práci s nabídkou.
	 *                                Data můžou sloužit pro editaci apod...
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru, jestli se export podařil nebo co selhalo.
	 */
	public function addListing(array $listingData, array $userData, array $images = array(), array $additionalParams = array())
	{
		try{
			$this->_backend->appendProperty($this->_xml, $listingData, $userData, $images, $additionalParams);
			return new Dalten_Export_Api_ServerResponse(true);
		} catch (Dalten_Export_Exception_HumanReadable $e){
			return new Dalten_Export_Api_ServerResponse(false, $e->getCode(), $e->getMessage());
		} catch (Exception $e){
			return new Dalten_Export_Api_ServerResponse(false,0,$e->getMessage(). "\n" .$e->getTraceAsString());
		}

	}

	/**
	 * Odstraní nabídku ze vzdáleného serveru.
	 *
	 * @param array $listingData      Data nabídky ve formátu iRest 1.
	 * @param array $additionalParams Specifická data pro daný export.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function deleteListing(array $listingData, array $additionalParams = array())
	{
		return new Dalten_Export_Api_ServerResponse(true);
	}

	/**
	 * Vrátí seznam nabídek, které jsou na vzdáleném serveru.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function getListingsList()
	{
		throw new BadMethodCallException('Not yet implemented!');
	}

	/**
	 * Zavírá spojení.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověd serveru.
	 */
	public function closeConnection()
	{
		try {
			if ($this->_backend->saveToFile($this->_xml, $this->_fileName)) {
				return new Dalten_Export_Api_ServerResponse(true);
			} else {
				return new Dalten_Export_Api_ServerResponse(false);
			}
		} catch (Dalten_Export_Exception_HumanReadable $e) {
			return new Dalten_Export_Api_ServerResponse(false, 0, $e->getMessage());
		} catch (Exception $e) {
			return new Dalten_Export_Api_ServerResponse(false);
		}
	}
}
