<?php

/**
 * Exportní API pro server living.sk.
 *
 * @category   Dalten
 * @package    Export
 * @subpackage Api
 */
class Dalten_Export_Api_Living implements Dalten_Export_Api_LoggableApiInterface
{

	/**
	 * Backend ktery se stara o exportovani dat.
	 *
	 * @var \Dalten_Export_Api_Backend_Living|null
	 */
	private $_backend = null;

	/**
	 * Data expotu.
	 *
	 * @var \Dalten_Export_Living|null
	 */
	private $_export = null;

	/**
	 * Konfigurace exportu.
	 *
	 * @var null|\Serenity_Config_Config
	 */
	private $_config = null;

	/**
	 * Zakryptovane heslo uzivatele.
	 *
	 * @var string
	 */
	private $_password = '';

	/**
	 * Sul pro vypocet hesla.
	 *
	 * @var string
	 */
	private $_passwordSalt = '';

	/**
	 * Id firmy pod kterou chceme exportovat.
	 *
	 * @var int
	 */
	private $_advertiserId = 0;

	/**
	 * Nastavi vlastnosti objektu.
	 *
	 * @param Dalten_Export_Living             $export  Data exportu.
	 * @param Serenity_Config_Config           $config  Nastaveni exportu.
	 * @param Dalten_Export_Api_Backend_Living $backend Rozhrani pro komunikaci s exportnim serverem.
	 */
	public function __construct(
		Dalten_Export_Living $export,
		Serenity_Config_Config $config,
		Dalten_Export_Api_Backend_Living $backend)
	{
		$this->_export = $export;
		$this->_config = $config;
		$this->_backend = $backend;
	}


	/**
	 * Naváže spojení se vzdáleným serverem.
	 *
	 * @param string $login            Přihlašovací jméno.
	 * @param string $password         Heslo.
	 * @param string $softwareKey      Softwarový klíč.
	 * @param array  $additionalParams Pole dalších parametrů (musí obsahovat klíč "advertiser_id").
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function openConnection($login, $password, $softwareKey = '', array $additionalParams = array())
	{
		if ($softwareKey === '') {
			new Dalten_Export_Api_ServerResponse(false, 500, 'Chybi softwarovy klic.');
		}

		if (!isset($additionalParams['advertiser_id'])) {
			return new Dalten_Export_Api_ServerResponse(
				false, 500, 'Nebylo předáno ID inzerenta.'
			);
		}

		$this->_advertiserId = $additionalParams['advertiser_id'];
		$this->_passwordSalt = $softwareKey;
		$this->_password = $this->_generatePassword($password, $this->_passwordSalt);
		return $this->_backend->login($login, $this->_password);
	}

	/**
	 * Vyexportuje nabídku.
	 *
	 * @param array $listingData      Data nabídky ve formátu iRest 1.
	 * @param array $userData         Data uživatele (makléře, kterému nabídka patří) ve formátu iRest 1.
	 * @param array $images           Pole fotografií nabídky ve formátu iRest 1.
	 * @param array $additionalParams Parametry potřebné pro editaci.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru, jestli se export podařil nebo co selhalo.
	 */
	public function addListing(array $listingData, array $userData, array $images = array(), array $additionalParams = array())
	{
		$userData['password'] = $this->_generatePassword($userData['id']);
		$data = $this->_export->convertEntityValues('listing', $listingData, $this->_config->listing);

		// Pro zjednodušení přidáme do dat ID inzerenta.
		$data['advertiser_id'] = $this->_advertiserId;

		return $this->_backend->addListing($data, $userData, $images, $additionalParams);
	}

	/**
	 * Odstraní nabídku ze vzdáleného serveru.
	 *
	 * @param array $listingData      Data nabídky ve formátu iRest 1.
	 * @param array $additionalParams Musi obsahovat klic 'ad-id'.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function deleteListing(array $listingData, array $additionalParams = array())
	{
		return $this->_backend->deleteListing($additionalParams['result']);
	}

	/**
	 * Vždy uspěje.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function closeConnection(){
		return new Dalten_Export_Api_ServerResponse(true);
	}

	/**
	 * Metoda zakryptuje heslo dle manualu na living.
	 *
	 * @param string $password Ciste heslo.
	 *
	 * @return string MD5 hash.
	 */
	private function _generatePassword($password)
	{
		return md5(md5($password) . md5($this->_passwordSalt));
	}

	/**
	 * Nastaví logger pro backend.
	 *
	 * Logger bude použit pouze pokud to backend dovoluje.
	 *
	 * @param Dalten_Export_Api_Backend_Logger_LoggerInterface $logger Instance loggeru.
	 *
	 * @return bool Podařilo se přiřadit backendu logger?
	 */
	public function setLogger(Dalten_Export_Api_Backend_Logger_LoggerInterface $logger)
	{
		if ($this->_backend instanceof Dalten_Export_Api_Backend_LoggableBackendInterface) {
			$this->_backend->setLogger($logger);
		}
		return false;
	}

	/**
	 * Odstraní nastavený logger pro backend.
	 *
	 * @return Dalten_Export_Api_LoggableApiInterface Fluent interface.
	 */
	public function removeLogger()
	{
		$this->_backend->removeLogger();
	}

	/**
	 * Implementace interface.
	 *
	 * @throws BadMethodCallException
	 */
	public function getListingsList()
	{
		throw new BadMethodCallException("Method 'getListingsList' IS NOT implemented yet.");
	}

	/**
	 * Vrátí pole serverů na které může přihlášený uživatel exportovat či importovat.
	 *
	 * @return Dalten_Export_Api_ServerResponse
	 */
	public function getServices()
	{
		return $this->_backend->getPortals();
	}
}
