<?php
/**
 * Vylepšovač adres.
 *
 * Na Annonci se adresy posílají v textové podobě. Tato třída upravuje adresu jdoucí z Irestu,
 * aby došlo k lepšímu zařazení na Annonci.
 *
 * Zároveň přidává PSČ pro zatřídění.
 *
 * @category Dalten
 * @package  AdressConverter
 */
class Dalten_AddressConverter_Annonce_Irest1 implements Dalten_AddressConverter_Annonce_Interface
{
	/**
	 * UIR okres_kod pro Prahu.
	 */
	const OKRES_KOD_PRAHA = 3100;


	protected $_db = null;
	protected $_dbPrefix = null;

	/**
	 * Konstruktor převodníku adres na Annonci. Předpokládá adresní databázi ve stylu Irestu 1.
	 *
	 * @param Irest_Database $db     Připojení do databáze.
	 * @param string         $prefix Název adresní databáze (např. adresy nebo adresy_sk).
	 */
	public function __construct(Irest_Database $db, $prefix = 'adresy')
	{
		$this->_db = $db;
		$this->_dbPrefix = $prefix;
	}


	/**
	 * Převede adresu u nemovitosti do jiného adresního systému.
	 *
	 * @param array $input Informace o nemovitosti.
	 *
	 * @return array Informace o nemovitosti s upravenou adresou.
	 */
	public function convertAddress(array $input)
	{
		$inCzechRepublic = ($input['nemovitost_uir_stat'] == 'CZ');
		$input['country'] = $input['nemovitost_uir_stat'];
		$input['city'] = $input['nemovitost_uir_obec_text'];
		$input['district'] = $input['nemovitost_uir_okres_text'];
		$input['street'] = '';

		if ($input['nemovitost_uir_ulice_text']) {
			$input['street'] = $input['nemovitost_uir_ulice_text'];
		}

		if ($inCzechRepublic && ($input['nemovitost_uir_okres_kod']==self::OKRES_KOD_PRAHA)) {
			$input['district'] = 'Praha hl.m.'; // v naší databázi je jinak, to se jim nelíbí
			$input['city'] = $input['nemovitost_uir_pobvod_text']; // jako město se posílá pražský obvod
		}

		$zip = $this->_findZIP($input);
		if ($zip) {
			$input['zip'] = $zip;
		}

		return $input;
	}

	/**
	 * Pokusí se najít odpovídající PSČ k adrese.
	 *
	 * Postupuje od nejvíce vyplněného k tomu nejméně vyplněnému.
	 *
	 * @param array $input Informace o nemovitosti (musí obsahovat adresu).
	 *
	 * @return mixed|null PSČ nebo null pokus se ho nepodaří najít.
	 */
	protected function _findZIP(array $input)
	{
		$zip = null;
		if ($input['nemovitost_uir_cobce_kod'] && $input['nemovitost_uir_mcast_kod'] && $input['nemovitost_uir_ulice_kod']) {
			$zip = $this->_db->fetchField(
				sprintf(
					'SELECT psc FROM %s.vazba WHERE cobce_kod=%d AND mcast_kod=%d AND ulice_kod=%d',
					$this->_dbPrefix,
					$input['nemovitost_uir_cobce_kod'],
					$input['nemovitost_uir_mcast_kod'],
					$input['nemovitost_uir_ulice_kod']
				)
			);
		}

		if ((!$zip) && $input['nemovitost_uir_cobce_kod'] && $input['nemovitost_uir_ulice_kod']) {
			$zip = $this->_db->fetchField(
				sprintf(
					'SELECT psc FROM %s.vazba WHERE cobce_kod=%d AND ulice_kod=%d',
						$this->_dbPrefix,
						$input['nemovitost_uir_cobce_kod'],
						$input['nemovitost_uir_ulice_kod']
				)
			);
		}

		if ((!$zip) && $input['nemovitost_uir_cobce_kod']) {
			$zip = $this->_db->fetchField(
				sprintf(
					'SELECT psc FROM %s.vazba WHERE cobce_kod=%d',
					$this->_dbPrefix,
					$input['nemovitost_uir_cobce_kod']
				)
			);
		}

		return $zip;
	}
}