<?php
/**
 * Dumpovací export, pro účely testování runneru.
 *
 * Plive vyexportované nemovitosti do adresářů na disku do Yamlu.
 *
 * @category   Dalten
 * @package    Export
 * @subpackage Api
 */
class Dalten_Export_Api_Dumper implements
	Dalten_Export_Api_ApiInterface,
	Dalten_Export_Api_ProjectApiInterface,
	Dalten_Export_Api_CloseableApiInterface,
	Dalten_Export_Api_LoggableApiInterface,
	Dalten_Export_Api_VerifiableApiInterface
{
	private $_dumpDir=null;
	private $_dumper=null;
	private $_logger=null;

	/**
	 * Konstruktor.
	 *
	 * @param string $dumpDir Adresář do kterého budeme dumpovat (musí mít podadresáře projekt a estate).
	 * @param object $dumper  Yaml dumper ze Symfony (nebo cokoliv podobně se chovajícího).
	 */
	function __construct($dumpDir,$dumper)
	{
		$this->_dumper=$dumper;
		$this->_dumpDir=$dumpDir;
	}

	/**
	 * Pomocná funkce. Získá jméno souboru.
	 *
	 * @param string $id  ID entity.
	 * @param string $typ Typ entity (project/estate).
	 *
	 * @return string Jméno souboru.
	 */
	private function _getFn($id, $typ)
	{
		return $this->_dumpDir . "/" . $typ ."/" . $id . ".txt";
	}

	/**
	 * Naváže spojení se vzdáleným serverem.
	 *
	 * @param string $login            Přihlašovací jméno.
	 * @param string $password         Heslo.
	 * @param string $softwareKey      Softwarový klíč.
	 * @param array  $additionalParams Pole dalších parametrů.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function openConnection($login, $password, $softwareKey = '', array $additionalParams = array())
	{
		if ($this->_logger) {
			$this->_logger->logRemoteCall('openConnection', array('login'=>$login, 'pass'=>$password));
		}
		if (
			file_put_contents(
				$this->_dumpDir . "/log.txt",
				sprintf("[login] %s:%s(%s) at %s" . PHP_EOL, $login, $password, $softwareKey, date('Y-m-d H:i:s')),
				FILE_APPEND
			)
		) {
			return new Dalten_Export_Api_ServerResponse(true);
		}
		return new Dalten_Export_Api_ServerResponse(false, 0, 'Selhal login!');
	}

	/**
	 * Vrátí seznam nabídek, které jsou na vzdáleném serveru.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function getListingsList()
	{
		throw new BadMethodCallException('Not yet implemented.');
	}

	/**
	 * Odstraní nabídku ze vzdáleného serveru.
	 *
	 * @param array $listingData      Data nabídky ve formátu iRest 1.
	 * @param array $additionalParams Specifická data pro daný export.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function deleteListing(array $listingData, array $additionalParams = array())
	{
		if ($this->_logger) {
			$this->_logger->logRemoteCall('deleteListing', $listingData['id']);
		}
		$id = $listingData['id'];
		if (file_exists($this->_getFn($id, 'estate'))) {
			if (file_put_contents($this->_getFn($id, 'estate'), "deleted")!==false) {
				return new Dalten_Export_Api_ServerResponse(true);
			}
			return new Dalten_Export_Api_ServerResponse(false, 0, 'Chyba disku - mazání!');
		}
		return new Dalten_Export_Api_ServerResponse(false, 0, 'Nabídka neexistuje!');
	}

	/**
	 * Vyexportuje nabídku.
	 *
	 * @param array $listingData      Data nabídky ve formátu iRest 1.
	 * @param array $userData         Data uživatele (makléře, kterému nabídka patří) ve formátu iRest 1.
	 * @param array $images           Pole fotografií nabídky ve formátu iRest 1.
	 * @param array $additionalParams Údaje které vrací server při exportu a jsou potřeba pro další
	 *                                práci s nabídkou.
	 *                                Data můžou sloužit pro editaci apod...
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru, jestli se export podařil nebo co selhalo.
	 */
	public function addListing(
		array $listingData, array $userData, array $images = array(), array $additionalParams = array()
	)
	{
		if ($this->_logger) {
			$this->_logger->logRemoteCall('addListing', $listingData['id']);
		}
		$id = $listingData['id'];
		$text = $this->_dumper->dump(array('listing'=>$listingData,'images'=>$images,'broker'=>$userData), 6);
		if (file_put_contents($this->_getFn($id, "estate"), $text)!==false) {
			$r=new Dalten_Export_Api_ServerResponse(true);

			$exported = (isset($additionalParams['result']['exported']) ? $additionalParams['result']['exported'] : array());
			$exported[] = date('Y-m-d H:i:s');

			$r->setData(array('exported'=>$exported));
			return $r;
		}
		return new Dalten_Export_Api_ServerResponse(false, 0, 'Chyba zápisu na disk!');
	}

	/**
	 * Zavírá spojení.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď zda se podařilo zavřít spojení.
	 */
	public function closeConnection()
	{
		if ($this->_logger) {
			$this->_logger->logRemoteCall('closeConnection', null);
		}
		file_put_contents(
			$this->_dumpDir . "/log.txt",
			sprintf("[logout] at %s" . PHP_EOL, date('Y-m-d H:i:s')),
			FILE_APPEND
		);
		return new Dalten_Export_Api_ServerResponse(true);
	}

	/**
	 * Vyexportuje nabídku.
	 *
	 * @param array $projectData      Data projektu ve formátu iRest 1.
	 * @param array $userData         Data uživatele (makléře, kterému projekt patří) ve formátu iRest 1.
	 * @param array $listings         Pole s ID nabídek.
	 * @param array $images           Pole fotografií projektu ve formátu iRest 1.
	 * @param array $additionalParams Údaje které vrací server při exportu a jsou potřeba pro další
	 *                                práci s nabídkou.
	 *                                Data můžou sloužit pro editaci apod...
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru, jestli se export podařil nebo co selhalo.
	 */
	function addProject(
		array $projectData,
		array $userData,
		array $listings=array(),
		array $images=array(),
		array $additionalParams=array()
	)
	{
		$id = $projectData['id'];
		if ($this->_logger) {
			$this->_logger->logRemoteCall('addProject', $id);
		}
		$text = $this->_dumper->dump(
			array(
				'project'=>$projectData,
				'listings'=>$listings,
				'images'=>$images,
				'broker'=>$userData,
				'estate_ids'=>$listings
			), 6
		);
		if (file_put_contents($this->_getFn($id, "projekt"), $text)!==false) {
			return new Dalten_Export_Api_ServerResponse(true);
		}
		return new Dalten_Export_Api_ServerResponse(false, 0, 'Chyba zápisu na disk!');
	}

	/**
	 * Odstraní nabídku ze vzdáleného serveru.
	 *
	 * @param array $projectData      Data projektu ve formátu iRest 1.
	 * @param array $additionalParams Specifická data pro daný export.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	function deleteProject(array $projectData, array $additionalParams=array())
	{
		$id = $projectData['id'];
		if ($this->_logger) {
			$this->_logger->logRemoteCall('deleteProject', $id);
		}
		if (file_exists($this->_getFn($id, 'projekt'))) {
			if (file_put_contents($this->_getFn($id, 'projekt'), "deleted")!==false) {
				return new Dalten_Export_Api_ServerResponse(true, 200, 'OK');
			}
			return new Dalten_Export_Api_ServerResponse(false, 0, 'Chyba disku - mazání!');
		}
		return new Dalten_Export_Api_ServerResponse(false, 0, 'Nabídka neexistuje!');
	}

	/**
	 * Odstraní nastavený logger pro backend.
	 *
	 * @return Dalten_Export_Api_LoggableApiInterface Fluent interface.
	 */
	public function removeLogger()
	{
		$this->_logger=null;
	}

	/**
	 * Nastaví logger pro backend.
	 * Logger bude použit pouze pokud to backend dovoluje.
	 *
	 * @param Dalten_Export_Api_Backend_Logger_LoggerInterface $logger Instance loggeru.
	 *
	 * @return bool Podařilo se přiřadit backendu logger?
	 */
	public function setLogger(Dalten_Export_Api_Backend_Logger_LoggerInterface $logger)
	{
		$this->_logger=$logger;
	}

	/**
	 * Ověří si v jakém stavu je export nabídky nemovitosti.
	 *
	 * @param array $listingData      Data nemovitosti.
	 * @param array $additionalParams Další parametry (jako při exportu).
	 *
	 * @return Dalten_Export_Api_ServerResponseVerifiable Odpověď serveru obsahující informaci o ověření.
	 */
	public function verifyListing(array $listingData, array $additionalParams)
	{
		if ($this->_logger) {
			$this->_logger->logRemoteCall('verifyListing', $listingData['id']);
		}

		if (strpos($listingData['nemovitost_titulek'], 'fail') !== false) {
			return new Dalten_Export_Api_ServerResponseVerifiable(false, 88, 'Fail!');
		}

		if (strpos($listingData['nemovitost_titulek'], 'progress') !== false) {
			$r= new Dalten_Export_Api_ServerResponseVerifiable(true, 88, 'In progress.');
			$r->setVerificationStatus(false);
			return $r;
		}

		if (strpos($listingData['nemovitost_titulek'], 'unverified') !== false) {
			$r= new Dalten_Export_Api_ServerResponseVerifiable(true, 88, 'Uvnerified');
			$r->setVerificationStatus(true, false);
			return $r;
		}

		if (strpos($listingData['nemovitost_titulek'], 'verified') !== false) {
			$r= new Dalten_Export_Api_ServerResponseVerifiable(true, 88, 'Verified');
			$r->setVerificationStatus(true, true);
			return $r;
		}

		return new Dalten_Export_Api_ServerResponseVerifiable(false, 88, 'fail || progress || verified || unverified');
	}


}