<?php
/**
 * Export na Hyperreality.cz.
 *
 * @category   Dalten
 * @package    Export
 * @subpackage Api
 */
class Dalten_Export_Api_Hyperreality implements Dalten_Export_Api_ApiInterface, Dalten_Export_Api_LoggableApiInterface,
Dalten_Export_Api_TopListingInterface, Dalten_Export_Api_ListListingsInterface
{
	private $_backend=null;
	private $_inDebugMode = false;

	/**
	 * Konstruktor.
	 *
	 * @param Dalten_Export_Api_Backend_Hyperreality $backend Backend.
	 */
	function __construct(Dalten_Export_Api_Backend_Hyperreality $backend)
	{
		$this->_backend = $backend;
	}

	/**
	 * Naváže spojení se vzdáleným serverem.
	 *
	 * @param string $login            Přihlašovací jméno.
	 * @param string $password         Heslo.
	 * @param string $softwareKey      Softwarový klíč.
	 * @param array  $additionalParams Pole dalších parametrů.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function openConnection($login, $password, $softwareKey = '', array $additionalParams = array())
	{
		if (!$this->_backend->login($login, $password)) {
			return new Dalten_Export_Api_ServerResponse(false, 88, 'Špatný login!');
		}
		if (!$this->_backend->checkFTPLogin()) {
			return new Dalten_Export_Api_ServerResponse(false, 88, 'Špatná konfigurace FTP!');
		}
		return new Dalten_Export_Api_ServerResponse(true);

	}

	/**
	 * Vrátí seznam nabídek, které jsou na vzdáleném serveru.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function getListingsList()
	{
		throw new BadMethodCallException('Not yet implemented!');
	}

	/**
	 * Vyexportuje nabídku.
	 *
	 * @param array $listingData      Data nabídky ve formátu iRest 1.
	 * @param array $userData         Data uživatele (makléře, kterému nabídka patří) ve formátu iRest 1.
	 * @param array $images           Pole fotografií nabídky ve formátu iRest 1.
	 * @param array $additionalParams Údaje které vrací server při exportu a jsou potřeba pro další
	 *                                práci s nabídkou.
	 *                                Data můžou sloužit pro editaci apod...
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru, jestli se export podařil nebo co selhalo.
	 */
	public function addListing(
		array $listingData,
		array $userData,
		array $images = array(),
		array $additionalParams = array()
	)
	{
		try {
			$response = $this->_backend->addAdvert($listingData, $userData, $images);
			return $this->_handleResponse($response);
		} catch (Dalten_Export_Exception_HumanReadable $e) {
			return new Dalten_Export_Api_ServerResponse(false, 88, $e->getMessage());
		} catch (Exception $e) {
			return $this->_handleExceptionResponse($e);
		}
	}


	/**
	 * Odstraní nabídku ze vzdáleného serveru.
	 *
	 * @param array $listingData      Data nabídky ve formátu iRest 1.
	 * @param array $additionalParams Specifická data pro daný export.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function deleteListing(array $listingData, array $additionalParams = array())
	{
		try {
			$response = $this->_backend->delAdvert($listingData);
			return $this->_handleResponse($response);
		} catch (Dalten_Export_Exception_HumanReadable $e) {
			return new Dalten_Export_Api_ServerResponse(false, 88, $e->getMessage());
		} catch (Exception $e) {
			return $this->_handleExceptionResponse($e);
		}
	}

	/**
	 * Vytopuje nemovitost.
	 *
	 * @param array $listingData      Data nemovitosti.
     * @param array $additionalParams Další parametry (nepoužívá se).
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function topListing(array $listingData, array $additionalParams = array())
	{
		try {
			$response = $this->_backend->topAdvert($listingData);
			return $this->_handleResponse($response);
		} catch (Dalten_Export_Exception_HumanReadable $e) {
			return new Dalten_Export_Api_ServerResponse(false, 88, $e->getMessage());
		} catch (Exception $e) {
			return $this->_handleExceptionResponse($e);
		}
	}

    /**
     * {@inheritDoc}
     */
    public function listListings()
    {
        try {
            $response = $this->_backend->listListings();

            if ($response && $response->code && $response->code==200) {
                $listingsList = [];
                foreach ($response->data->adverts->advert as $advert) {
                    $listingsList[] = [
                        'id'=>$advert->foreign_id
                    ];
                }
                return new Dalten_Export_Api_ServerResponse(true, 200, 'OK', $listingsList);
            }
            if ($response && $response->code && $response->code!=200) {
                return new Dalten_Export_Api_ServerResponse(
                    false,
                    $response->status,
                    ($response->message ? $response->message : '')
                );
            }
            return new Dalten_Export_Api_ServerResponse(false, 88, 'Response timed out!');
        } catch (Dalten_Export_Exception_HumanReadable $e) {
            return new Dalten_Export_Api_ServerResponse(false, 88, $e->getMessage());
        } catch (Exception $e) {
            return $this->_handleExceptionResponse($e);
        }
    }


    /**
	 * Z odpovědi vytvoří ServerResponse.
	 *
	 * @param mixed $response Response, v ideálním případě SimpleXMLElement.
	 *
	 * @return Dalten_Export_Api_ServerResponse Nastavená odpověď.
	 */
	private function _handleResponse($response)
	{
		if ($response && $response->code && $response->code==200) {
			$return = new Dalten_Export_Api_ServerResponse(
				true,
				$response->code,
				$response->status
			);
			if (isset($response->data->advert->detail)) {
				$return->setData(
					array(
						'detailUrl' => (string) $response->data->advert->detail, // standardní předávání URL detailu do Irestu
						'remoteUrl' => (string) $response->data->advert->detail // zpětná kompatibilita
					)
				);
			}
			return $return;
		}
		if ($response && $response->code && $response->code!=200) {
			return new Dalten_Export_Api_ServerResponse(
				false,
				$response->status,
				($response->message ? $response->message : '')
			);
		}
		return new Dalten_Export_Api_ServerResponse(false, 88, 'Response timed out!');
	}

	/**
	 * Nastaví logger pro backend.
	 * Logger bude použit pouze pokud to backend dovoluje.
	 *
	 * @param Dalten_Export_Api_Backend_Logger_LoggerInterface $logger Instance loggeru.
	 *
	 * @return bool Podařilo se přiřadit backendu logger?
	 */
	public function setLogger(Dalten_Export_Api_Backend_Logger_LoggerInterface $logger)
	{
		$this->_backend->setLogger($logger);
		return true;
	}

	/**
	 * Odstraní nastavený logger pro backend.
	 *
	 * @return Dalten_Export_Api_LoggableApiInterface Fluent interface.
	 */
	public function removeLogger()
	{
		$this->_backend->removeLogger();
		return $this;
	}



	/**
	 * Naformátuje vyjímku do odpovědi exportu, ale jen pokud má třída nastavený debug mód.
	 *
	 * @param Exception $e Vyjímka k vypsání.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	private function _handleExceptionResponse(Exception $e)
	{
		if ($this->_inDebugMode) {
			return new Dalten_Export_Api_ServerResponse(
				false,
				88,
				get_class($e) . ':' . $e->getMessage() . PHP_EOL .
					$e->getFile() . ':' . $e->getLine() . PHP_EOL .
					PHP_EOL . $e->getTraceAsString()
			);
		}
		return new Dalten_Export_Api_ServerResponse(false);
	}
}
