<?php
/**
 * Převodník z adresního sytému Irestů (UIR) do adresního systému používaného globálním RE/MAXem.
 *
 * Protože se krmí soubory je univerzální skrz všecky Iresty.
 *
 * @category Dalten
 * @package  AdressConverter
 */
class Dalten_AddressConverter_Rex_Universal implements Dalten_AddressConverter_Rex_Interface
{
	private $_map = array();

	/**
	 * Kontruktor. Nahrává převodní mapu z INI souboru.
	 *
	 * @param string $mapFile Cesta k INI souboru s převodníkem.
	 */
	function __construct($mapFile)
	{
		$this->_map = parse_ini_file($mapFile, true);
	}

	/**
	 * Vrátí, zda jsme schopni tuto adresu převést.
	 *
	 * Vrací false, pokud pro daný stát nemáme číselníky.
	 *
	 * @param array $input (Původní) informace o nemovitosti.
	 *
	 * @return bool Jsme schopni tuto adresu převést?
	 */
	function isValidRegion(array $input)
	{
		if ($input['nemovitost_uir_stat']==$this->_map['meta']['stat']) {
			return true;
		}
		return false;
	}

	/**
	 * Převede adresu u nemovitosti do adresního systému používaného globálním REMAXem.
	 *
	 * Pozor: Převedená adresa obsahuje jejich druh údajů: RegionID, CityID, LocalZoneID
	 *
	 * @param array $input Informace o nemovitosti.
	 *
	 * @return array Informace o nemovitosti s upravenou adresou.
	 */
	function convertAddress(array $input)
	{
		$originalOkres = $input['nemovitost_uir_okres_kod'];
		if (!isset($input['nemovitost_uir_kraj_kod'])) {
			$originalKraj = $this->_map['okres_na_kraj'][$originalOkres];
		} else {
			$originalKraj = $input['nemovitost_uir_kraj_kod'];
		}
		$originalObec = $input['nemovitost_uir_obec_kod'];

		$input['RegionRowID'] = $this->_map['kraj'][$originalKraj];
		if (isset($this->_map['obec'][$originalObec])) {
			$input['CityID'] = $this->_map['obec'][$originalObec];
		}

		if (isset($input['nemovitost_uir_cobce_kod'])) {
			$originalCobce = $input['nemovitost_uir_cobce_kod'];
			if (isset($this->_map['cobce'][$originalCobce])) {
				$input['CityID'] = $this->_map['cobce'][$originalCobce];
			}
		}
		if (isset($this->_map['LocalLanguageZoneName'][$originalObec])) {
			$input['LocalZoneID'] = $this->_map['LocalLanguageZoneName'][$originalObec];
		}

		if (!isset($input['CityID'])) {
			$input['CityID'] = $this->_map['fallback']['CityID'];
			// CityID musíme dát jinak globál nepřijme celý soubor.
		}

		return $input;
	}

	/**
	 * Převede adresu ale vrátí jen CityID. Tato funkce se používá při zařazování kanceláří.
	 *
	 * @param array $originalData Originální data adresy v poli.
	 *                            Musí minimálně obsahovat klíče nemovitost_uir_okres_kod
	 *                            a nemovitost_uir_obec_kod.
	 *
	 * @return int Převedené CityID
	 */
	function convertCityID(array $originalData)
	{
		$address = $this->convertAddress($originalData);
		if (!isset($address['CityID'])) {
			$address['CityID'] = $this->_map['fallback']['CityID'];
		}
		return $address['CityID'];
	}

}