<?php
/**
 * Export na Expressz.hu. konvetorová část.
 *
 * @category Dalten
 * @package  Export
 */
class Dalten_Export_Expresszhu extends Dalten_Export_AbstractExport
{
	const ZEMEDELSKE_OBJEKTY = 1;
	const KOMERCNI_OBJEKTY = 2;
	const POZEMKY = 3;
	const BYTY = 4;
	const HISTORICKE_OBJEKTY = 5;
	const DOMY_A_VILY = 6;
	const HOTELY = 7;
	const NAJEMNI_DOMY = 8;
	const KOMERCNI_PROSTORY = 9;
	const CHATY = 10;
	const MALE_OBJEKTY = 11;

	const PRODEJ = 1;
	const PRONAJEM = 2;

	const PODTYP_RESTAURACE = 3;

	const PODTYP_SKLAD = 1;
	const PODTYP_VYROBA = 2;
	const PODTYP_OBCHODNI_CENTRUM = 3;
	const PODTYP_NAKUPNI_STREDISKO = 4;
	const PODTYP_PRUMYSLOVY_AREAL = 5;
	const PODTYP_ADMINISTRATIVNI_BUDOVA = 7;

	const KANCELARE = 1;
	const OBCHODNI = 2;
	const SKLADOVACI = 3;
	const JINE = 4;
	const VYROBNI = 5;

	const PRO_KOMERCNI_VYSTAVBU = 1;
	const PRO_BYDLENI = 2;
	const ZEMEDELSKA_PUDA = 3;
	const LES = 4;
	const LOUKA = 5;
	const ZAHRADA = 6;
	const OSTATNI = 7;

	const DISPOZICE_BYTU_GARSONKA = 1;

	const DRUH_BYTU_PANEL = 3;

	const DRUH_OBJEKTU_CIHLA = 2;
	const DRUH_OBJEKTU_PANEL = 5;

	const POLOHA_RADOVY=1;
	const POLOHA_ROHOVY=2;
	const POLOHA_V_BLOKU=3;
	const POLOHA_SAMOSTATNY=4;

	const KATEGORIE_KANCELARE_JINA=0;
	const KATEGORIE_KANCELARE_A=1;
	const KATEGORIE_KANCELARE_B=2;
	const KATEGORIE_KANCELARE_C=3;

	const OBEC_BUDAPEST = 1;

	private $_adresar = array();

	private $_kerulety = array(
		1=>1,
		2=>2,
		3=>3,
		4=>4,
		5=>9,
		6=>5,
		7=>6,
		8=>7,
		9=>8,
		10=>10,
		11=>11,
		12=>12,
		13=>13,
		14=>14,
		15=>19,
		16=>15,
		17=>16,
		18=>17,
		19=>18,
		20=>20,
		21=>21,
		22=>22,
		23=>23
	);

	/**
	 * Konstruktor.
	 *
	 * @param Serenity_Config_Config                          $codebooks Číselníky.
	 * @param Serenity_Config_Config                          $entities  Definice entit.
	 * @param Dalten_Config_Processer_ProcesserInterface|null $processer Zpracovatel hodnot.
	 * @param array                                           $adresy    Pole pro převod adres
	 */
	public function __construct(
		Serenity_Config_Config $codebooks,
		Serenity_Config_Config $entities,
		Dalten_Config_Processer_ProcesserInterface $processer = null,
		$adresy = array()
	)
	{
		$this->_codebooks = $codebooks;
		$this->_entities = $entities;
		$this->_processer = ($processer)
			? $processer
			: new Dalten_Config_Processer_DefaultProcesser($codebooks);
		$this->_adresar = $adresy;
	}

	/**
	 * Provadi zavercene upravy na prevedenych hodnotach.
	 *
	 * @param array  $convertedValues Prevedene hodnoty
	 * @param array  $originalvalues  Originalni hodnoty
	 * @param string $entityName      Jmeno entity
	 *
	 * @throws Dalten_Export_Exception_HumanReadable Chyba s vysvětlením jaké údaje chybí.
	 *
	 * @return array Doupravené hodnoty.
	 */
	function _convert(array $convertedValues, array $originalvalues, $entityName)
	{

		if ($entityName == 'ad_meta') {
			$convertedValues['category'] = $this->convertCategory($originalvalues);
			$today = new DateTime();
			$dateFormat = 'Y.m.d H:i:s';
			$convertedValues['appears-at'] = $today->format($dateFormat);
			$convertedValues['expires-at'] = $today->add(new DateInterval('P14D'))->format($dateFormat);

			$krajKod = $originalvalues['nemovitost_uir_kraj_kod'];
			$obecKod = $originalvalues['nemovitost_uir_obec_kod'];
			$convertedValues['region'] = (
			isset($this->_adresar['county'][$krajKod])
				? $this->_adresar['county'][$krajKod]
				: '/magyar'
			);

			if ($originalvalues['nemovitost_mena']==2) {
				throw new Dalten_Export_Exception_HumanReadable('Hiba: USD is invalid currency!');
			}

			$this->_adresar['region'][self::OBEC_BUDAPEST] = '/magyar/budapest/special';
			if (!isset($this->_adresar['region'][$obecKod])) {
				//tohle znamena: Chyba: Spatna adresa! Zadejte prosim mesto!
				throw new Dalten_Export_Exception_HumanReadable('Hiba: Helytelen a cím! Adja meg a várost!');
			}
			if (isset($this->_adresar['region'][$obecKod])) {
				$convertedValues['region'] = $this->_adresar['region'][$obecKod];
			}

			if ($obecKod==self::OBEC_BUDAPEST) {
				$kodKeruletu = $originalvalues['nemovitost_uir_mcast_kod'];
				if ($kodKeruletu>0 and $kodKeruletu<24) {
					$convertedValues['region'] = '/magyar/budapest/' . $this->_kerulety[$kodKeruletu] . 'ker';
				} else {
					//tohle znamena: Chyba: Spatna adresa! Dodejte okrsek Budapesti!
					throw new Dalten_Export_Exception_HumanReadable(
						'Hiba: Helytelen a cím! Adja meg Budapest egyik kerületét!'
					);
				}
			}

		}

		if ($entityName == 'items') {
			$category = $this->convertCategory($originalvalues);
			if ($this->startsWith('/ingatlan/mezogazdasag', $category)) {
				switch ($category) {
					case '/ingatlan/mezogazdasag/szantofold':
						$ucelVidli = 'szántóföld';
						break;
					case '/ingatlan/mezogazdasag/erdo':
						$ucelVidli = 'erdő';
						break;
					default:
						$ucelVidli = 'egyéb';
						break;
				}
				$convertedValues['terulet_hasznositasa'] = $ucelVidli;
			}

			$patro = (isset($originalvalues['nemovitost_podlazi_cislo'])
				? intval($originalvalues['nemovitost_podlazi_cislo']) : 1);
			if ($patro == 1) {
				$convertedValues['emelet'] = 'földszint';
			} elseif ($patro < 0) {
				$convertedValues['emelet'] = 'szuterén';
			} elseif ($patro > 20) {
				$convertedValues['emelet'] = 'tetőtér';
			} else {
				$convertedValues['emelet'] = ($patro + 1) . '.';
			}

			if (strpos($originalvalues['nemovitost_elektrina'], '1')!==false) {
				$convertedValues['villany'] = '1';
			}

			//přidáváme plochu podlahy a pozemku
			$plochy = $this->convertAreas($originalvalues);
			foreach ($plochy as $k=>$v) {
				$convertedValues[$k] = $v;
			}

			if ($convertedValues['szobaszam']==1 and $originalvalues['nemovitost_typ']==self::BYTY) {
				$prevodnikVelikostiBytu = array(
					1 => 1,
					2 => 1,
					3 => 2,
					4 => 3,
					5 => 4,
					6 => 5,
					7 => 6,
					8 => 7,
					9 => 1,
					10 => 2,
					11 => 3,
					12 => 4,
					13 => 5,
					14 => 6,
					15 => 7,
					16 => 3,
					17 => 3

				);
				$convertedValues['szobaszam'] = $prevodnikVelikostiBytu[$originalvalues['nemovitost_dispozice']];
			}
		}

		if (($entityName == 'ad') or ($entityName == 'ad_meta') or ($entityName == 'items')) {
			//tohleto unsetuje podle jejich kategorie inzerátu
			$category = $this->convertCategory($originalvalues);
			foreach ($this->_entities->$entityName as $item) {

				if (isset($item->invalidForCategory) and isset($convertedValues[$item->column])) {
					if (!is_array($item->invalidForCategory)) {
						$item->invalidForCategory = array($item->invalidForCategory);
					}
					foreach ($item->invalidForCategory as $checkedCategory) {
						if ($this->startsWith($checkedCategory, $category)) {
							unset($convertedValues[$item->column]);
						}
					}
				}

				if (isset($item->onlyForCategory) and isset($convertedValues[$item->column])) {
					if (!is_array($item->onlyForCategory)) {
						$item->onlyForCategory = array($item->onlyForCategory);
					}
					$willUnset = true;
					foreach ($item->onlyForCategory as $whiteListed) {
						if ($this->startsWith($whiteListed, $category)) {
							$willUnset = false;
							break;
						}
					}
					if ($willUnset) {
						unset($convertedValues[$item->column]);
					}
				}

			}
		}

		return $convertedValues;
	}

	/**
	 * Predpripravi hodnoty na zpracovani v newspeaku (INI souboru)
	 *
	 * @param array|mixed $values     Hodnoty
	 * @param string      $config     Konfigurace
	 * @param string      $entityName Nazev entity
	 *
	 * @return array|mixed
	 */
	function _format(array $values, $config, $entityName)
	{
		foreach ($values as $k => $v) {
			if (isset($config->$k)) {
				if (isset($config->$k->solvedInPHP)) {
					switch ($config->$k->solvedInPHP) {
						case 'first checked':
							// převod z multicheckboxu na select (vybereme první zašktlý)
							$pos = strpos($v, '1');
							$values[$k] = ($pos === false ? -1 : $pos);
							break;
						default:
							break;
					}
				}
			}
		}
		return $values;
	}

	/**
	 * Pomocna funkce, vraci zda haystack zacina na needle
	 *
	 * @param string $needle   Po cem patra
	 * @param string $haystack Co kontroluje
	 *
	 * @return bool
	 */
	function startsWith($needle, $haystack)
	{
		return (stripos($haystack, $needle, 0) === 0);
	}

	/**
	 * Vybere správné typy ploch pro kombinace našich a jejich typů.
	 *
	 * @param array $originalValues Pole původních hodnot.
	 *
	 * @return array Pole přidaných ploch.
	 */
	function convertAreas($originalValues)
	{
		$typ = $originalValues['nemovitost_typ'];
		$plochy = array();
		switch ($typ) {
			case self::ZEMEDELSKE_OBJEKTY:
				$plochy['telekterulete'] = $originalValues['nemovitost_plocha_uzitna'];
				break;
			case self::KOMERCNI_OBJEKTY:
				$plochy['alapterulet'] = $originalValues['nemovitost_plocha_celkova'];
				break;
			case self::POZEMKY:
				$plochy['telekterulete'] =  $originalValues['nemovitost_plocha_celkova'];
				break;
			case self::BYTY:
				$plochy['alapterulet'] = $originalValues['nemovitost_plocha_celkova'];
				break;
			case self::HISTORICKE_OBJEKTY:
				//oni nemaji pro egyeb kategorii plochy
				break;
			case self::DOMY_A_VILY:
				$plochy['alapterulet'] = $originalValues['nemovitost_plocha_uzitna'];
				$plochy['telekterulete'] = $originalValues['nemovitost_plocha_parcely'];
				break;
			case self::HOTELY:
				$plochy['alapterulet'] = $originalValues['nemovitost_plocha_uzitna'];
				break;
			case self::NAJEMNI_DOMY:
			case self::KOMERCNI_PROSTORY:
			case self::CHATY:
				$nenulove = ( $originalValues['nemovitost_plocha_uzitna']
					? $originalValues['nemovitost_plocha_uzitna'] : (
						$originalValues['nemovitost_plocha_parcely']
							? $originalValues['nemovitost_plocha_parcely'] : 1)
				);
				$plochy['alapterulet'] = $nenulove;
				break;
			case self::MALE_OBJEKTY:
				$plochy['alapterulet'] = $originalValues['nemovitost_plocha_zastavena'];
			default:
				break;
		}
		return $plochy;
	}

	/**
	 * Odvodi kategorii nemovitosti (specifickou pro Expressz.hu) podle puvodnich informaci o nemovitosti
	 *
	 * @param array $originalvalues Puvodni informace o nemovitosti
	 *
	 * @return string
	 */
	function convertCategory($originalvalues)
	{
		$typ = $originalvalues['nemovitost_typ'];

		//fallback
		$category = '/ingatlan/egyeb';

		switch ($typ) {
			case self::ZEMEDELSKE_OBJEKTY:
				$category = '/ingatlan/mezogazdasag/egyeb';
				break;
			case self::KOMERCNI_OBJEKTY:
				switch ($originalvalues['nemovitost_ucel_budovy']) {
					case self::PODTYP_SKLAD:
						$category = '/ingatlan/ipari/raktar';
						break;
					case self::PODTYP_OBCHODNI_CENTRUM:
					case self::PODTYP_NAKUPNI_STREDISKO:
						$category = '/ingatlan/kereskedelmi/uzlet';
						break;
					case self::PODTYP_PRUMYSLOVY_AREAL:
						$category = '/ingatlan/ipari/telephely';
						break;
					case self::PODTYP_VYROBA:
						$category = '/ingatlan/ipari/gyarepulet';
						break;
					case self::PODTYP_ADMINISTRATIVNI_BUDOVA:
						switch ($originalvalues['nemovitost_kategorie_kancelare']){
							case self::KATEGORIE_KANCELARE_A:
								$category = '/ingatlan/iroda/a-irodahaz';
								break;
							case self::KATEGORIE_KANCELARE_B:
								$category = '/ingatlan/iroda/b-irodahaz';
								break;
							case self::KATEGORIE_KANCELARE_C:
								$category = '/ingatlan/iroda/c-irodahaz';
								break;
							default:
								$category = '/ingatlan/iroda/egyeb';
								break;
						}
						break;
					default:
						$category = '/ingatlan/kereskedelmi/egyeb';
						break;
				}
				break;
			case self::KOMERCNI_PROSTORY:
				switch ($originalvalues['nemovitost_druh_prostor']) {
					case self::KANCELARE:
						switch ($originalvalues['nemovitost_kategorie_kancelare']){
							case self::KATEGORIE_KANCELARE_A:
								$category = '/ingatlan/iroda/a-irodahaz';
								break;
							case self::KATEGORIE_KANCELARE_B:
								$category = '/ingatlan/iroda/b-irodahaz';
								break;
							case self::KATEGORIE_KANCELARE_C:
								$category = '/ingatlan/iroda/c-irodahaz';
								break;
							default:
								$category = '/ingatlan/iroda/egyeb';
								break;
						}
						break;
					case self::SKLADOVACI:
						$category = '/ingatlan/ipari/raktar';
						break;
					default:
						$category = '/ingatlan/kereskedelmi/egyeb';
						break;
				}
				break;
			case self::POZEMKY:
				switch ($originalvalues['nemovitost_druh_pozemku']) {
					case self::PRO_KOMERCNI_VYSTAVBU:
						$category = '/ingatlan/fejlesztesi-terulet/kereskedelmi-szolgaltato';
						break;
					case self::PRO_BYDLENI:
						$category = '/ingatlan/telek/lako';
						break;
					case self::ZEMEDELSKA_PUDA:
						$category = '/ingatlan/mezogazdasag/szantofold';
						break;
					case self::LES:
						$category = '/ingatlan/mezogazdasag/erdo';
						break;
					case self::LOUKA:
						$category = '/ingatlan/mezogazdasag/legelo';
						break;
					default:
						$category = '/ingatlan/telek/egyeb';
						break;
				}
				break;
			case self::BYTY:
				if ($originalvalues['zakazka_typ'] == self::PRODEJ) {
					if ($originalvalues['nemovitost_dispozice'] == self::DISPOZICE_BYTU_GARSONKA) {
						$category = '/ingatlan/lakas-haz/garzon';
					} else {
						if ($originalvalues['nemovitost_druh_objektu'] == self::DRUH_OBJEKTU_PANEL) {
							$category = '/ingatlan/lakas-haz/panel';
						} else {
							$category = '/ingatlan/lakas-haz/tegla';
						}
					}
				} else {
					if ($originalvalues['nemovitost_dispozice'] == self::DISPOZICE_BYTU_GARSONKA) {
						$category = '/ingatlan/alberlet/garzon-lakas';
					} else {
						if ($originalvalues['nemovitost_druh_objektu'] == self::DRUH_OBJEKTU_PANEL) {
							$category = '/ingatlan/alberlet/panellakas';
						} else {
							$category = '/ingatlan/alberlet/tegla-lakas';
						}

					}
				}
				break;
			case self::DOMY_A_VILY:
				if ($originalvalues['zakazka_typ'] == self::PRODEJ) {
					$category = '/ingatlan/lakas-haz/egyeb';
					switch ($originalvalues['nemovitost_poloha_objektu']) {
						case self::POLOHA_RADOVY:
							$category = '/ingatlan/lakas-haz/csaladi';
							break;
						case self::POLOHA_ROHOVY:
							$category = '/ingatlan/lakas-haz/ikerhaz';
							break;
						case self::POLOHA_V_BLOKU:
						case self::POLOHA_SAMOSTATNY:
						default:
							$category = '/ingatlan/lakas-haz/sorhaz';
							break;
					}
				} else {
					$category = '/ingatlan/alberlet/egyeb';
					switch ($originalvalues['nemovitost_poloha_objektu']){
						case self::POLOHA_RADOVY:
							$category = '/ingatlan/alberlet/csaladi-haz';
							break;
						case self::POLOHA_ROHOVY:
							$category = '/ingatlan/alberlet/ikerhaz';
							break;
						case self::POLOHA_V_BLOKU:
						case self::POLOHA_SAMOSTATNY:
						default:
							$category = '/ingatlan/alberlet/sorhaz';
							break;
					}
				}
				break;
			case self::HOTELY:
				$category = '/ingatlan/kereskedelmi/egyeb';
				if ($originalvalues['nemovitost_typ_zarizeni'] == self::PODTYP_RESTAURACE) {
					$category = '/ingatlan/kereskedelmi/vendeglato';
				}
				break;
			case self::NAJEMNI_DOMY:
				if ($originalvalues['zakazka_typ'] == self::PRODEJ) {
					$category = '/ingatlan/lakas-haz/egyeb';
				} else {
					$category = '/ingatlan/alberlet/egyeb';
				}
				break;
			case self::CHATY:
				$category = '/ingatlan/udulo/nyaralo';
				break;
			case self::MALE_OBJEKTY:
				switch ($originalvalues['nemovitost_poloha_objektu']){
					case self::POLOHA_RADOVY:
						$category = '/ingatlan/garazs/onallo';
						break;
					case self::POLOHA_ROHOVY:
					case self::POLOHA_SAMOSTATNY:
					case self::POLOHA_V_BLOKU:
					default:
						$category = '/ingatlan/garazs/teremgarazs';
						break;
				}
				break;

			default:
				break;
		}
		return $category;
	}
}