<?php
/**
 * Backend API Srealit.
 * Jednotlivé metody by měly odpovídat metodám na vzdáleném serveru.
 * Pro jejich volání je použito XML RPC.
 *
 * @category   Dalten
 * @package    Export
 * @subpackage Api
 */
class Dalten_Export_Api_Backend_ViarealityAccounts implements Dalten_Export_Api_Backend_LoggableBackendInterface
{
	/**
	 * Adpatér pro volání metod přes XML RPC.
	 *
	 * @var Serenity_Rpc_XmlRpc
	 */
	private $_adapter;

	/**
	 * Konstruktor.
	 *
	 * @param Serenity_Rpc_XmlRpc $adapter Adpatér pro volání metod přes XML RPC.
	 */
	public function __construct(Serenity_Rpc_XmlRpc $adapter)
	{
		$this->_adapter = $adapter;
	}

	/**
	 * Získání řetězce zvaného session id, který je použit pro hashování hesla.
	 * Metoda slouží k získání řetězce pro hashování hesla.
	 * Po zavolání s existujícím client id je vrácen status 200 a v output je
	 * session id a hashovací klíč. Volá se vždy na začátku relace.
	 *
	 * @param int $clientId ID klienta.
	 *
	 * @return Dalten_Data_ArrayObject Po zavolání se správným clientId je vrácen objekt:
	 *                                 <code>
	 *                                 {
	 *                                     status: 200,
	 *                                     statusMessage: '...',
	 *
	 *                                     output: {
	 *                                         sessionId: '...',
	 *                                         hashKey: '...'
	 *                                     }
	 *                                 }
	 *                                 </code>
	 */
	public function getHash($clientId)
	{
		$response = $this->_adapter->getHash((int) $clientId);

		$output = array(null, null);
		if (!empty($response['output']) && is_array($response['output'])) {
			$output = (array) current($response['output']);
		}

		$response['output'] = array(
			'sessionId' => current($output),
			'hashKey' => next($output)
		);

		return new Dalten_Data_ArrayObject($response);
	}

	/**
	 * Potvrzení přihlášení.
	 * Metoda login potvrdí session jako autorizovanou. V kladném případě se ověří, že výpočet
	 * nového session id odpovídá očekávání na straně serveru. Metodě login se předávají
	 * parametry session id z metody getHash. Pozor, session id už musí být přepočítáno pomocí vzorečku.
	 * Softwarový klíč je přidělen administrátorem srealit zvlášť pro každého klienta (RK) v závislosti
	 * na použitém exportním software. Nazpět je vrácen status úspěšnosti.
	 *
	 * Poznámka: Bez úspěšně dokončené metody login je celá importní relace ve stavu nepřihlášen
	 * a tedy nelze v exportu zakázek pokračovat.
	 *
	 * Vrací-li metoda opakovaně 407 Bad session je špatně zadané heslo, importní klíč a
	 * nebo client id u klienta nebo na importním serveru. Je třeba zkontrolovat tyto údaje. Heslo
	 * na import si nastavuje klient sám, importní klíč lze měnit přes obchodního zástupce,
	 * clientid bylo firmě přiděleno.
	 *
	 * @param string $sessionId ID relace.
	 *
	 * @return Dalten_Data_ArrayObject <code>
	 *                                 {
	 *                                     status: xxx,
	 *                                     statusMessage: '...',
	 *
	 *                                     output: []
	 *                                 }
	 *                                 </code>
	 */
	public function login($sessionId)
	{
		$response = $this->_adapter->login((string) $sessionId);

		return new Dalten_Data_ArrayObject($response);
	}

	/**
	 * Odhlášení.
	 *
	 * @param string $sessionId ID relace.
	 *
	 * @return Dalten_Data_ArrayObject <code>
	 *                                 {
	 *                                     status: xxx,
	 *                                     statusMessage: '...',
	 *
	 *                                     output: []
	 *                                 }
	 *                                 </code>
	 */
	public function logout($sessionId)
	{
		return new Dalten_Data_ArrayObject($this->_adapter->logout((string) $sessionId));
	}


	public function addAccount($sessionId, array $accountData)
	{
		$response = $this->_adapter->addAccount((string) $sessionId, $accountData);
		return $response;
	}

	public function addLogo($sessionId, $accountId, $logo)
	{
		$logo = $this->_adapter->setType($logo, 'base64');
		$response = $this->_adapter->addLogo((string) $sessionId, (int) $accountId, $logo);
		return $response;
	}

	public function delAccount($sessionId, $accountId)
	{
		$response = $this->_adapter->delAccount((string) $sessionId, (int) $accountId);
		return $response;
	}

	public function listAccount($sessionId)
	{
		$response = $this->_adapter->listAccount((string) $sessionId);
		return $response;
	}

	/**
	 * Nastaví logger pro backend.
	 *
	 * @param \Dalten_Export_Api_Backend_Logger_LoggerInterface $logger Instance loggeru.
	 *
	 * @return Dalten_Export_Api_Backend_Sreality Fluent interface.
	 */
	public function setLogger(Dalten_Export_Api_Backend_Logger_LoggerInterface $logger)
	{
		if ($this->_adapter instanceof Dalten_Export_Api_Backend_LoggableBackendInterface) {
			$this->_adapter->setLogger($logger);
		}

		return $this;
	}

	/**
	 * Odstraní nastavený logger pro backend.
	 *
	 * @return Dalten_Export_Api_Backend_Sreality Fluent interface.
	 */
	public function removeLogger()
	{
		if ($this->_adapter instanceof Dalten_Export_Api_Backend_LoggableBackendInterface) {
			$this->_adapter->removeLogger();
		}

		return $this;
	}
}