<?php

/**
 * API pro export na vlastní web RK.
 *
 * @category   Dalten
 * @package    Export
 * @subpackage Api
 */
class Dalten_Export_Api_Website implements Dalten_Export_Api_LoggableApiInterface, Dalten_Export_Api_CloseableApiInterface, Dalten_Export_Api_ProjectApiInterface
{
	/**
	 * @var Dalten_Export_Api_Backend_Website
	 */
	protected  $_backend = null;

	protected  $_sessionId = null;
    private Serenity_Config_Config $_config;
    private Dalten_Export_Website $_export;

    /**
	 * Konstruktor. Nastavuje závislosti.
	 *
	 * @param Dalten_Export_Website             $export  Instance exportu.
	 * @param Serenity_Config_Config            $config  Konfigurace exportu.
	 * @param Dalten_Export_Api_Backend_Website $backend Instance backendu.
	 */
	public function __construct(Dalten_Export_Website $export, Serenity_Config_Config $config,
		Dalten_Export_Api_Backend_Website $backend)
	{
		$this->_export = $export;
		$this->_config = $config;
		$this->_backend = $backend;
	}

	/**
	 * Naváže spojení se vzdáleným serverem.
	 *
	 * @param string $login			   Přihlašovací jméno.
	 * @param string $password		   Heslo.
	 * @param string $softwareKey	   Softwarový klíč.
	 * @param array  $additionalParams Pole dalších parametrů.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function openConnection($login, $password, $softwareKey = '', array $additionalParams = array())
	{
		$response = $this->_backend->getHash($login);
		$hashKey = empty($response['output']['hash']) ? null : $response['output']['hash'];
		$this->_sessionId = empty($response['output']['session_id']) ? null : $response['output']['session_id'];
		$response = $this->_backend->login($this->_sessionId, md5(md5($password). $hashKey));

		return $this->_convertDataObjectIntoServerResponse($response);
	}

	/**
	 * Odhlašuje přihlášené spojení.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď zda se podařilo zavřít spojení.
	 */
	public function closeConnection()
	{
		return $this->_convertDataObjectIntoServerResponse(
			$this->_backend->logout($this->_sessionId)
		);
	}

	/**
	 * Vyexportuje nabídku.
	 *
	 * Pokud je pole userData prázdné, uživatel nebude exportován (může být přidán až později přes synchronizaci).
	 *
	 * @param array $listingData      Data nabídky ve formátu iRest 1.
	 * @param array $userData         Data uživatele (makléře, kterému nabídka patří) ve formátu iRest 1.
	 * @param array $images           Pole fotografií nabídky ve formátu iRest 1.
	 * @param array $additionalParams Pro tento export, nejsou přídavná data potřebná.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru, jestli se export podařil nebo co selhalo.
	 */
	public function addListing(array $listingData, array $userData, array $images = array(), array $additionalParams = array())
	{
		$convertedImages = array();
		foreach ($images as $image) {
			$image['typ_parent'] = 1;
			$image['typ_prilohy'] = 1;
			$image['id_parent'] = $listingData['id'];
			$convertedImages[] = $image;
		}
		$listingData['priloha'] = $convertedImages;

		if (!empty($additionalParams['listing']['hide_price'])) {
			$listingData['nemovitost_cena_pronajem'] = $listingData['nemovitost_cena_prodej'] = 0;
		}

		if (isset($additionalParams['listing']['show_panorama'])) {
			$listingData['zobrazit_panorama'] = $additionalParams['listing']['show_panorama'] ? 1 : 0;
		}

		$response = $this->_backend->addNabidka($this->_sessionId, $listingData);

		if ($response['status'] >= 200 && $response['status'] < 300 && $userData) {

			$userResponse = $this->addUser($userData);

			// Pokud je export úspěšný, tak tam server rve co se nedalo do db... což nemusí uživatelé vidět
			$response['statusMessage'] = '';

			if ($userResponse->wasSuccessful()) {
				$response['status'] = $userResponse->getCode();
				// message nepřikládáme, protože je v ní opět bordel
			}
		}

		return $this->_convertDataObjectIntoServerResponse($response);
	}

	/**
	 * Odstraní nabídku ze vzdáleného serveru.
	 *
	 * @param array $listingData      Data nabídky ve formátu iRest 1.
	 * @param array $additionalParams Specifická data pro daný export (zde prázdné pole).
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function deleteListing(array $listingData, array $additionalParams = array())
	{
		return $this->_convertDataObjectIntoServerResponse(
			$this->_backend->delNabidka($this->_sessionId, $listingData['id'])
		);
	}

	/**
	 * Vrátí seznam inzerátů na serveru.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru (getData() => pole id nabídek).
	 */
	public function getListingsList()
	{
		return $this->_convertDataObjectIntoServerResponse($this->_backend->listNabidka($this->_sessionId));
	}

	/**
	 * Přidá uživatele.
	 *
	 * @param array $userData Pole dat uživatele ve formátu iRest1.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function addUser(array $userData)
	{
		$exportWhitelist = array(
			'id',
			'id_firma',
			'id_pobocka',
			'id_uroven',
			'email',
			'telefon',
			'jmeno_cele',
			'uzivatel_os_jmeno',
			'uzivatel_os_prijmeni',
			'uzivatel_tituly_pred_jmenem',
			'uzivatel_tituly_za_jmenem',
			'exportovat_foto',
			'foto',
			'moto',
			'body',
			'body_ze_site',
			'datum_narozeni',
			'vzdelani',
			'oceneni',
			'id_pozice',
			'zobrazovat_na_webu',
			'hodnoceni',
			'certifikace',
			'dalsi_pozice',
            'makler_ico',
            'datova_schranka_id'
		);

		$exportedUser = array();
		foreach ($exportWhitelist as $itemName) {
			if (isset($userData[$itemName])) {
				$exportedUser[$itemName] = $userData[$itemName];
			}
		}

		return $this->_convertDataObjectIntoServerResponse($this->_backend->addUzivatel($this->_sessionId, $exportedUser));
	}

	/**
	 * Odstraní uživatele.
	 *
	 * @param array $userData Pole dat uživatele ve formátu iRest1.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function deleteUser(array $userData)
	{
		return $this->_convertDataObjectIntoServerResponse(
			$this->_backend->delUzivatel($this->_sessionId, $userData['id'])
		);
	}

	/**
	 * Vrátí seznam id exportovaných uživatelů.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru (getData() => pole id uživatelů).
	 */
	public function getUsersList()
	{
		return $this->_convertDataObjectIntoServerResponse(
			$this->_backend->listUzivatel($this->_sessionId)
		);
	}

	/**
	 * Nastaví logger pro backend.
	 *
	 * Logger bude použit pouze pokud to backend dovoluje.
	 *
	 * @param Dalten_Export_Api_Backend_Logger_LoggerInterface $logger Instance loggeru.
	 *
	 * @return bool Podařilo se přiřadit backendu logger?
	 */
	public function setLogger(Dalten_Export_Api_Backend_Logger_LoggerInterface $logger)
	{
		if ($this->_backend instanceof Dalten_Export_Api_Backend_LoggableBackendInterface) {
			$this->_backend->setLogger($logger);
		}
		return false;
	}

	/**
	 * Odstraní nastavený logger pro backend.
	 *
	 * @return Dalten_Export_Api_LoggableApiInterface Fluent interface.
	 */
	public function removeLogger()
	{
		$this->_backend->removeLogger();

		return $this;
	}

	/**
	 * Převádí odpověď od backendu do formátu Dalten_Export_Api_ServerResponse.
	 *
	 * @param Dalten_Data_ArrayObject $data Odpověď backendu.
	 *
	 * @return Dalten_Export_Api_ServerResponse Přeformátovaná data jako Dalten_Export_Api_ServerResponse.
	 */
	protected function _convertDataObjectIntoServerResponse(Dalten_Data_ArrayObject $data)
	{
        $status = isset($data->status) ? $data->status : 500;

        return new Dalten_Export_Api_ServerResponse(
			$status >= 200 && $status < 300,
            $status,
			empty($data->statusMessage) ? null : $data->statusMessage,
			empty($data->output) ? array() : $data->output->toArray()
		);
	}

	public function addFirma(array $firma)
	{
		return $this->_convertDataObjectIntoServerResponse(
			$this->_backend->addFirma($this->_sessionId, $firma)
		);
	}

	public function delFirma($firmaId)
	{
		return $this->_convertDataObjectIntoServerResponse(
			$this->_backend->delFirma($this->_sessionId, $firmaId)
		);
	}

	public function listFirma()
	{
		return $this->_convertDataObjectIntoServerResponse(
			$this->_backend->listFirma($this->_sessionId)
		);
	}

	public function addReferenceUzivatele(array $reference)
	{
		return $this->_convertDataObjectIntoServerResponse(
			$this->_backend->addReferenceUzivatele($this->_sessionId, $reference)
		);
	}

	public function delReferenceUzivatele($referenceId)
	{
		return $this->_convertDataObjectIntoServerResponse(
			$this->_backend->delReferenceUzivatele($this->_sessionId, $referenceId)
		);
	}

	public function version()
	{
		return $this->_convertDataObjectIntoServerResponse(
			$this->_backend->version()
		);
	}

	public function addAktuality(array $aktualita)
	{
		return $this->_convertDataObjectIntoServerResponse(
			$this->_backend->addAktuality($this->_sessionId, $aktualita)
		);
	}

	public function delAktuality($aktualitaId)
	{
		return $this->_convertDataObjectIntoServerResponse(
			$this->_backend->delAktuality($this->_sessionId, $aktualitaId)
		);
	}

	public function getProjectsList()
	{
		return $this->_convertDataObjectIntoServerResponse(
			$this->_backend->listProject($this->_sessionId)
		);
	}

	/**
	 * Vyexportuje projekt.
	 *
	 * @param array $projectData      Data projektu ve formátu iRest 1.
	 * @param array $userData         Data uživatele (makléře, kterému projekt patří) ve formátu iRest 1.
	 * @param array $listingIds       Pole s IDčky nabídek (ve tvaru id=>kod).
	 * @param array $images           Pole fotografií projektu ve formátu iRest 1.
	 * @param array $additionalParams Údaje které vrací server při exportu a jsou potřeba pro další práci s nabídkou.
	 *                                Data můžou sloužit pro editaci apod...
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru, jestli se export podařil nebo co selhalo.
	 */
	public function addProject(array $projectData, array $userData, array $listingIds = array(),
		array $images = array(), array $additionalParams = array())
	{
		$convertedImages = array();
		foreach ($images as $image) {
			$image['typ_parent'] = 2;
			$image['typ_prilohy'] = 1;
			$image['id_parent'] = $projectData['id'];
			$convertedImages[] = $image;
		}
		$projectData['priloha'] = $convertedImages;

		$response = $this->_backend->addProjekt($this->_sessionId, $projectData);

		if ($response['status'] >= 200 && $response['status'] < 300 && $userData) {

			$userResponse = $this->addUser($userData);

			// Pokud je export úspěšný, tak tam server rve co se nedalo do db... což nemusí uživatelé vidět
			$response['statusMessage'] = '';

			if ($userResponse->wasSuccessful()) {
				$response['status'] = $userResponse->getCode();
				// message nepřikládáme, protože je v ní opět bordel
			}
		}

		return $this->_convertDataObjectIntoServerResponse($response);
	}

	/**
	 * Odstraní projekt ze vzdáleného serveru.
	 *
	 * @param array $projectData      Data projektu ve formátu iRest 1.
	 * @param array $additionalParams Specifická data pro daný export.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function deleteProject(array $projectData, array $additionalParams = array())
	{
		return $this->_convertDataObjectIntoServerResponse(
			$this->_backend->delProjekt($this->_sessionId, $projectData['id'])
		);
	}
}
