<?php
/**
 * Backend pro export na Igatlanbazar.hu.
 *
 * @category   Dalten
 * @package    Export
 * @subpackage Api
 */
class Dalten_Export_Api_Backend_Ingatlanbazar
{

	private $_export;
	private $_entities;
	private $_fileName;
	private $_xmlHead = <<<XMLHEAD
<?xml version="1.0" encoding="UTF-8"?>
<import xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns="http://schemas.ingatlanbazar.hu/Import/" xsi:schemaLocation="http://schemas.ingatlanbazar.hu/Import/ http://schemas.ingatlanbazar.hu/Import/xsd/import.xsd"/>
XMLHEAD;
	private $_xml;
	private $_officeList;
	private $_includedAgents = array();
	private $_adList;
	private $_agentList;
	private $_suburbLookup = array();

	/**
	 * Konstruktor.
	 *
	 * @param Dalten_Export_Ingatlanbazar $export      Odpovídající převodník hodnot.
	 * @param Serenity_Config_Config      $entities    Konfigurace exportu.
	 * @param string                      $fileName    Soubor do kterého to všechno půjde.
	 * @param array                       $suburbTable Předvodní pole pro nemovitost_ctvrt (ve tvaru číslo=>název).
	 */
	public function __construct(
		Dalten_Export_Ingatlanbazar $export, Serenity_Config_Config $entities, $fileName,array $suburbTable
	)
	{
		$this->_export = $export;
		$this->_entities = $entities;
		$this->_fileName = $fileName;
		$this->_suburbLookup=$suburbTable;
	}

	/**
	 * Připraví kostru výsledného XML.
	 *
	 * @param string $login ID uživatele na serveru Ingatlanbazar (IngatlanbazarId).
	 * @param string $email E-mail na který se mají posílat stížnosti od importujícího.
	 */
	public function prepareXml()
	{
		$xml = new Dalten_Xml_SimpleXml($this->_xmlHead);
		$this->_officeList = $xml->addChild('officeList');
		$this->_xml = $xml;
	}

	/**
	 * @param Dalten_Xml_SimpleXml $officeList
	 */
	public function addOffice($login, $email='null@exaple.org'){
		$dataRoot = $this->_officeList->addChild('office');
		$dataRoot->addAttribute('ingatlanbazarId', $login);
		$dataRoot->addChild('information')->addChild('notificationmailList')->addChild('notificationmailText', $email);
		$this->_agentList = $dataRoot->addChild('agentList');
		$this->_adList = $dataRoot->addChild('adList');
	}

	/**
	 * Připraví část XML dle zadaných dat.
	 *
	 * Právě tato funkce balí data do správné struktury tak jak je popsána v konfigu.
	 *
	 * Pokud má položka v konfigu typ int uloží se data do atributu 'option' pokud není
	 * v konfiguráku nastaveno jiné attrName.
	 *
	 * Jinak se ukládá do vnitřku tagu jako text. Avšak pokud je v konfiguráku u položky
	 * přítomna CDATA, uloží se v tagu do CDAT.
	 *
	 * @param Dalten_Xml_SimpleXml $parentXml Xml element který se touto částí naplní
	 * @param string               $partName  Jméno části v konfigu.
	 * @param string               $inputData Vstupní data.
	 */
	public function preparePart(Dalten_Xml_SimpleXml $parentXml, $partName, $inputData)
	{
		$visited = array();
		$thisPart = $this->_entities->__get($partName);

		$convertedData = $this->_export->convertEntityValues($partName, $inputData, $thisPart);

		$currentXml = $parentXml->addChild($partName);

		foreach ($thisPart as $subNode) {
			$subNodeName = $subNode->column;
			if (!in_array($subNodeName, $visited)) {
				if (isset($convertedData[$subNodeName])) {
					$currentValue = $convertedData[$subNodeName];
					if ($subNode->type == "int") {
						$attrName = 'option';
						if (isset($subNode->attrName)) {
							$attrName = $subNode->attrName;
						}
						$currentXml->addChild($subNodeName)->addAttribute($attrName, $currentValue);
					} else {
						if (isset($subNode->CDATA)) {
							$currentXml->addChild($subNodeName)->addCData($currentValue);
						} else {
							$currentXml->addChild($subNodeName, $currentValue);
						}
					}
				}
				$visited[] = $subNodeName;
			}
		}

		return $currentXml;
	}

	/**
	 * Přidá další nemovitost do XML.
	 *
	 * @param array $listingData Údaje o nemovitosti.
	 * @param array $userData    Údaje o makléři.
	 * @param array $images      Pole s obrázky.
	 */
	public function appendAd(array $listingData, array $userData, array $images)
	{
		$adNode = $this->preparePart($this->_adList, 'ad', $listingData);
		$adNode->addAttribute('foreignId', $listingData['id']);
		$adNode->addAttribute('agentId', $userData['id']);

		if (isset($this->_suburbLookup[$listingData['nemovitost_ctvrt']])) {
			$listingData['nemovitost_ctvrt'] = $this->_suburbLookup[$listingData['nemovitost_ctvrt']];
		}

		if (count($images) > 0) {
			$imageList = $adNode->addChild('imageList');
			foreach ($images as $image) {
				$imageList->addChild('image')->addAttribute('href', $image['url']);
			}
		}
	}

	/**
	 * Přidá uživatele do XML.
	 *
	 * @param array $listingData Údaje o nabídce.
	 * @param array $userData    Údaje o uživateli.
	 */
	public function appendAgent(array $listingData, array $userData)
	{
		$userId = $userData['id'];
		if (!in_array($userId, $this->_includedAgents)) {
			$this->_includedAgents[] = $userId;
			$agentNode = $this->preparePart($this->_agentList, 'agent', $userData);
			$agentNode->addAttribute('foreignId', $userId);
		}
	}

	/**
	 * Uloží připravené XML.
	 */
	public function finishXml()
	{
		$this->_xml->asXML($this->_fileName);
	}
}