<?php
/**
 * Export do formátu Českých realit.
 *
 * @category Dalten
 * @package  Export
 */
class Dalten_Export_CeskerealityThree extends Dalten_Export_Ceskereality
{
	/**
	 * Zformátuje a podstrčí některé informace před exportem skrz Newspeak.
	 *
	 * @param array  $values     Pole s daty (POZOR! po zákroku hyperoptimalizátoru).
	 * @param string $config     Definice entity.
	 * @param string $entityName Jméno entity.
	 *
	 * @return array Zformátovaná data.
	 */
	protected function _format(array $values, $config, $entityName)
	{
		if ($entityName == 'listing') {
			if (!empty($values['nemovitost_podlazi_cislo'])) {
				$values['nemovitost_podlazi_cislo'] = $this->_podlaziCisloNaText($values['nemovitost_podlazi_cislo']);
			}
		}
		return $values;
	}

	/**
	 * Provede závěrečné úpravy po převodu Newspeakem.
	 *
	 * @param array  $convertedValues Převedené hodnoty.
	 * @param array  $originalValues  Originální údaje o nemovitosti.
	 * @param string $entityName      Jméno entity.
	 *
	 * @return array Definitivně převedená data entity.
	 */
	protected function _convert(array $convertedValues, array $originalValues, $entityName)
	{
		if ($entityName == 'listing') {
			$convertedValues['subtyp_n'] = $this->_getSubtyp($originalValues);

			$plochy = $this->_tranformAreas($originalValues);
			$videos = $this->_transformVideo($originalValues);

			if (!empty($originalValues['nemovitost_popis_zakladni'])) {
				$convertedValues['popis'] =
					$originalValues['nemovitost_popis_zakladni'] . ' ' . $originalValues['nemovitost_popis_detailni'];
			}

			$convertedValues['podrobnosti'] = $this->_podrobnosti($originalValues);

			if (!empty($originalValues['nemovitost_uir_ulice_kod'])) {
				$convertedValues['ruian_uroven'] = 7;
				$convertedValues['ruian_kod'] = (int)$originalValues['nemovitost_uir_ulice_kod'];
			} elseif (!empty($originalValues['nemovitost_uir_mcast_kod'])) {
				$convertedValues['ruian_uroven'] = 17;
				$convertedValues['ruian_kod'] = (int)$originalValues['nemovitost_uir_mcast_kod'];
			} elseif (!empty($originalValues['nemovitost_uir_cobce_kod'])) {
				$convertedValues['ruian_uroven'] = 5;
				$convertedValues['ruian_kod'] = (int)$originalValues['nemovitost_uir_cobce_kod'];
			} elseif (!empty($originalValues['nemovitost_uir_obec_kod'])) {
				$convertedValues['ruian_uroven'] = 3;
				$convertedValues['ruian_kod'] = (int)$originalValues['nemovitost_uir_obec_kod'];
			} elseif (!empty($originalValues['nemovitost_uir_okres_kod'])) {
				$convertedValues['ruian_uroven'] = 1;
				$convertedValues['ruian_kod'] = (int)$originalValues['nemovitost_uir_okres_kod'];
			}

			$convertedValues = $convertedValues + $plochy + $videos;

			if ($originalValues['nabidka_gps_presnost']>1) {
				$convertedValues['gps_lat'] = 0;
				$convertedValues['gps_lon'] = 0;
			}

			if ($convertedValues['cena']<5) {
			    $convertedValues['cena_neuvadet'] = 1;
            }

		}

		if ($entityName=='project') {
            $kategorie = [];
            if ($originalValues['pocet_bytu']>0) {
                $kategorie[] = '1'; // Byty a bytové domy
            }
            if ($originalValues['pocet_domu']>0) {
                $kategorie[] = '2'; // Rodinné domy
            }
            if ($originalValues['komercni_plochy_pocet']>0 || $originalValues['pocet_kancelari']>0) {
                $kategorie[] = '6'; // Komerční objekty
            }
            if (($originalValues['komercni_plochy_pocet']>0 || $originalValues['pocet_kancelari']>0) && $originalValues['pocet_bytu']>0) {
                $kategorie[] = '3'; // Multifunkční domy
            }
            if (empty($kategorie)) {
                $kategorie[] = '3'; // + ještě jednou Multifunkční domy - záložní kategorie kam padají všechny ostatní projekty
            }
            $convertedValues['kategorie'] = $kategorie;
        }

		if ($entityName=='broker') {
			$jmenoCele = $originalValues['uzivatel_os_jmeno'] . (!empty($originalvalues['uzivatel_os_prostredni_jmeno']) ?  ' ' . $originalvalues['uzivatel_os_prostredni_jmeno'] : '') . ' ' . $originalValues['uzivatel_os_prijmeni'];

			if (!empty($originalValues['uzivatel_tituly_pred_jmenem'])) {
				$jmenoCele = $originalValues['uzivatel_tituly_pred_jmenem'] . ' ' . $jmenoCele;
			}

			if (!empty($originalValues['uzivatel_tituly_za_jmenem'])) {
				$jmenoCele = $jmenoCele . ' ' . $originalValues['uzivatel_tituly_za_jmenem'];
			}

			$convertedValues['makler_jmeno'] = $jmenoCele;
		}

		return $convertedValues;
	}

	protected function _getSubtyp(array $originalValues)
	{
		$typ = $originalValues['nemovitost_typ'];

		$subtyp_n = 599; // ostatní komerční prostory

		if ($typ==4) {
			// byty
			$prevodBytu = array(
				1 => 202,
				2 => 202,
				9 => 203,
				3 => 204,
				10 => 205,
				4 => 206,
				11 => 207,
				5 => 208,
				12 => 209,
				6 => 210,
				13 => 211,
				7 => 211,
				14 => 211,
				8 => 211,
				16 => 211,
				17 => 211,
                18 => 220 // Pokoje
			);
			$subtyp_n = $prevodBytu[$originalValues['nemovitost_dispozice']];
		} elseif ($typ==6) {
			// domy a vily
			$subtyp_n = 101; // neumíme rozlišit RD od vily
		} elseif ($typ==10) {
			// chaty / chalupy
			if ($originalValues['nemovitost_typ_rekreacniho_objektu']==1) {
				$subtyp_n = 108; // chaty
			} else {
				$subtyp_n = 109; // chalupy
			}
		} elseif ($typ==1) {
			// zemědělské objekty
			$subtyp_n = 508;
		} elseif ($typ==3) {
			// pozemky
			$prevodPozemku = array(
				1 => 304,
				2 => 301,
				3 => 303,
				4 => 306,
				5 => 305,
				6 => 302,
				7 => 399,
                8 => 307
			);
			$subtyp_n = $prevodPozemku[$originalValues['nemovitost_druh_pozemku']];
		} elseif ($typ==11) {
			// malé objekty a garáže
			$prevodMalychObjektu = array(
				1 => 901, // garáž
				2 => 599, // ostatní > ostatní komerční prostory
				3 => 906, // garážová stání
				4 => 905, // vinné sklepy
			);
			$subtyp_n = $prevodMalychObjektu[$originalValues['nemovitost_typ_maleho_objektu']];
		} elseif ($typ==9) {
			// komerční prostory
			$prevodKomercniProstory = array(
				1 => 503,
				2 => 504,
				3 => 502,
				4 => 599,
				5 => 501
			);
			$subtyp_n = $prevodKomercniProstory[$originalValues['nemovitost_druh_prostor']];
		} elseif ($typ==2) {
			// komerční objekty
			$prevodKomercniObjekty = array(
				1 => 502,
				2 => 501,
				3 => 504,
				4 => 504,
				5 => 501,
				7 => 503,
				6 => 599
			);
			$subtyp_n = $prevodKomercniObjekty[$originalValues['nemovitost_ucel_budovy']];
		} elseif ($typ==5) {
			// historické objekty
			$subtyp_n = 110;
		} elseif ($typ==7) {
			// hotely, penziony a restaurace
			$prevodHotely = array(
				1 => 506, // hotel
				2 => 506, // penzion
				3 => 505 // restaurace
			);
			$subtyp_n = $prevodHotely[$originalValues['nemovitost_typ_zarizeni']];
		} elseif ($typ==8) {
			// nájemní domy
			$subtyp_n = 105;
		}

		return $subtyp_n;
	}

	/**
	 * Převede naše plochy na plochy Českých realit. Filozofie převodu převzaná z původního exportu.
	 *
	 * @param array $originalValues Pole obsahující originální informace o nemovitosti.
	 *
	 * @return array Pole jejich ploch.
	 */
	private function _tranformAreas(array $originalValues)
	{
		$plocha['1']  = $originalValues['nemovitost_plocha_uzitna']; // zemědělské
		$plocha['2']  = $originalValues['nemovitost_plocha_uzitna']; // komerční objekty
		$plocha['3']  = $originalValues['nemovitost_plocha_celkova']; // pozemky
		$plocha['4']  = $originalValues['nemovitost_plocha_celkova']; // byty
		$plocha['5']  = $originalValues['nemovitost_plocha_celkova']; // historické
		$plocha['6']  = $originalValues['nemovitost_plocha_uzitna']; // domy
		$plocha['7']  = $originalValues['nemovitost_plocha_uzitna']; // hotely
		$plocha['8']  = $originalValues['nemovitost_plocha_uzitna']; // nájemní domy
		$plocha['9']  = $originalValues['nemovitost_plocha_celkova']; // komerční prostory
		$plocha['10'] = $originalValues['nemovitost_plocha_uzitna']; // chaty
		$plocha['11'] = $originalValues['nemovitost_plocha_zastavena']; // malé objekty

		$areas = array();
        $areas['pl_celkova'] = $originalValues['nemovitost_plocha_celkova'];
        // pl_celkova je celková plocha, pokud pro daný typ nabídky neplatí něco jiného:
        if (!empty($plocha[ $originalValues['nemovitost_typ'] ])) {
            $areas['pl_celkova'] = $plocha[ $originalValues['nemovitost_typ'] ];
        }
        if ($originalValues['nemovitost_typ']==9 && empty($areas['pl_celkova'])) {
            // pro komerční prostory posíláme užitnou jako celkovou
            $areas['pl_celkova'] = $originalValues['nemovitost_plocha_uzitna'];
        }
		if ($originalValues['nemovitost_plocha_zastavena']) {
			$areas['pl_zastavena'] = $originalValues['nemovitost_plocha_zastavena'];
		}
		if ($originalValues['nemovitost_plocha_uzitna']) {
			$areas['pl_uzitna'] = $originalValues['nemovitost_plocha_uzitna'];
		}
		if ($originalValues['nemovitost_plocha_parcely']) {
			$areas['pl_pozemku'] = $originalValues['nemovitost_plocha_parcely'];
		}

		// plochy, které jsou povinné v závislosti na typech
		if ($originalValues['nemovitost_typ']==2) {
			$areas['pl_uzitna'] = $originalValues['nemovitost_plocha_celkova'];
		}
		if ($originalValues['nemovitost_typ']==3) {
			$areas['pl_pozemku'] = $originalValues['nemovitost_plocha_celkova'];
		}
		if ($originalValues['nemovitost_typ']==4) {
			$areas['pl_uzitna'] = $originalValues['nemovitost_plocha_uzitna'];
		}
		if ($originalValues['nemovitost_typ']==11) {
			$areas['pl_uzitna'] = $originalValues['nemovitost_plocha_zastavena'];
		}

		return $areas;
	}

    /**
     * Přeloží export virtuální a video prohlídky.
     *
     * @param array $originalValues
     * @return array
     */
    private function _transformVideo(array $originalValues)
    {
        $youtubeId = Dalten_Export_Helper_YoutubeIdExtractor::extractYoutubeId($originalValues['nemovitost_video_prohlidka']);
        $matterportId = $this->_getMatterportId($originalValues['nemovitost_virtualni_prohlidka']);

        $videos = array();
        if (!empty($youtubeId)) {
            $videos['video'] = $youtubeId;
        }
        if (!empty($matterportId)) {
            $videos['prohlidka3d'] = $matterportId;
            $videos['prohlidka3d_typ'] = 1;
        }

        return $videos;
    }

    /**
     * Vytáhne ID virtuální prohlíky Matterport.
     *
     * @param string $url Odkaz na YouTube.
     *
     * @return string
     */
    protected function _getMatterportId($url) {
        // https://my.matterport.com/show/?m=pGSH9wuaKZH
        $parsedUrl = parse_url($url);
        if (isset($parsedUrl['host']) && $parsedUrl['host']=='my.matterport.com' &&  trim($parsedUrl['path'],'/')=='show') {
            $parameters = array();
            parse_str($parsedUrl['query'], $parameters);
            if (!empty($parameters['m'])) {
                return $parameters['m'];
            }
        }
        return '';
    }

	/**
	 * Převede číslo podlaží na text.
	 *
	 * @param int $cisloPodlazi Číslo podlaží
	 *                          (viz http://upload.wikimedia.org/wikipedia/commons/2/24/Podlazi_vs_patro.svg)
	 *
	 * @return null|string Textové vyjádření v patrech.
	 */
	private function _podlaziCisloNaText($cisloPodlazi)
	{
		if ($cisloPodlazi == 1) {
			return 'přízemí';
		} elseif ($cisloPodlazi>1) {
			return ($cisloPodlazi-1) . '. patro';
		} elseif ($cisloPodlazi<0) {
			return abs($cisloPodlazi) . '. podzemní podlaží';
		}
		return null;
	}



	protected function _podrobnosti($originalValues)
	{
		$sablona = array(

			array(				//Dopravní spojení - 0
				"Autobus",
				"Vlak",
				"Metro",
				"MHD",
				"Silnice",
				"Dálnice"
			),

			array(				//Parkování - 1
				"Garáž",
				"Parkování na ulici",
				"Parkoviště",
				"Vlastní parkovací stání"
			),

			array(				//Příjezd - 2
				"Nezpevněný",
				"Rampa",
				"Zpevněný",
				"Beton",
				"Dlažba",
				"Asfalt"
			),

			array(				//Elektrický proud - 3
				"110 V",
				"230 V",
				"400 V",
				"Na hranici pozemku",
				"Vlastní zdroj"
			),

			array(				//Kanalizace - 4
				"Veřejná kanalizace",
				"Vlastní ČOV",
				"Žumpa, septik"
			),

			array(      				//Zavedené inž. sítě - 5
				"Internet",
				"Kabelová TV",
				"Na hranici pozemku",
				"Plyn",
				"Telefon"
			),

			array(				//Zdroj vody - 6
				"Domácí vodárna",
				"Pramen",
				"Studna kopaná",
				"Studna vrtaná",
				"Veřejný vodovod",
				"Místní zdroj"
			),

			array( 				//Způsob vytápění - 7
				"Lokální - plynové",
				"Lokální - tuhá paliva",
				"Lokální - elektrické",
				"Ústřední - plynové",
				"Ústřední - tuhá paliva",
				"Ústřední - Elektrické",
				"Ústřední - Dálkové"
			),

			array(				//Střecha - 8
				"Hambalkový",
				"Kombinovaný",
				"Příhradový",
				"Pultový",
				"Sedlový",
				"Valbový"
			),

			array(				//Typ konstrukce - 9
				"Dřevěná konstrukce",
				"Jiná",
				"Ocelová",
				"Roubená",
				"Skeletová",
				"Montovaná",
				"Zděná",
				"Kamenná",
				"Panelová"
			),

			array(				//WC - 10
				"Splachovací",
				"Společný",
				"Suchý",
			),

			array(       				//Okna - 11
				"Dřevěná",
				"Plastová",
				"Hliníková",
				"Kombinovaná",
			),

			array(				//Balkony, lodžie, terasy - 12
				"Balkón",
				"Lodžie",
				"Terasa"
			),

			array(				//Bazény - 13
				"Venkovní bazén",
				"Vnitřní bazén",
				"Koupací jezírko"
			),

			array(				//Druh bytu - 14
				"Klasický",
				"Mezonet",
				"Loft",
				"Podkroví",
				"Ateliér",
				"Atypický"
			),

			array(				//Vybavení bytu - 15
				"S vybavením",
				"Bez vybavení",
				"Částečně vybaveno"
			),

			array(				//Vybavení podrobnosti - 16
				"Kuchyňská linka",
				"Lednice",
				"Pračka",
				"Sporák",
				"Myčka",
				"Obývací pokoj",
				"Postele"
			),

			array(				//Vytápění podrobnosti - 17
				"Uhlí",
				"Dřevo",
				"Peletky, biomasa",
				"Plyn",
				"El.kotel",
				"El.přímotop",
				"Infra-panely",
				"Čerpadlo s vrtem",
				"Čerpadlo bez vrtu",
				"Solární term panely",
				"Fotovoltaické panely",
				"Rekuperační jednotka",
				"Klimatizace",
				"Krbová vložka",
				"Otevřený krb",
				"Krbová kamna",
				"Radiátory",
				"Podlahové vytápění"

			),

			array(				//Ochrany a omezení - 18
				"Ochranné pásmo",
				"Národní park",
				"CHKO",
				"Historické centrum, památková zona",
				"Památka",
				"Věcné břemeno"
			),

			array(				//Cena nezahrnuje - 19
				"DPH",
				"Poplatky",
				"Právní servis",
				"Provize"
			),

			array(              			//Zateplení - 20
				"Ano",
				"Zateplená polystyrenem",
				"Zateplená vatou",
			),
		);

		$nalezeno = array();

		if ($this->_isMultiChecked($originalValues, 'nemovitost_doprava', 0)) $nalezeno[] = 'Vlak';
		if ($this->_isMultiChecked($originalValues, 'nemovitost_doprava', 1)) $nalezeno[] = 'Dálnice';
		if ($this->_isMultiChecked($originalValues, 'nemovitost_doprava', 2)) $nalezeno[] = 'Silnice';
		if ($this->_isMultiChecked($originalValues, 'nemovitost_doprava', 3)) $nalezeno[] = 'MHD';
		if ($this->_isMultiChecked($originalValues, 'nemovitost_doprava', 4)) $nalezeno[] = 'Autobus';

		// todo Garáž

		if ($this->_isMultiChecked($originalValues, 'nemovitost_elektrina', 0)) $nalezeno[] = '120 V';
		if ($this->_isMultiChecked($originalValues, 'nemovitost_elektrina', 1)) $nalezeno[] = '230 V';
		if ($this->_isMultiChecked($originalValues, 'nemovitost_elektrina', 3)) $nalezeno[] = '400 V';

		if ($this->_isMultiChecked($originalValues, 'nemovitost_odpad', 0)) $nalezeno[] = 'Veřejná kanalizace';
		if ($this->_isMultiChecked($originalValues, 'nemovitost_odpad', 1)) $nalezeno[] = 'Vlastní ČOV';
		if ($this->_isMultiChecked($originalValues, 'nemovitost_odpad', 2)) $nalezeno[] = 'Žumpa, septik';

		if ($this->_isMultiChecked($originalValues, 'nemovitost_telekomunikace', 0)) $nalezeno[] = 'Telefon';
		if ($this->_isMultiChecked($originalValues, 'nemovitost_telekomunikace', 1)) $nalezeno[] = 'Internet';
		if ($this->_isMultiChecked($originalValues, 'nemovitost_ostatni_rozvody', 1)) $nalezeno[] = 'Kabelová TV';
		if ($this->_isMultiChecked($originalValues, 'nemovitost_plyn', 0) || $this->_isMultiChecked($originalValues, 'nemovitost_plyn', 1)) {
			$nalezeno[] = 'Plyn';
		}

		if ($this->_isMultiChecked($originalValues, 'nemovitost_topeni', 0)) $nalezeno[] = 'Lokální - plynové';
		if ($this->_isMultiChecked($originalValues, 'nemovitost_topeni', 1)) $nalezeno[] = 'Lokální - tuhá paliva';
		if ($this->_isMultiChecked($originalValues, 'nemovitost_topeni', 2)) $nalezeno[] = 'Lokální - elektrické';
		if ($this->_isMultiChecked($originalValues, 'nemovitost_topeni', 3)) $nalezeno[] = 'Ústřední - plynové';
		if ($this->_isMultiChecked($originalValues, 'nemovitost_topeni', 4)) $nalezeno[] = 'Ústřední - tuhá paliva';
		if ($this->_isMultiChecked($originalValues, 'nemovitost_topeni', 5)) $nalezeno[] = 'Ústřední - Elektrické';
		if ($this->_isMultiChecked($originalValues, 'nemovitost_topeni', 6)) $nalezeno[] = 'Ústřední - Dálkové';

		if ($this->_isMultiChecked($originalValues, 'nemovitost_druh_objektu', 1)) $nalezeno[] = 'Dřevěná konstrukce';
		if ($this->_isMultiChecked($originalValues, 'nemovitost_druh_objektu', 2)) $nalezeno[] = 'Zděná';
		if ($this->_isMultiChecked($originalValues, 'nemovitost_druh_objektu', 3)) $nalezeno[] = 'Kamenná';
		if ($this->_isMultiChecked($originalValues, 'nemovitost_druh_objektu', 4)) $nalezeno[] = 'Montovaná';
		if ($this->_isMultiChecked($originalValues, 'nemovitost_druh_objektu', 5)) $nalezeno[] = 'Panelová';
		if ($this->_isMultiChecked($originalValues, 'nemovitost_druh_objektu', 6)) $nalezeno[] = 'Skeletová';
		if ($this->_isMultiChecked($originalValues, 'nemovitost_druh_objektu', 7)) $nalezeno[] = 'Jiná';

		if ($this->_isMultiChecked($originalValues, 'nemovitost_voda', 0)) $nalezeno[] = 'Místní zdroj';  // Zdroj pro celý objekt
		if ($this->_isMultiChecked($originalValues, 'nemovitost_voda', 1)) $nalezeno[] = 'Veřejný vodovod'; // Dálkový rozvod

		if ($originalValues['nemovitost_balkon']==1) {
			$nalezeno[] = 'Balkón';
		}
		if ($originalValues['nemovitost_lodzie']==1) {
			$nalezeno[] = 'Lodžie';
		}
		if ($originalValues['nemovitost_terasa']==1) {
			$nalezeno[] = 'Terasa';
		}

		if ($originalValues['nemovitost_garaz']==1) {
			$nalezeno[] = 'Garáž';
		}

		if ($originalValues['pocet_mist_k_parkovani']>0) {
			$nalezeno[] = 'Vlastní parkovací stání';
		}

		$out = array();
		foreach ($sablona as $sekce) {
			$str = str_repeat('0', count($sekce));
			foreach ($sekce as $poloha=>$nazev) {
				if (in_array($nazev, $nalezeno)) {
					$str[$poloha] = '1';
				}
			}
			$out[] = $str;
		}
		return implode(' ', $out);
	}

	/**
	 * Vrací je-li v poli $data zaškrtnuté pole $fieldName jako multicheckbox na hodnotě $value.
	 *
	 * @param array  $data      Pole s daty.
	 * @param string $fieldName Název políčka.
	 * @param int    $value     Pořadí zašktávátka.
	 *
	 * @return bool Je to tak?
	 */
	private function _isMultiChecked(array $data, $fieldName, $value) {
		if (isset($data[$fieldName])) {
			if (substr($data[$fieldName], $value, 1)==1) {
				return true;
			}
		}
		return false;
	}
}
