<?php
/**
 * Export na Expressz.hu
 *
 * @category   Dalten
 * @package    Export
 * @subpackage Api
 */
class Dalten_Export_Api_Expresszhu implements Dalten_Export_Api_CloseableApiInterface,
	Dalten_Export_Api_TransactionApiInterface
{
	/**
	 * @var null|Dalten_Export_Api_Backend_Expresszhu Backend tohoto exportu.
	 */
	private $_backend = null;

	/**
	 * @var Dalten_Xml_SimpleXml XML
	 */
	private $_xml;

	private $_userId;

	private $_index = 1;

	private $_filename = '';

	/**
	 * Konstruktor
	 *
	 * @param Dalten_Export_Api_Backend_Expresszhu $backend  Instance odpovídajícího backendu.
	 * @param string                               $fileName Název souboru kam se export uloží.
	 */
	public function __construct(Dalten_Export_Api_Backend_Expresszhu $backend, $fileName)
	{
		$this->_backend = $backend;
		$this->_filename = $fileName;
	}

	/**
	 * Naváže spojení se vzdáleným serverem.
	 *
	 * @param string $login            Přihlašovací jméno.
	 * @param string $password         Heslo.
	 * @param string $softwareKey      Softwarový klíč.
	 * @param array  $additionalParams Pole dalších parametrů.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function openConnection($login, $password, $softwareKey = '', array $additionalParams = array())
	{
		$this->_userId = $login;
		return new Dalten_Export_Api_ServerResponse(true);
	}

	/**
	 * Vyexportuje nabídku.
	 *
	 * @param array $listingData      Data nabídky ve formátu iRest 1.
	 * @param array $userData         Data uživatele (makléře, kterému nabídka patří) ve formátu iRest 1.
	 * @param array $images           Pole fotografií nabídky ve formátu iRest 1.
	 * @param array $additionalParams Údaje které vrací server při exportu a jsou potřeba pro další práci s
	 *                                nabídkou. Data můžou sloužit pro editaci apod...
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru, jestli se export podařil nebo co selhalo.
	 */
	public function addListing(
		array $listingData, array $userData, array $images = array(), array $additionalParams = array()
	)
	{
		try {
			if ($this->_backend->appendProperty(
				$this->_xml, $listingData, $userData, $images, $this->_userId, $this->_index
			)) {
				$this->_index++;
				return new Dalten_Export_Api_ServerResponse(true);
			} else {
				return new Dalten_Export_Api_ServerResponse(false);
			}
		} catch (Dalten_Export_Exception_HumanReadable $e) {
			return new Dalten_Export_Api_ServerResponse(false, 0, $e->getMessage());
		} catch (Exception $e) {
			return new Dalten_Export_Api_ServerResponse(false);
		}
	}

	/**
	 * Odstraní nabídku ze vzdáleného serveru.
	 * Akce se vždycky povede, ve skutečnosti je to jen nezapsání nabídky.
	 *
	 * @param array $listingData      Data nabídky ve formátu iRest 1.
	 * @param array $additionalParams Specifická data pro daný export.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function deleteListing(array $listingData, array $additionalParams = array())
	{
		return new Dalten_Export_Api_ServerResponse(true);
	}

	/**
	 * Vrátí seznam nabídek, které jsou na vzdáleném serveru.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function getListingsList()
	{
		throw new BadMethodCallException('Not yet implemented!');
	}

	/**
	 * Zavře spojení.
	 *
	 * @return Dalten_Export_Api_ServerResponse
	 */
	public function closeConnection()
	{
		return new Dalten_Export_Api_ServerResponse(true);
	}

	/**
	 * Začíná transakci.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru, jestli se export podařil nebo co selhalo.
	 */
	public function beginTransaction()
	{
		$this->_xml = $this->_backend->prepareXml();
		$this->_index = 1;
		return new Dalten_Export_Api_ServerResponse(true);
	}


	/**
	 * Končí transakci.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru, jestli se export podařil nebo co selhalo.
	 */
	public function commitTransaction()
	{
		try {
			if ($this->_backend->saveToFile($this->_xml, $this->_filename)) {
				return new Dalten_Export_Api_ServerResponse(true);
			} else {
				return new Dalten_Export_Api_ServerResponse(false);
			}
		} catch (Exception $e) {
			return new Dalten_Export_Api_ServerResponse(false);
		}
	}
}
