<?php

/**
 * Exportní API serveru Sreality.
 *
 * @category   Dalten
 * @package    Export
 * @subpackage Api
 */
class Dalten_Export_Api_Toprealitysk implements Dalten_Export_Api_ApiInterface, Dalten_Export_Api_LoggableApiInterface
{
    /**
     * Konvertor hodnot.
     *
     * @var Dalten_Export_Toprealitysk
     */
    private $_export;

    /**
     * Nastavení.
     *
     * @var Serenity_Config_Config
     */
    private $_config;

    /**
     * Backend.
     *
     * @var Dalten_Export_Toprealitysk
     */
    private $_backend;

    /**
     * Konstruktor.
     *
     * @param Dalten_Export_Toprealitysk             $export  Konvertor hodnot.
     * @param Serenity_Config_Config                 $config  Nastavení.
     * @param Dalten_Export_Api_Backend_Toprealitysk $backend Backend.
     */
    public function __construct(Dalten_Export_Toprealitysk $export,
                                Serenity_Config_Config $config,
                                Dalten_Export_Api_Backend_Toprealitysk $backend)
    {
        $this->_export = $export;
        $this->_config = $config;
        $this->_backend = $backend;
    }

    /**
     * Naváže spojení se vzdáleným serverem.
     *
     * @param string $login            Přihlašovací jméno.
     * @param string $password         Heslo.
     * @param string $softwareKey      Softwarový klíč.
     * @param array  $additionalParams Pole dalších parametrů.
     *
     * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
     */
    public function openConnection($login, $password, $softwareKey = '', array $additionalParams = array())
    {
	    if (empty($password)) {
		    return new Dalten_Export_Api_ServerResponse(false, 500, 'Je nutne zadat heslo');
	    }

        $this->_backend->login($login, $password);
        return new Dalten_Export_Api_ServerResponse(true);
    }


    /**
     * Vyexportuje nabídku.
     *
     * @param array $listingData      Data nabídky ve formátu iRest 1.
     * @param array $userData         Data uživatele (makléře, kterému nabídka patří) ve formátu iRest 1.
     * @param array $images           Pole fotografií nabídky ve formátu iRest 1.
     * @param array $additionalParams Pro tento export, nejsou přídavná data potřebná.
     *
     * @return Dalten_Export_Api_ServerResponse Odpověď serveru, jestli se export podařil nebo co selhalo.
     */
    public function addListing(
        array $listingData, array $userData, array $images = array(), array $additionalParams = array()
    )
    {
        if ($listingData['nemovitost_uir_stat'] != 'SK') {
            return new Dalten_Export_Api_ServerResponse(
				false,
				0,
				'Timto exportem nelze exportovat nemovitosti mimo uzemi Slovenska!'
			);
        }


        $cfg = $this->_config->getData();
        $data = $this->_export->convertEntityValues('obj', $listingData, $cfg->obj);
        if (!empty($additionalParams['listing']['hide_price'])) {
            //tato kombinace funguje jako cena dohodou
			$data['price_unit'] = "1";
			$data['price'] = "1";
        }
        $langData = $this->_export->convertEntityValues('langData', $listingData, $cfg->langData);
        $addingUser = $this->_backend->addUser($userData);
        if ($addingUser->wasSuccessful()) {
            $adddingEstate = $this->_backend->addListing($data, $langData, $images);
            return $adddingEstate;
        } else {
            return $addingUser;
        }
    }

    /**
     * Odstraní nabídku ze vzdáleného serveru.
     *
     * @param array $listingData      Data nabídky ve formátu iRest 1.
     * @param array $additionalParams Specifická data pro daný export (zde prázdné pole).
     *
     * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
     */
    public function deleteListing(array $listingData, array $additionalParams = array())
    {
        $id = $listingData['id'];
        return $this->_backend->deleteListing($id);
    }

    /**
     * Implementace interface.
     *
     * @throws BadMethodCallException
     */
    public function getListingsList()
    {
        throw new BadMethodCallException("Method 'getListingsList' IS NOT implemented yet.");
    }

	/**
	 * Vždy uspěje.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function closeConnection(){
		return new Dalten_Export_Api_ServerResponse(true);
	}

    /**
     * Nastaví logger pro backend.
     *
     * Logger bude použit pouze pokud to backend dovoluje.
     *
     * @param Dalten_Export_Api_Backend_Logger_LoggerInterface $logger Instance loggeru.
     *
     * @return bool Podařilo se přiřadit backendu logger?
     */
    public function setLogger(Dalten_Export_Api_Backend_Logger_LoggerInterface $logger)
    {
        if ($this->_backend instanceof Dalten_Export_Api_Backend_LoggableBackendInterface) {
            $this->_backend->setLogger($logger);

            return true;
        }

        return false;
    }

    /**
     * Odstraní nastavený logger pro backend.
     *
     * @return Dalten_Export_Api_Toprealitysk Fluent interface.
     */
    public function removeLogger()
    {
        if ($this->_backend instanceof Dalten_Export_Api_Backend_LoggableBackendInterface) {
            $this->_backend->removeLogger();
        }

        return $this;
    }
}
