<?php
/**
 * Exportní API serveru Reality.sk.
 *
 * @category   Dalten
 * @package    Export
 * @subpackage Api
 */
class Dalten_Export_Api_RealitySk implements Dalten_Export_Api_ApiInterface, Dalten_Export_Api_LoggableApiInterface
{
	/**
	 * Backend ktery se stara o exportovani dat.
	 *
	 * @var \Dalten_Export_Api_Backend_RealitySk|null
	 */
	private $_backend = null;

	/**
	 * Data expotu.
	 *
	 * @var \Dalten_Export_RealitySk|null
	 */
	private $_export = null;

	/**
	 * Konfigurace exportu.
	 *
	 * @var null|\Serenity_Config_Config
	 */
	private $_config = null;
    private string $_userAgentPassword;
    private string $_userAgentId;

    /**
	 * Nastavi vlastnosti objektu.
	 *
	 * @param Dalten_Export_RealitySk             $export  Data exportu.
	 * @param Serenity_Config_Config              $config  Nastaveni exportu.
	 * @param Dalten_Export_Api_Backend_RealitySk $backend Rozhrani pro komunikaci s exportnim serverem.
	 */
	public function __construct(Dalten_Export_RealitySk $export,
		Serenity_Config_Config $config,
		Dalten_Export_Api_Backend_RealitySk $backend)
	{
		$this->_export = $export;
		$this->_config = $config;
		$this->_backend = $backend;
	}

	/**
	 * Naváže spojení se vzdáleným serverem.
	 *
	 * @param string $login            Přihlašovací jméno.
	 * @param string $password         Heslo.
	 * @param string $softwareKey      Softwarový klíč.
	 * @param array  $additionalParams Pole dalších parametrů (musí obsahovat klíč "salt").
	 *
	 * @return \Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function openConnection($login, $password, $softwareKey = '', array $additionalParams = array())
	{
		$this->_userAgentId = $login;
		$this->_userAgentPassword = $password;

		return new \Dalten_Export_Api_ServerResponse(
			true,
			200,
			'Nastavení pro otevření spojení proběhlo v pořádku. (ale zatim nic nespojujeme, když nejsou user data)'
		);
	}

	/**
	 * Vyexportuje nabídku.
	 *
	 * @param array $listingData      Data nabídky ve formátu iRest 1.
	 * @param array $userData         Data uživatele (makléře, kterému nabídka patří) ve formátu iRest 1.
	 * @param array $images           Pole fotografií nabídky ve formátu iRest 1.
	 * @param array $additionalParams Parametry potrebne pro editaci.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru, jestli se export podařil nebo co selhalo.
	 */
	public function addListing(array $listingData, array $userData, array $images = array(), array $additionalParams = array())
	{
		if (empty($userData['realitysk_login'])) {
			new Dalten_Export_Api_ServerResponse(false, 500, 'Makléř nemá nastavené přihlašovací jméno pro exportní službu.');
		}

		$convData = $this->_export->convertEntityValues('listing', $listingData, $this->_config->listing);
		$convData['descriptions'] = $this->_getDescriptions($listingData); // presunout do backendu

		$login = $userData['realitysk_login'];
		$password = $userData['realitysk_pass'];

		// pokud jiz neni dany user zalogovan, udelame tak.
		if (!$this->_backend->isLogged($login)) {
			$this->_backend->login($this->_userAgentId, $this->_userAgentPassword, $login, $password);
		}

		// pokud nemame uniqueIdentifier, je to nova polozka
		if (!isset($additionalParams['result']['uniqueIdentifier'])) {
			$response = $this->_backend->addAdvert($login, $convData, $userData, $images);
		} else {
			$params = isset($additionalParams['result']) ? $additionalParams['result'] : array();
			$response = $this->_backend->editAdvert($login, $convData, $userData, $images, $params);
		}

		return $response;
	}

	/**
	 * Odstraní nabídku ze vzdáleného serveru.
	 *
	 * @param array $listingData      Data nabídky ve formátu iRest 1.
	 * @param array $additionalParams Musi obsahovat klic 'ad-id'.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function deleteListing(array $listingData, array $additionalParams = array())
	{
		if (
			!isset($additionalParams['result']['userData'])
			|| !isset($additionalParams['result']['uniqueIdentifier'])
		)
		{
			new Dalten_Export_Api_ServerResponse(
				false,
				500,
				'Nabídku nelze smazat, protože nemá přiřazené povinné parametry v additionalParams.'
			);
		}
		$login = $additionalParams['result']['userData']['realitysk_login'];
		$password = $additionalParams['result']['userData']['realitysk_pass'];

		if (!$this->_backend->isLogged($login)) {
			$this->_backend->login($this->_userAgentId, $this->_userAgentPassword, $login, $password);
		}

		return $this->_backend->deleteAdvert($login, $additionalParams['result']);
	}

	/**
	 * Implementace interface.
	 *
	 * @throws BadMethodCallException
	 */
	public function getListingsList()
	{
		throw new BadMethodCallException("Method 'getListingsList' IS NOT implemented yet.");
	}

	/**
	 * Vždy uspěje.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function closeConnection(){
		return new Dalten_Export_Api_ServerResponse(true);
	}

	/**
	 * Vrátí lokalizované názvy a popisy nabídky.
	 * Je to zde, protože přes jazyk konfiguráku to moc udělat nejde.
	 *
	 * @param array $listingData pole hodnot nabídky ve formátu iRest 1.
	 *
	 * @return array
	 */
	private function _getDescriptions($listingData)
	{
		// prapodivny zapis - @TODO priprava na dalsi jazyky
		$d = array(
			'language' => strtolower(trim($listingData['nemovitost_uir_stat'])),
			'title' => $listingData['nemovitost_titulek'],
			'description' => (isset($listingData['nemovitost_popis_zakladni']) ? $listingData['nemovitost_popis_zakladni'] . ' ' : '') . $listingData['nemovitost_popis_detailni'],
		);

		return array($d);
	}

	/**
	 * Nastaví logger pro backend.
	 *
	 * Logger bude použit pouze pokud to backend dovoluje.
	 *
	 * @param Dalten_Export_Api_Backend_Logger_LoggerInterface $logger Instance loggeru.
	 *
	 * @return bool Podařilo se přiřadit backendu logger?
	 */
	public function setLogger(Dalten_Export_Api_Backend_Logger_LoggerInterface $logger)
	{
		if ($this->_backend instanceof Dalten_Export_Api_Backend_LoggableBackendInterface) {
			$this->_backend->setLogger($logger);

			return true;
		}

		return false;
	}

	/**
	 * Odstraní nastavený logger pro backend.
	 *
	 * @return Dalten_Export_Api_Sreality Fluent interface.
	 */
	public function removeLogger()
	{
		if ($this->_backend instanceof Dalten_Export_Api_Backend_LoggableBackendInterface) {
			$this->_backend->removeLogger();
		}

		return $this;
	}
}
