<?php

/**
 * Backend API Srealit.
 * Jednotlivé metody by měly odpovídat metodám na vzdáleném serveru.
 * Pro jejich volání je použito XML RPC.
 *
 * @category   Dalten
 * @package    Export
 * @subpackage Api
 */
class Dalten_Export_Api_Backend_Sreality implements Dalten_Export_Api_Backend_LoggableBackendInterface
{
    /**
	 * Adpatér pro volání metod přes XML RPC.
	 *
	 * @var Serenity_Rpc_XmlRpc
	 */
	protected $_adapter;

    /**
	 * Konstruktor.
	 *
	 * @param Serenity_Rpc_XmlRpc $adapter Adpatér pro volání metod přes XML RPC.
	 */
	public function __construct(Serenity_Rpc_XmlRpc $adapter)
	{
		$this->_adapter = $adapter;
	}

    /**
     * Převede odpověď serveru na arrayObject a detekuje přitom odhlášení.
     *
     * @param mixed $response Odpověď serveru.
     * @param bool $recursive Chceme uložit jako objekt rekurzivně?
     * @return Dalten_Data_ArrayObject
     */
    protected function _responseToArrayObject($response, $recursive=true)
    {
        if (!$response || !isset($response['status'])) {
            return new Dalten_Data_ArrayObject(array('status'=>407, 'statusMessage'=>'Během exportu došlo vlivem neznámé chyby k odhlášení.'));
        }
        return new Dalten_Data_ArrayObject($response, $recursive);
    }

	/**
	 * Získání řetězce zvaného session id, který je použit pro hashování hesla.
	 * Metoda slouží k získání řetězce pro hashování hesla.
	 * Po zavolání s existujícím client id je vrácen status 200 a v output je
	 * session id a hashovací klíč. Volá se vždy na začátku relace.
	 *
	 * @param int $clientId ID klienta.
	 *
	 * @return Dalten_Data_ArrayObject Po zavolání se správným clientId je vrácen objekt:
     *                                 <code>
     *                                 {
     *                                     status: 200,
     *                                     statusMessage: '...',
     *
     *                                     output: {
	 *                                         sessionId: '...',
     *                                         hashKey: '...'
     *                                     }
     *                                 }
     *                                 </code>
	 */
	public function getHash($clientId)
	{
        $response = $this->_adapter->getHash((int) $clientId);

		$output = array(null, null);
		if (!empty($response['output']) && is_array($response['output'])) {
			$output = (array) current($response['output']);
		}

		$response['output'] = array(
			'sessionId' => current($output),
			'hashKey' => next($output)
		);

		return new Dalten_Data_ArrayObject($response);
	}

	/**
     * Potvrzení přihlášení.
	 * Metoda login potvrdí session jako autorizovanou. V kladném případě se ověří, že výpočet
	 * nového session id odpovídá očekávání na straně serveru. Metodě login se předávají
	 * parametry session id z metody getHash. Pozor, session id už musí být přepočítáno pomocí vzorečku.
	 * Softwarový klíč je přidělen administrátorem srealit zvlášť pro každého klienta (RK) v závislosti
	 * na použitém exportním software. Nazpět je vrácen status úspěšnosti.
	 *
	 * Poznámka: Bez úspěšně dokončené metody login je celá importní relace ve stavu nepřihlášen
	 * a tedy nelze v exportu zakázek pokračovat.
	 *
	 * Vrací-li metoda opakovaně 407 Bad session je špatně zadané heslo, importní klíč a
	 * nebo client id u klienta nebo na importním serveru. Je třeba zkontrolovat tyto údaje. Heslo
	 * na import si nastavuje klient sám, importní klíč lze měnit přes obchodního zástupce,
	 * clientid bylo firmě přiděleno.
     *
     * @param string $sessionId ID relace.
     *
     * @return Dalten_Data_ArrayObject <code>
     *                                 {
     *                                     status: xxx,
     *                                     statusMessage: '...',
     *
     *                                     output: []
     *                                 }
     *                                 </code>
     */
    public function login($sessionId)
	{
        $response = $this->_adapter->login((string) $sessionId);

		return new Dalten_Data_ArrayObject($response);
	}

	/**
     * Odhlášení.
     *
     * @param string $sessionId ID relace.
     *
     * @return Dalten_Data_ArrayObject <code>
     *                                 {
     *                                     status: xxx,
     *                                     statusMessage: '...',
     *
     *                                     output: []
     *                                 }
     *                                 </code>
     */
    public function logout($sessionId)
    {
        $response = $this->_adapter->logout((string) $sessionId);
        return $this->_responseToArrayObject($response);
    }

	/**
	 * Přidání / editace inzerátu.
	 * Metodě se předává slovník advert data, který v sobě nese veškeré číselníkové i nečíselníkové
	 * údaje o inzerátu. Pokud chybí povinná položka, addAdvert končí s chybou. Pozor, některé položky jsou
	 * povinné v závislosti na typu (kategorii) inzerátu. Pozor, datová struktura advert data nemusí obsahovat
	 * všechny předepsané atributy, jsou vyžadovány pouze povinné atributy. Pokud atribut není ve slovníku
	 * obsažen, server odpovídající položku nechává nedotčenou. Tedy nemění samovolně její hodnotu. Metoda
	 * vrací status a advert id, které je vhodné si uchovat pro další operace s uloženým inzerátem. Pokud
	 * je vyplněn parametr advert rkid, musí být v rámci inzerce konkrétního klienta unikátní. Tento parametr
	 * obsahuje vlastní identifikátor inzerátu a tento identifikátor lze v budoucnu použít pro práci s inzerátem
	 * přes importní rozhraní, kde plně zastupuje advert id.
	 * Editace inzerátu se zajistí vyplněním advert id již uloženého inzerátu, nebo je možno zadat existující
	 * advert rkid. Adresu inzerátu lze vyplnit více způsoby: klasické zadání, UIR-ADR, GPS souřadnice.
	 *
	 * Poznámka: Pokud zadáte cenu 0 nebo 1, bude místo ceny zobrazeno Info o ceně u RK.
	 *
	 * @param string $sessionId  ID relace.
	 * @param array  $advertData Data nabídky.
	 *
	 * @return Dalten_Data_ArrayObject <code>
     *                                 {
     *                                     status: xxx,
     *                                     statusMessage: '...',
     *
     *                                     output: {
	 *                                         advertId: xxx
	 *                                     }
     *                                 }
     *                                 </code>
     */
    public function addAdvert($sessionId, array $advertData)
    {
        if (isset($advertData['ready_date'])) {
			$advertData['ready_date'] = $this->_adapter->setType($advertData['ready_date'], 'datetime');
		}

		if (isset($advertData['energy_performance_attachment'])) {
			$advertData['energy_performance_attachment'] = $this->_adapter->setType($advertData['energy_performance_attachment'], 'base64');
		}

		$response = $this->_adapter->addAdvert((string) $sessionId, $advertData);

		$output = (array) current($response['output']);

		$response['output'] = array(
			'advertId' => current($output)
		);

		return $this->_responseToArrayObject($response);
    }

	/**
	 * Vymazání inzerátu.
	 * Pomocí advert id nebo advert rkid se identifikuje inzerát. Uložené fotografie u inzerátu a statistiky budou
	 * smazány. Pokud inzerát neexistuje bude vrácen status OK (200).
	 *
	 * Poznámka: Inzeráty nejsou smazány ihned po zavolání této metody, aktuálně se drží ještě po
	 * nějakou dobu než se opravdu smažou.
	 *
	 * @param string $sessionId  ID relace.
	 * @param int    $advertId   ID nabídky.
	 * @param string $advertRkId Vlastní ID nabídky.
	 *
	 * @return Dalten_Data_ArrayObject <code>
     *                                 {
     *                                     status: xxx,
     *                                     statusMessage: '...',
     *
     *                                     output: []
     *                                 }
     *                                 </code>
	 */
	public function delAdvert($sessionId, $advertId, $advertRkId)
	{
		$response = $this->_adapter->delAdvert(
			(string) $sessionId,
			(int) $advertId,
			$advertRkId
		);

		return $this->_responseToArrayObject($response);
	}

	/**
	 * Vložení fotografie k již uloženému inzerátu.
	 * Vstupními parametry jsou advert id nebo advert rkid a struktura data:
	 * <code>
	 * array(
	 *		binary data vlastní obrázek
	 *		int main 1=hlavní fotka, 0=ostatní
	 *   	int order nepovinné pořadí v rámci vedlejších fotek
	 *		string alt nepovinný údaj, popisek obrázku
	 *		string photo_id nepovinný údaj, interní id fotky
	 *		string photo_rkid nepovinný údaj, rkid fotky
	 * )
	 * </code>
	 * Pomocí advert id nebo advert rkid se identifikuje inzerát. Výstupem je photo id, které je vhodné si uložit
	 * pro mazání fotky. Pokud je vkládána vedlejší fotografie, a přitom není u inzerátu žádná,
	 * stává se tato automaticky hlavní fotografií. Pokud je vkládána hlavní fotografie, a přitom
	 * u inzerátu již jedna je, stane se vložená fotka fotkou hlavní. Minimální rozlišení fotografie je
	 * 480x360 a maximální velikost souboru 500kB. Po překročení této velikosti server vrací chybu a fotografii
	 * nezpracuje. Je možné obrázek pouze editovat, což znamená, že se nepřeukládá vlastní obrázek,
	 * pouze se editují jeho parametry (hlavní, pořadí, popisek). Pro snadnější orientaci se dá vložit
	 * photo rkid, pomocí kterého se později fotka adresuje.
	 *
	 * @param string $sessionId  ID relace.
	 * @param int    $advertId   ID nabídky.
	 * @param string $advertRkId Vlastní ID nabídky.
	 * @param array  $data       Data fotografie.
	 *
	 * @return Dalten_Data_ArrayObject <code>
     *                                 {
     *                                     status: xxx,
     *                                     statusMessage: '...',
     *
     *                                     output: {
	 *                                         photoId: xxx
	 *                                     }
     *                                 }
     *                                 </code>
	 */
	public function addPhoto($sessionId, $advertId, $advertRkId, array $data)
	{
		$data['data'] = $this->_adapter->setType($data['data'], 'base64');
		$data['main'] = (int) $data['main'];

		if (isset($data['order'])) {
			$data['order'] = (int) $data['order'];
		}
		if (isset($data['alt'])) {
			$data['alt'] = (string) $data['alt'];
		}
		if (isset($data['photo_id'])) {
			$data['photo_id'] = (string) $data['photo_id'];
		}
		if (isset($data['photo_rkid'])) {
			$data['photo_rkid'] = (string) $data['photo_rkid'];
		}

		$response = $this->_adapter->addPhoto(
			(string) $sessionId,
			(int) $advertId,
			(string) $advertRkId,
			$data
		);
        if (!empty($response['output']) && is_array($response['output'])) {
            $output = (array) current($response['output']);

            $response['output'] = array(
                'photoId' => current($output)
            );
        }

		return $this->_responseToArrayObject($response);
	}

    /**
	 * Vymazání fotografie.
	 * Pomocí photo id nebo photo rkid se identifikuje fotografie. Pokud je mazána hlavní fotografie,
	 * hlavní se automaticky stane první vedlejší. Pokud fotografie neexistuje bude vracen status OK (200).
	 *
	 * @param string $sessionId ID relace.
	 * @param int    $photoId   ID fotografie.
	 * @param string $photoRkId Vlastní ID fotografie.
	 *
	 * @return Dalten_Data_ArrayObject <code>
     *                                 {
     *                                     status: xxx,
     *                                     statusMessage: '...',
     *
     *                                     output: []
     *                                 }
     *                                 </code>
	 */
	public function delPhoto($sessionId, $photoId, $photoRkId)
	{
	    $response = $this->_adapter->delPhoto((string) $sessionId, (int) $photoId, (string) $photoRkId);
		return $this->_responseToArrayObject($response);
	}

	/**
	 * Výpis fotografií existujícího inzerátu.
	 *
	 * Pomocí advert id nebo advert rkid se identifikuje inzerát. V output je vráceno pole, kde každý prvek pole
	 * obsahuje strukturu (viz níže). Fotografie jsou seřazeny podle pořadí vložení.
	 *
	 * @param string $sessionId  ID relace.
	 * @param int    $advertId   ID nabídky.
	 * @param string $advertRkId Vlastní ID nabídky.
	 *
	 * @return Dalten_Data_ArrayObject <code>
     *                                 {
     *                                     status: xxx,
     *                                     statusMessage: '...',
     *
     *                                     output: [
	 * 										           {
	 * 										               photo_id: int,
	 * 										               proto_rkid: string,
	 * 										               main: int (1,0)
	 * 										           }
	 *                                             ]
     *                                 }
     *                                 </code>
	 */
	public function listPhoto($sessionId, $advertId, $advertRkId)
	{
	    $response = $this->_adapter->listPhoto((string) $sessionId, (int) $advertId, (string) $advertRkId);
		return $this->_responseToArrayObject($response);
	}

	/**
	 * Přidání nového makléře.
	 * Aby makléř byl vložen, je potřeba vyplnit povinné položky v rámci struktury client data.
	 * <code>
	 * array(
	 *		string client_login emailový účet, kterým se dá přihlásit na seznam.cz
	 *		string client_domain nepovinný údaj, doména pro login, lze ji uvést přímo do client_login
	 *		string client_name jméno makléře
	 *		string contact_gsm mobil makléře
	 *		string contact_gsm_code nepovinný údaj, mezinárodní předvolba makléře
     *      int    client_ic IČO makléře
     *      int    client_is_employee Je makléř zaměstnanec?
	 *		string contact_phone nepovinný údaj, telefon
	 *		string contact_phone_code nepovinný údaj, mezinárodní předvolba
	 *		string contact_email nepovinný údaj, kontaktní email se může lišit
	 *		string makler_note nepovinný údaj, poznámka
	 *		binary photo nepovinný údaj, obrázek makléře
	 * )
	 * </code>
	 * Pomocí seller id nebo seller rkid se identifikuje makléř. Při vkládaní inzerátu, pokud má vlastního
	 * makléře, je tedy na výběr mezi seller rkid a seller id. Ostatní položky ze struktury client data jako
	 * contact phone (telefon), makler note (poznámka makléře) a photo (fotografie makléře) jsou nepovinné.
	 * U korektně vloženého makléře je získáno seller id číslo.
	 *
	 * @param string $sessionId  ID relace.
	 * @param string $sellerId   ID makléře.
	 * @param string $sellerRkId Vlastní ID makléře.
	 * @param array  $clientData Data makléře.
	 *
	 * @return Dalten_Data_ArrayObject <code>
     *                                 {
     *                                     status: xxx,
     *                                     statusMessage: '...',
     *
     *                                     output: {
	 *                                         sellerId: xxx
	 *                                     }
     *                                 }
     *                                 </code>
	 */
	public function addSeller($sessionId, $sellerId, $sellerRkId, array $clientData)
	{
	    $ic = isset($clientData['client_ic']) ? (int) $clientData['client_ic'] : null;
	    $isEmployee = isset($clientData['client_is_employee']) ? (int) $clientData['client_is_employee'] : null;
		$clientData = array_map('strval', $clientData);
		if (isset($clientData['photo'])) {
			$clientData['photo'] = $this->_adapter->setType($clientData['photo'], 'base64');
		}
		// vrátíme zpátky int hodnoty (tak aby neprošly přes strval výše)
		unset($clientData['client_ic'], $clientData['client_is_employee']);
		if (null !== $ic && !$isEmployee) {
		    $clientData['client_ic'] = $ic;
        }
		if (null !== $isEmployee) {
		    $clientData['client_is_employee'] = $isEmployee;
        }

		$response = $this->_adapter->addSeller((string) $sessionId, (int) $sellerId, (string) $sellerRkId, $clientData);

		if (isset($response['output']) && $response['output']) {
            $output = (array) current($response['output']);

            $response['output'] = array('sellerId' => current($output));
        }

		return $this->_responseToArrayObject($response);
	}

	/**
	 * Odebrání existujího makléře.
	 *
	 * @param string $sessionId  ID relace.
	 * @param string $sellerId   ID makléře.
	 * @param string $sellerRkId Vlastní ID makléře.
	 *
	 * @return Dalten_Data_ArrayObject
	 */
	public function delSeller($sessionId, $sellerId, $sellerRkId)
	{
		$response = $this->_adapter->delSeller((string) $sessionId, (int) $sellerId, (string) $sellerRkId);
		return $this->_responseToArrayObject($response);
	}

	/**
	 * Výpis makléřů.
	 *
	 * V output je vráceno pole, kde každý prvek obsahuje strukturu (viz níže). Makléři jsou seřazeni podle pořadí vložení.
	 *
	 * @param string $sessionId ID relace.
	 *
	 * @return Dalten_Data_ArrayObject
	 */
	public function listSeller($sessionId)
	{
		$response = $this->_adapter->listSeller((string) $sessionId);
		return $this->_responseToArrayObject($response, false);
	}

	/**
	 * Výpis statistik inzerátu/inzerátů.
	 *
	 * @param string $sessionId  ID relace.
	 * @param array  $advertId   Pole ID nabídek.
	 * @param array  $advertRkId Pole vlastních ID nabídek.
	 *
	 * @return Dalten_Data_ArrayObject Vrácené XML.
	 */
	public function listStat($sessionId, array $advertId, array $advertRkId)
	{
		$response = $this->_adapter->listStat((string) $sessionId, $advertId, $advertRkId);
		return $this->_responseToArrayObject($response, false);
	}

	/**
	 * Výpis denní statistiky inzerátu.
	 *
	 * @param string $sessionId  ID relace.
	 * @param int    $advertId   ID nabídky.
	 * @param string $advertRkId Vlastní ID nabídky.
	 *
	 * @return Dalten_Data_ArrayObject
	 */
	public function listDailyStat($sessionId, $advertId, $advertRkId)
	{
		$response = $this->_adapter->listDailyStat((string) $sessionId, (int) $advertId, (string) $advertRkId);
		return $this->_responseToArrayObject($response, false);
	}

	/**
	 * Výpis statistiky všech inzerátů daného klienta za konkrétní den určený parametrem date.
	 *
	 * @param string $sessionId ID relace.
	 * @param string $date      Datum, kterého se statistika týká.
	 *
	 * @return Dalten_Data_ArrayObject
	 */
	public function listAllDailyStat($sessionId, $date)
	{
		$date = $this->_adapter->setType($date, 'datetime');
		$response = $this->_adapter->listAllDailyStat((string) $sessionId, $date);
		return $this->_responseToArrayObject($response, false);
	}

	/**
	 * Výpis všech inzerátů přihlášené realitní kanceláře.
	 *
	 * @param string $sessionId ID relace.
	 *
	 * @return Dalten_Data_ArrayObject
	 */
	public function listAdvert($sessionId)
	{
		$response = $this->_adapter->listAdvert((string) $sessionId);
		return $this->_responseToArrayObject($response, false);
	}

	/**
	 * Nastaví zvýhodnění inzerátu pro daný den.
	 *
	 * @param string $sessionId  ID relace.
	 * @param int    $advertId   ID nabídky.
	 * @param string $advertRkId Vlastní ID nabídky.
	 *
	 * @return Dalten_Data_ArrayObject
	 */
	public function topAdvert($sessionId, $advertId, $advertRkId)
	{
		$response = $this->_adapter->topAdvert(
			(string) $sessionId,
			(int) $advertId,
			$advertRkId
		);
		return $this->_responseToArrayObject($response);
	}

	/**
	 * Přidání nového developerského projektu.
	 *
	 * Metodě se předává slovník project_data, který v sobě nese veškeré číselníkové i nečíselníkové
	 * údaje o projektu.
	 * Tyto atributy naleznete v tabulce 2 na straně 22. Pokud chybí povinná položka, addProject končí s chybou.
	 *
	 * @param int   $sessionId   ID relace.
	 * @param array $projectData Naformátovaná data projektu.
	 *
	 * @return Dalten_Data_ArrayObject Odpověď serveru.
	 */
	public function addProject($sessionId, array $projectData)
	{
		if (isset($projectData['date_construct_completion'])) {
			$projectData['date_construct_completion'] = $this->_adapter->setType(
				$projectData['date_construct_completion'], 'datetime'
			);
		}

		if (isset($projectData['date_move'])) {
			$projectData['date_move'] = $this->_adapter->setType($projectData['date_move'], 'datetime');
		}

		if (isset($projectData['date_sale'])) {
			$projectData['date_sale'] = $this->_adapter->setType($projectData['date_sale'], 'datetime');
		}

		$response = $this->_adapter->addProject((string) $sessionId, $projectData);

		$output = (array) current($response['output']);

		$response['output'] = array(
			'projectId' => current($output)
		);

		return $this->_responseToArrayObject($response);
	}

	/**
	 * Odebrání existujícího developerského projektu.
	 *
	 * Pomocí project_id nebo project_rkid se identiﬁkuje projekt (viz kapitola 1.6), který má být smazán.
	 * Uložené fotograﬁe u projektu budou smazány.
	 *
	 * @param int    $sessionId  ID relace.
	 * @param int    $advertId   ID projektu.
	 * @param string $advertRkId Vlastní ID projektu.
	 *
	 * @return Dalten_Data_ArrayObject Odpověď serveru.
	 */
	public function delProject($sessionId, $advertId, $advertRkId)
	{
		$response = $this->_adapter->delProject(
			(string) $sessionId,
			(int) $advertId,
			(string) $advertRkId
		);

		return $this->_responseToArrayObject($response);
	}

	/**
	 * Vyexportuje fotografii projektu.
	 *
	 * @param string $sessionId   ID relace.
	 * @param int    $projectId   ID projektu.
	 * @param string $projectRkId Vlastní ID projektu.
	 * @param array  $data        Pole s informacemi o fotografii.
	 *
	 * @return Dalten_Data_ArrayObject Odpověď serveru.
	 */
	public function addProjectPhoto($sessionId, $projectId, $projectRkId, array $data)
	{
		$data['data'] = $this->_adapter->setType($data['data'], 'base64');
		$data['main'] = (int) $data['main'];

		if (isset($data['order'])) {
			$data['order'] = (int) $data['order'];
		}
		if (isset($data['alt'])) {
			$data['alt'] = (string) $data['alt'];
		}
		if (isset($data['photo_id'])) {
			$data['photo_id'] = (string) $data['photo_id'];
		}
		if (isset($data['photo_rkid'])) {
			$data['photo_rkid'] = (string) $data['photo_rkid'];
		}

		$response = $this->_adapter->addProjectPhoto(
			(string) $sessionId,
			(int) $projectId,
			(string) $projectRkId,
			$data
		);
		$output = (array) current($response['output']);

		$response['output'] = array(
			'photoId' => current($output)
		);

		return $this->_responseToArrayObject($response);
	}

	/**
	 * Vrátí pole s informacemi o přítomných fotkách projektu.
	 *
	 * @param string $sessionId   ID relace.
	 * @param int    $projectId   ID projektu.
	 * @param string $projectRkId Vlastní ID projektu.
	 *
	 * @return Dalten_Data_ArrayObject Odpověď serveru.
	 */
	public function listProjectPhoto($sessionId, $projectId, $projectRkId)
	{
		$response = $this->_adapter->listProjectPhoto((string) $sessionId, (int) $projectId, (string) $projectRkId);
		return $this->_responseToArrayObject($response);
	}

	/**
	 * Smaže fotografii od projektu.
	 *
	 * @param string $sessionId ID relace.
	 * @param int    $photoId   ID fotografie.
	 * @param string $photoRkId Vlastní ID fotografie.
	 *
	 * @return Dalten_Data_ArrayObject Odpověď serveru.
	 */
	public function delProjectPhoto($sessionId, $photoId, $photoRkId)
	{
		$response = $this->_adapter->delProjectPhoto((string) $sessionId, (int) $photoId, (string) $photoRkId);
		return $this->_responseToArrayObject($response);
	}

    /**
     * Vložení videoprohlíkdy k inzerátu.
     *
     * @param string $sessionId  ID relace.
     * @param int    $advertId   ID nabídky.
     * @param string $advertRkId Vlastní ID nabídky.
     * @param array  $data       Vlastní ID nabídky.
     *
     * @return Dalten_Data_ArrayObject Odpověď serveru.
     */
    public function addVideo($sessionId, $advertId, $advertRkId, array $data)
    {
        $data['video_data'] = $this->_adapter->setType($data['video_data'], 'base64');
        $response = $this->_adapter->addVideo((string) $sessionId, (int) $advertId, (string) $advertRkId, $data);
        return $this->_responseToArrayObject($response);
    }

    /**
     * Smazaní videoprohlíkdy inzerátu.
     *
     * @param string $sessionId  ID relace.
     * @param int    $advertId   ID nabídky.
     * @param string $advertRkId Vlastní ID nabídky.
     *
     * @return Dalten_Data_ArrayObject Odpověď serveru.
     */
    public function delVideo($sessionId, $advertId, $advertRkId, $videoId=null)
    {
        $response = $this->_adapter->delVideo((string) $sessionId, (int) $advertId, (string) $advertRkId, $videoId);
        return $this->_responseToArrayObject($response);
    }

	/**
	 * Nastaví logger pro backend.
	 *
	 * @param \Dalten_Export_Api_Backend_Logger_LoggerInterface $logger Instance loggeru.
	 *
	 * @return Dalten_Export_Api_Backend_Sreality Fluent interface.
	 */
	public function setLogger(Dalten_Export_Api_Backend_Logger_LoggerInterface $logger)
	{
		if ($this->_adapter instanceof Dalten_Export_Api_Backend_LoggableBackendInterface) {
			$this->_adapter->setLogger($logger);
		}

		return $this;
	}

	/**
	 * Odstraní nastavený logger pro backend.
	 *
	 * @return Dalten_Export_Api_Backend_Sreality Fluent interface.
	 */
	public function removeLogger()
	{
		if ($this->_adapter instanceof Dalten_Export_Api_Backend_LoggableBackendInterface) {
			$this->_adapter->removeLogger();
		}

		return $this;
	}

    /**
     * Výpis celých zpráv, ze všech inzerátu, odeslaných na RK v zadané datum.
     *
     * @param string $sessionId ID relace.
     * @param string $date      Datum, kterého se statistika týká.
     * @return Dalten_Data_ArrayObject Odpověď serveru.
     */
    public function listFullInquiry($sessionId, $date)
    {
        $date = $this->_adapter->setType($date, 'datetime');
        $response = $this->_adapter->listFullInquiry((string) $sessionId, $date);
        return $this->_responseToArrayObject($response, false);
    }

    /**
     * Získání informací o (jedné) videoprohlídce existujícího inzerátu.
     *
     * @param string $sessionId  ID relace.
     * @param int    $advertId   ID nabídky.
     * @param string $advertRkId Vlastní ID nabídky.
     *
     * @return Dalten_Data_ArrayObject
     */
    public function listVideo($sessionId, $advertId, $advertRkId)
    {
        $response = $this->_adapter->listVideo(
            (string) $sessionId,
            (int) $advertId,
            $advertRkId
        );
        $response['output'] = current($response['output']);
        return $this->_responseToArrayObject($response, false);
    }

    /**
     * Získání informací o videoprohlídkách existujícího inzerátu.
     *
     * @param string $sessionId  ID relace.
     * @param int    $advertId   ID nabídky.
     * @param string $advertRkId Vlastní ID nabídky.
     *
     * @return Dalten_Data_ArrayObject
     */
    public function listVideos($sessionId, $advertId, $advertRkId)
    {
        $response = $this->_adapter->listVideos(
            (string) $sessionId,
            (int) $advertId,
            $advertRkId
        );
        return $this->_responseToArrayObject($response, false);
    }
}
