<?php

/**
 * Export do formátu reality.idnes.
 *
 * @category   Dalten
 * @package    Export
 */
class Dalten_Export_Idnes extends Dalten_Export_AbstractExport
{
	protected $_additionalParams = array();

	public function setAdditionalParams($additionalParams)
	{
		$this->_additionalParams = $additionalParams;
	}

	/**
	 * Umožňuje exportu upravit položky PO jejich převedením pomocí processeru..
	 *
	 * @param array  $convertedValues Pole převedených hodnot entity.
	 * @param array  $originalValues  Pole původních (nepřevedených) hodnot entity.
	 * @param string $entityName      Název entity.
	 *
	 * @return array Výsledné pole hodnot.
	 */
	protected function _convert(array $convertedValues, array $originalValues, $entityName)
	{
		$convertedValues = $this->_addMulBalkonField($originalValues, $convertedValues);
		$convertedValues = $this->_addMulParkovani($originalValues, $convertedValues);

		$convertedValues['vybavenost'] = $this->_vybavenost($originalValues);
		$convertedValues['dostupnost'] = $this->_dostupnost($originalValues);

		$podlazi = $this->_osetriPodlazi($originalValues['nemovitost_podlazi_cislo']);
		if ($podlazi) {
			$convertedValues['f_podlazi_nr'] = $podlazi;
		}

		if (in_array($originalValues['nemovitost_typ'], array(2, 4))
			&& empty($originalValues['nemovitost_plocha_uzitna'])) {
			$convertedValues['pl_uzitna'] = $originalValues['nemovitost_plocha_celkova'];
		}

		if ($originalValues['nemovitost_typ'] == 9 && empty($originalValues['nemovitost_plocha_uzitna'])) {
			$convertedValues['pl_uzitna'] = $originalValues['nemovitost_plocha_kancelari'];
		}

		if ($originalValues['nemovitost_typ'] == 11 && empty($originalValues['nemovitost_plocha_uzitna'])) {
			$convertedValues['pl_uzitna'] = $originalValues['nemovitost_plocha_zastavena'];
		}

		if ($originalValues['nemovitost_typ'] == 3 && empty($originalValues['nemovitost_plocha_parcely'])) {
			$convertedValues['pl_parcela'] = $originalValues['nemovitost_plocha_celkova'];
		}

		if (empty($convertedValues['pl_balkon']) && !empty($originalValues['nemovitost_lodzie']) && !empty($originalValues['nemovitost_lodzie_plocha'])) {
			// pokud není vyplněn balkon, ale je vyplněna lodžie
			// posíláme lodžii jako balkon
			$convertedValues['pl_balkon'] = $originalValues['nemovitost_lodzie_plocha'];
		}

		return $convertedValues;
	}

	/**
	 * Přidá mezi hodnoty informaci o balkonu, lodžii či terase.
	 *
	 * @param array $originalValues  Původní (nepřevedené hodnoty).
	 * @param array $convertedValues Převedené hodnoty.
	 *
	 * @return array Převedené hodnoty s doplněným klíčem mul_balkon (za předpokladu, že je čím ho naplnit).
	 */
	private function _addMulBalkonField($originalValues, $convertedValues)
	{
		static $map = array(
			'nemovitost_balkon' => 1,
			'nemovitost_lodzie' => 2,
			'nemovitost_terasa' => 3
		);
		foreach ($map as $key => $value) {
			if (isset($originalValues[$key]) && $originalValues[$key] == 1) {
				$convertedValues['mul_balkon'][] = $value;
			}
		}

		return $convertedValues;
	}

	/**
	 * Přidá parkování.
	 *
	 * @param array $originalValues  Původní (nepřevedené hodnoty).
	 * @param array $convertedValues Převedené hodnoty.
	 *
	 * @return array Převedené hodnoty s doplněným klíčem mul_parkovani (za předpokladu, že je čím ho naplnit).
	 */
	private function _addMulParkovani($originalValues, $convertedValues)
	{
		$parkovani = array();
		// garáž
		if ($originalValues['nemovitost_garaz']==1) {
			$parkovani[] = 1;
		}
		if ($this->_isMultiChecked($originalValues, 'nemovitost_dalsi_nebytove_prostory', 1) || $this->_isMultiChecked($originalValues, 'nemovitost_dalsi_nebytove_prostory', 1)) {
			$parkovani[] = 1;
		}
		if ($this->_isMultiChecked($originalValues, 'nemovitost_ostatni', 3)) {
			$parkovani[] = 1;
		}

		$convertedValues['mul_parkovani'] = array_unique($parkovani);

		return $convertedValues;
	}

	/**
	 * Vypíše vybavenost textově.
	 *
	 * @param array $originalValues
	 * @return string
	 */
	private function _vybavenost($originalValues)
	{
		$vybavenost = array();
		if ($this->_isMultiChecked($originalValues, 'nemovitost_obcanska_vybavenost', 0)) $vybavenost[] = 'škola';
		if ($this->_isMultiChecked($originalValues, 'nemovitost_obcanska_vybavenost', 1)) $vybavenost[] = 'školka';
		if ($this->_isMultiChecked($originalValues, 'nemovitost_obcanska_vybavenost', 2)) $vybavenost[] = 'zdravotnická zařízení';
		if ($this->_isMultiChecked($originalValues, 'nemovitost_obcanska_vybavenost', 3)) $vybavenost[] = 'pošta';
		if ($this->_isMultiChecked($originalValues, 'nemovitost_obcanska_vybavenost', 4)) $vybavenost[] = 'supermarket';
		if ($this->_isMultiChecked($originalValues, 'nemovitost_obcanska_vybavenost', 5)) $vybavenost[] = 'kompletní síť obchodů a služeb';

		return implode(', ', $vybavenost);
	}

	/**
	 * Vypíše dostupnost textově.
	 *
	 * @param array $originalValues
	 * @return string
	 */
	private function _dostupnost($originalValues)
	{
		$dostupnost = array();
		if ($this->_isMultiChecked($originalValues, 'nemovitost_doprava', 0)) $dostupnost[] = 'vlak';
		if ($this->_isMultiChecked($originalValues, 'nemovitost_doprava', 1)) $dostupnost[] = 'dálnice';
		if ($this->_isMultiChecked($originalValues, 'nemovitost_doprava', 2)) $dostupnost[] = 'silnice';
		if ($this->_isMultiChecked($originalValues, 'nemovitost_doprava', 3)) $dostupnost[] = 'MHD';
		if ($this->_isMultiChecked($originalValues, 'nemovitost_doprava', 4)) $dostupnost[] = 'autobus';

		return implode(', ', $dostupnost);
	}

	/**
	 * Převede číslo podlaží z Irestu do číselníku používaného Idnesem.
	 *
	 * @param int $cislo_podlazi Číslo podlaží jak jde z Irestu.
	 *
	 * @return int Číslo podlaží tak jak ho bere Idnes.
	 */
	private function _osetriPodlazi($cislo_podlazi)
	{
		if ($cislo_podlazi>1) {
			// nad přízemím:
			if ($cislo_podlazi>21) {
				// číslo podlaží je víc jak 21
				return 999;
			} else {
				// pro nás je přízemí==1, pro Idnes ne
				return $cislo_podlazi-1;
			}
		} elseif ($cislo_podlazi==1) {
			// přízemí:
			return 1000;
		} elseif ($cislo_podlazi==-1) {
			// suterén:
			return 1002;
		} else {
			// cokoliv pod suterénem:
			return $cislo_podlazi;
		}
	}

	/**
	 * Vrací je-li v poli $data zaškrtnuté pole $fieldName jako multicheckbox na hodnotě $value.
	 *
	 * @param array  $data      Pole s daty.
	 * @param string $fieldName Název políčka.
	 * @param int    $value     Pořadí zašktávátka.
	 *
	 * @return bool Je to tak?
	 */
	private function _isMultiChecked(array $data, $fieldName, $value) {
		if (isset($data[$fieldName])) {
			if (substr($data[$fieldName], $value, 1)==1) {
				return true;
			}
		}
		return false;
	}
}
