<?php
/**
 * Převaděč do formátu Ingatlanbazar.hu
 *
 * @category Dalten
 * @package  Export
 */
class Dalten_Export_Ingatlanbazar extends Dalten_Export_AbstractExport
{
	const TYP_ZEMEDELSKE_OBJEKTY = 1;
	const TYP_KOMERCNI_OBJEKTY = 2;
	const TYP_POZEMKY = 3;
	const TYP_BYTY = 4;
	const TYP_HISTORICKE_OBJEKTY = 5;
	const TYP_DOMY_A_VILY = 6;
	const TYP_HOTELY = 7;
	const TYP_NAJEMNI_DOMY = 8;
	const TYP_KOMERCNI_PROSTORY = 9;
	const TYP_CHATY = 10;
	const TYP_MALE_OBJEKTY = 11;

	const DRUH_OBJEKTU_DREVO = 1;
	const DRUH_OBJEKTU_CIHLA = 2;
	const DRUH_OBJEKTU_KAMEN = 3;
	const DRUH_OBJEKTU_MONTOVANY = 4;
	const DRUH_OBJEKTU_PANEL = 5;
	const DRUH_OBJEKTU_SKELETOVY = 6;
	const DRUH_OBJEKTU_SMISENY = 7;

	const PODTYP_SKLAD = 1;
	const PODTYP_OBCHODNI_CENTRUM = 3;
	const PODTYP_NAKUPNI_STREDISKO = 4;
	const PODTYP_ADMINISTRATIVNI_BUDOVA = 7;

	const POLOHA_RADOVY=1;
	const POLOHA_ROHOVY=2;
	const POLOHA_V_BLOKU=3;
	const POLOHA_SAMOSTATNY=4;

	const PROSTORY_KANCELARE = 1;
	const PROSTORY_OBCHODNI = 2;
	const PROSTORY_SKLADOVACI = 3;
	const PROSTORY_JINE = 4;
	const PROSTORY_VYROBNI = 5;


	const CIL_ZEMEDELSKY = 9;
	const CIL_POZEMEK = 6;
	const CIL_BYT_CIHLA = 3;
	const CIL_BYT_PANEL = 4;
	const CIL_BYT_JINY = 18;
	const CIL_JINE_KOMERCNI = 24;
	const CIL_SKLAD = 26;
	const CIL_KANCELAR = 20;
	const CIL_OBCHODNI_DUM = 21;
	const CIL_CSALADI_HAZ = 10;
	const CIL_IKERHAZ = 11;
	const CIL_SORHAZ = 13;
	const CIL_GARAZ = 5;
	const CIL_CHATA = 2;
	const CIL_SIDLO = 23;
	const CIL_DILNA = 25;


	private $_extras = array(
		'nemovitost_balkon'=>'erkély',
		'nemovitost_terasa'=>'terasz',
		'nemovitost_sklep'=>'pince',
		'nemovitost_garaz'=>'garázs',
		'nemovitost_ostatni_rozvody'=>array(
			0=>'műholdvevő',
			1=>'kábel tv'
		),
		'nemovitost_obcanska_vybavenost'=>array(
			0=>'iskola a közelben',
			5=>'boltok a közelben'
		),
		'nemovitost_plyn'=>'gáz',
		'nemovitost_ostatni'=>array(
			4=>'lift',
			6=>'riasztó'
		)
	);

	/**
	 * Predpripravi hodnoty na zpracovani v newspeaku (INI souboru).
	 *
	 * @param array|mixed $values     Hodnoty
	 * @param string      $config     Konfigurace
	 * @param string      $entityName Nazev entity
	 *
	 * @return array|mixed
	 */
	function _format(array $values, $config, $entityName)
	{
		foreach ($values as $k => $v) {
			if (isset($config->$k)) {
				if (isset($config->$k->solvedInPHP)) {
					switch ($config->$k->solvedInPHP) {
						case 'first checked':
							$pos = strpos($v, '1');
							$values[$k] = ($pos === false ? -1 : $pos);
							break;
						default:
							break;
					}
				}
			}
		}
		return $values;
	}

	/**
	 * Dodělá převadění pole hodnot načaté v konfiguráku.
	 *
	 * Konkrétně v tomto exportu tato funkce ,,hádá" počet místností bytu pokud není výslovně uveden.
	 *
	 * Další funkce je oprava Budapešti. Pokud je kraj Budapešť, bude jako město uveden jeho okrsek.
	 *
	 * Zatím poslední je ošetření panelového bytu. Pokud byt není panelový, považuje se automaticky
	 * za cihlový.
	 *
	 * @param array  $convertedValues Hodnoty převedené pomocí konfiguráku.
	 * @param array  $originalValues  Originální hodnoty.
	 * @param string $entityName      Jméno právě převáděné entity.
	 *
	 * @return array Převedené pole.
	 */
	function _convert($convertedValues, $originalValues, $entityName)
	{
		if ($entityName == 'ad') {
			if ($originalValues['nemovitost_mena']==2) {
				throw new Dalten_Export_Exception_HumanReadable('Hiba: USD is invalid currency!');
			}
			if ($originalValues['nemovitost_typ'] == self::TYP_BYTY and
				($originalValues['nemovitost_fullroom']==0 or $originalValues['nemovitost_fullroom']=='')
			) {
				$prevodnikVelikostiBytu = array(
					1 => 1,
					2 => 1,
					3 => 2,
					4 => 3,
					5 => 4,
					6 => 5,
					7 => 6,
					8 => 7,
					9 => 1,
					10 => 2,
					11 => 3,
					12 => 4,
					13 => 5,
					14 => 6,
					15 => 7,
					16 => 3,
					17 => 3

				);
				$convertedValues['rooms'] = $prevodnikVelikostiBytu[$originalValues['nemovitost_dispozice']];
			}

			if ($originalValues['nemovitost_uir_kraj_kod'] == 1) {
				$convertedValues['cityText'] = $originalValues['nemovitost_uir_mcast_text'];
			}

			$type = $this->convertCategory($originalValues);
			$convertedValues['type'] = $type;
			if ($type==self::CIL_POZEMEK){
				unset($convertedValues['heating']);
				unset($convertedValues['condition']);
			}
			$tags = $this->_createExtras($originalValues);

			if ($tags){
				$convertedValues['tagsText'] = $tags;
			}

		}
		return $convertedValues;
	}

	/**
	 * Poskládá ingatlanbazaří tagy.
	 *
	 * @param $originalValues Originální položky.
	 *
	 * @return string         Poskládané tagy.
	 */
	private function _createExtras($originalValues){
		$tagy = array();
		foreach ($this->_extras as $klic=>$mame) {
			if (is_array($mame)){
				foreach ($mame as $poradi=>$ficura) {
					if (substr($originalValues[$klic],$poradi ,1)) {
						$tagy[] = $ficura;
					}
				}
			} else {
				if ($originalValues[$klic]){
					$tagy[] = $mame;
				}
			}
		}
		return implode(" ", $tagy);
	}

	/**
	 * Vybere správnou kategorii pro tento export.
	 *
	 * @param array $originalValues Vstupní data nemovitosti ve formátu Irest 1.
	 *
	 * @return int Číslo kategorie dle Igatlanbazar,hu.
	 */
	function convertCategory($originalValues)
	{
		switch ($originalValues['nemovitost_typ']){
			case self::TYP_ZEMEDELSKE_OBJEKTY:
				$cat = self::CIL_ZEMEDELSKY;
				break;
			case self::TYP_KOMERCNI_OBJEKTY:
				switch ($originalValues['nemovitost_ucel_budovy']){
					case self::PODTYP_SKLAD:
						$cat = self::CIL_SKLAD;
						break;
					case self::PODTYP_ADMINISTRATIVNI_BUDOVA:
						$cat = self::CIL_KANCELAR;
						break;
					case self::PODTYP_OBCHODNI_CENTRUM:
					case self::PODTYP_NAKUPNI_STREDISKO:
						$cat = self::CIL_OBCHODNI_DUM;
						break;
					default:
						$cat = self::CIL_JINE_KOMERCNI;
						break;
				}
				break;
			case self::TYP_POZEMKY:
				$cat = self::CIL_POZEMEK;
				break;
			case self::TYP_BYTY:
				if ($originalValues['nemovitost_druh_objektu'] == self::DRUH_OBJEKTU_PANEL) {
					$cat = self::CIL_BYT_PANEL;
				} elseif ($originalValues['nemovitost_druh_objektu'] == self::DRUH_OBJEKTU_CIHLA) {
					$cat = self::CIL_BYT_CIHLA;
				} else {
					$cat = self::CIL_BYT_JINY;
				}
				break;
			case self::TYP_DOMY_A_VILY:
				switch ($originalValues['nemovitost_poloha_objektu']){
					case self::POLOHA_ROHOVY:
						$cat = self::CIL_IKERHAZ;
						break;
					case self::POLOHA_V_BLOKU:
						$cat = self::CIL_SORHAZ;
						break;
					case self::POLOHA_SAMOSTATNY:
					default:
						$cat = self::CIL_CSALADI_HAZ;
						break;
				}
				break;
			case self::TYP_KOMERCNI_PROSTORY:
				switch ($originalValues['nemovitost_druh_prostor']){
					case self::PROSTORY_KANCELARE:
						$cat = self::CIL_KANCELAR;
						break;
					case self::PROSTORY_OBCHODNI:
						$cat = self::CIL_OBCHODNI_DUM;
						break;
					case self::PROSTORY_SKLADOVACI:
						$cat = self::CIL_SKLAD;
						break;
					default:
						$cat = self::CIL_JINE_KOMERCNI;
						break;
				}
				break;
			case self::TYP_HOTELY:
				$cat = self::CIL_JINE_KOMERCNI;
				break;
			case self::TYP_CHATY:
				$cat = self::CIL_CHATA;
				break;
			case self::TYP_MALE_OBJEKTY:
				$cat = self::CIL_GARAZ;
				break;
			case self::TYP_HISTORICKE_OBJEKTY:
				switch ($originalValues['nemovitost_druh_objektu']){
					case self::DRUH_OBJEKTU_CIHLA:
					case self::DRUH_OBJEKTU_KAMEN:
						$cat = self::CIL_BYT_CIHLA;
						break;
					case self::DRUH_OBJEKTU_PANEL:
						$cat = self::CIL_BYT_PANEL;
						break;
					default:
						$cat = self::CIL_BYT_JINY;
						break;
				}
				break;
			case self::TYP_NAJEMNI_DOMY:
				switch ($originalValues['nemovitost_druh_objektu']) {
					case self::DRUH_OBJEKTU_PANEL:
						$cat = self::CIL_BYT_PANEL;
						break;
					case self::DRUH_OBJEKTU_CIHLA:
						$cat = self::CIL_BYT_CIHLA;
						break;
					default:
						$cat = self::CIL_BYT_JINY;
						break;
				}
				break;
			default:
				$cat = self::CIL_CSALADI_HAZ;
				break;
		}
		return $cat;
	}
}
