<?php
/**
 * Exportní API serveru České reality.
 *
 * @category   Dalten
 * @package    Export
 * @subpackage Api
 */
class Dalten_Export_Api_Ceskereality implements Dalten_Export_Api_ApiInterface, Dalten_Export_Api_LoggableApiInterface
{
	private $_backend = null;
	private $_export = null;
	private $_addressConverter = null;
	private $_entities;

	/**
	 * Konstruktor. Nastavuje závislosti.
	 *
	 * @param Dalten_Export_Ceskereality                     $export          Převodník hodnot.
	 * @param Dalten_Export_Api_Backend_Ceskereality         $backend         Backend exportu.
	 * @param Dalten_AddressConverter_Ceskereality_Interface $adressConverter Převodník adres.
	 * @param Serenity_Config_Config                         $entities        Definice entit.
	 */
	function __construct(
		Dalten_Export_Ceskereality $export,
		Dalten_Export_Api_Backend_Ceskereality $backend,
		Dalten_AddressConverter_Ceskereality_Interface $adressConverter,
		Serenity_Config_Config $entities
	)
	{
		$this->_export = $export;
		$this->_backend = $backend;
		$this->_addressConverter = $adressConverter;
		$this->_entities = $entities;
	}

	/**
	 * Naváže spojení se vzdáleným serverem.
	 *
	 * @param string $login            Přihlašovací jméno.
	 * @param string $password         Heslo.
	 * @param string $softwareKey      Softwarový klíč.
	 * @param array  $additionalParams Pole dalších parametrů.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function openConnection($login, $password, $softwareKey = '', array $additionalParams = array())
	{
		try {
			return $this->_backend->checkLogin($login, $password, $softwareKey, $additionalParams);
		} catch (Dalten_Export_Exception_HumanReadable $e) {
			return new Dalten_Export_Api_ServerResponse(false, 500, $e->getMessage());
		} catch (Exception $e) {
			return new Dalten_Export_Api_ServerResponse(false, 500);
		}
	}

	/**
	 * Vyexportuje nabídku.
	 *
	 * @param array $listingData      Data nabídky ve formátu iRest 1.
	 * @param array $userData         Data uživatele (makléře, kterému nabídka patří) ve formátu iRest 1.
	 * @param array $images           Pole fotografií nabídky ve formátu iRest 1.
	 * @param array $additionalParams Údaje které vrací server při exportu a jsou potřeba pro další práci
	 *                                s nabídkou. Data můžou sloužit pro editaci apod...
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru, jestli se export podařil nebo co selhalo.
	 */
	public function addListing(
		array $listingData,
		array $userData,
		array $images = array(),
		array $additionalParams = array()
	)
	{
		try {
		    $idNabidka = $listingData['id'];

			if (!empty($additionalParams['listing']['hide_price'])) {
				// Cena dohodou
				$listingData['nemovitost_cena_prodej'] = 0;
				$listingData['nemovitost_cena_pronajem'] = 0;
				$listingData['nemovitost_cena_poznamka'] = 'Informace v RK';
			}

			$listingData = $this->_addressConverter->convertAddress($listingData);
			$listingData = $this->_export->convertEntityValues('listing', $listingData, $this->_entities->listing);
			$userData = $this->_export->convertEntityValues('broker', $userData, $this->_entities->broker);
			return $this->_backend->addAdvert($idNabidka, $listingData, $userData, $images);
		} catch (Dalten_Export_Exception_HumanReadable $e) {
			return new Dalten_Export_Api_ServerResponse(false, 500, $e->getMessage());
		} catch (Exception $e) {
			return new Dalten_Export_Api_ServerResponse(false, 500);
		}
	}

	/**
	 * Odstraní nabídku ze vzdáleného serveru.
	 *
	 * @param array $listingData      Data nabídky ve formátu iRest 1.
	 * @param array $additionalParams Specifická data pro daný export.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function deleteListing(array $listingData, array $additionalParams = array())
	{
		try {
			$idNabidka = $listingData['id'];
			$listingData = $this->_addressConverter->convertAddress($listingData);
			$listingData = $this->_export->convertEntityValues('listing', $listingData, $this->_entities->listing);
			return $this->_backend->delAdvert($idNabidka, $listingData);
		} catch (Dalten_Export_Exception_HumanReadable $e) {
			return new Dalten_Export_Api_ServerResponse(false, 500, $e->getMessage());
		} catch (Exception $e) {
			return new Dalten_Export_Api_ServerResponse(false, 500);
		}
	}

	/**
	 * Vrátí seznam nabídek, které jsou na vzdáleném serveru.
	 *
	 * @return Dalten_Export_Api_ServerResponse Odpověď serveru.
	 */
	public function getListingsList()
	{
		throw new BadMethodCallException('Surprisingly not working!');
	}

	/**
	 * Nastaví logger pro backend.
	 * Logger bude použit pouze pokud to backend dovoluje.
	 *
	 * @param Dalten_Export_Api_Backend_Logger_LoggerInterface $logger Instance loggeru.
	 *
	 * @return bool Podařilo se přiřadit backendu logger?
	 */
	public function setLogger(Dalten_Export_Api_Backend_Logger_LoggerInterface $logger)
	{
		if ($this->_backend instanceof Dalten_Export_Api_Backend_LoggableBackendInterface) {
			$this->_backend->setLogger($logger);
		}

		return $this;
	}

	/**
	 * Odstraní nastavený logger pro backend.
	 *
	 * @return Dalten_Export_Api_LoggableApiInterface Fluent interface.
	 */
	public function removeLogger()
	{
		if ($this->_backend instanceof Dalten_Export_Api_Backend_LoggableBackendInterface) {
			$this->_backend->removeLogger();
		}

		return $this;
	}
}