<?php
/**
 * Export do formátu Annonce.cz
 *
 * @category Dalten
 * @package  Export
 */
class Dalten_Export_Annonce extends Dalten_Export_AbstractExport
{
	const ZEMEDELSKE_OBJEKTY = 1;
	const KOMERCNI_OBJEKTY = 2;
	const POZEMKY = 3;
	const BYTY = 4;
	const HISTORICKE_OBJEKTY = 5;
	const DOMY_A_VILY = 6;
	const HOTELY = 7;
	const NAJEMNI_DOMY = 8;
	const KOMERCNI_PROSTORY = 9;
	const CHATY = 10;
	const MALE_OBJEKTY = 11;

	const CENA_ZA_NEMOVITOST = 1;
	const CENA_ZA_METR = 3;
	const CENA_ZA_MESIC = 2;
	const CENA_ZA_METR_ZA_MESIC = 4;
	const CENA_ZA_METR_ZA_ROK = 5;
	const CENA_ZA_ROK = 6;
	const CENA_ZA_DEN = 7;

	/**
	 * Vybere hlavní plochu.
	 *
	 * @param array $convertedValues Převedené položky.
	 * @param array $originalvalues  Původní položky.
	 *
	 * @return null|float|int Hlavní plocha nebo null pokud ji nejde u tohoto typu najít.
	 */
	protected function _getHlavniPlocha(array $convertedValues, array $originalvalues)
	{
		$plochyTypu = array();
		$plochyTypu[2] = $plochyTypu[3] = $plochyTypu[4] = $plochyTypu[11] = 'nemovitost_plocha_celkova';
		$plochyTypu[1] = $plochyTypu[5] = $plochyTypu[6] = $plochyTypu[8] = $plochyTypu[7] = $plochyTypu[10]  = 'nemovitost_plocha_uzitna';
		$plochyTypu[9] = 'nemovitost_plocha_kancelari';
		$nazevPlochy = $plochyTypu[$originalvalues['nemovitost_typ']];
		if ($nazevPlochy && $originalvalues[$nazevPlochy]) {
			return $originalvalues[$nazevPlochy];
		}
		return null;
	}

	/**
	 * Doplní do převedených položek informace o ceně.
	 *
	 * Pokud je cena v korunách, spočítá unified price, kterou Annonce používá k řazení podle ceny.
	 *
	 * @param array $convertedValues Převedené položky.
	 * @param array $originalvalues  Původní položky.
	 *
	 * @return array Převedené položky s doplněnou cenou.
	 */
	protected function _convertPrice(array $convertedValues, array $originalvalues)
	{
		$pocitatUnifiedPrice = true;
		$meny = array(1=>'Kč',2=>'USD',3=>'EUR');
		$mena = $meny[$originalvalues['nemovitost_mena']];
		if ($originalvalues['nemovitost_mena']!=1) {
			$pocitatUnifiedPrice = false;
		}
		if ($originalvalues['zakazka_typ']==1) {
			$cena = $originalvalues['nemovitost_cena_prodej'];
			$cenaJednotka = $originalvalues['nemovitost_cena_prodej_jednotka'];
			$hlavniPlocha = $this->_getHlavniPlocha($convertedValues, $originalvalues);
			if ($pocitatUnifiedPrice && $hlavniPlocha) {
				$unified = $cena;
				if ($cenaJednotka==self::CENA_ZA_METR) {
					$unified = $cena * $hlavniPlocha;
				}
				$convertedValues['unified-price'] = sprintf('%d', $unified);
			}
			$typyCen = array(1=>'za nemovitost', 3=>'za m2');
			$convertedValues['price-type'] = $typyCen[$cenaJednotka];
			$convertedValues['price'] = sprintf('%d', $originalvalues['nemovitost_cena_prodej'] ) . ' ' . $mena . ' ' . $typyCen[$cenaJednotka];
			$convertedValues['other_payments'] = $originalvalues['nemovitost_cena_poznamka'];
		} else {
			$cena = $originalvalues['nemovitost_cena_pronajem'];
			$cenaJednotka = $originalvalues['nemovitost_cena_pronajem_jednotka'];
			$hlavniPlocha = $this->_getHlavniPlocha($convertedValues, $originalvalues);
			if ($pocitatUnifiedPrice && $hlavniPlocha) {
				switch ($cenaJednotka) {
					case self::CENA_ZA_MESIC:
						$unified = $cena * 12;
						break;
					case self::CENA_ZA_METR_ZA_MESIC:
						$unified = $cena * 12 * $hlavniPlocha;
						break;
					case self::CENA_ZA_METR_ZA_ROK:
						$unified = $cena * $hlavniPlocha;
						break;
					case self::CENA_ZA_ROK:
					default:
						$unified = $cena;
						break;
					case self::CENA_ZA_DEN:
						$unified = $cena * 365;
						break;
				}
				$convertedValues['unified-price'] = $unified;
			}
			$typyCen = array(2=>'za měsíc', 4=>'za m2/měsíc', 5=>'za m2/rok', 6=>'za rok', 7=>'za den');
			$convertedValues['price-type'] = $typyCen[$cenaJednotka];
			$convertedValues['price'] = sprintf('%d', $originalvalues['nemovitost_cena_pronajem'] ) . ' ' . $mena . ' ' . $typyCen[$cenaJednotka];
			$convertedValues['other_payments'] = $originalvalues['nemovitost_cena_poznamka'];

		}
		return $convertedValues;
	}

	/**
	 * Doplní některé plochy, které jsou závislé na typech nemovitosti.
	 *
	 * @param array $convertedValues Převedené položky.
	 * @param array $originalvalues  Původní položky.
	 *
	 * @return array Převedené položky s doplněnými plochami.
	 */
	protected function _convertAreas(array $convertedValues, array $originalvalues)
	{
		switch ($originalvalues['nemovitost_typ']) {
			case self::KOMERCNI_OBJEKTY:
				$convertedValues['PLOCHA_UZITNA'] = $originalvalues['nemovitost_plocha_celkova'];
				break;
			case self::POZEMKY:
				$convertedValues['PLOCHA_PARCELY'] = $originalvalues['nemovitost_plocha_celkova'];
				break;
			case self::BYTY:
				$convertedValues['PLOCHA_OBYTNA'] = $originalvalues['nemovitost_plocha_celkova'];
				if (isset($originalvalues['nemovitost_plocha_vlastni_pozemek'])) {
					$convertedValues['PLOCHA_PARCELY'] = $originalvalues['nemovitost_plocha_vlastni_pozemek'];
				}
				break;
			case self::HISTORICKE_OBJEKTY:
				$convertedValues['PLOCHA_OBYTNA'] = $originalvalues['nemovitost_plocha_celkova'];
				break;
			case self::HOTELY:
				$convertedValues['PLOCHA_NEBYTOVA'] = $originalvalues['nemovitost_plocha_celkova'];
				break;
		}
		return $convertedValues;
	}

	/**
	 * Doplní informace o nemovitosti o složitěji rozhodnutelné položky.
	 *
	 * @param array  $convertedValues Pozměněné položky.
	 * @param array  $originalvalues  Původní položky.
	 * @param string $entityName      Název entity - listing je nemovitost.
	 *
	 * @return array Doplněné pozměněné položky.
	 */
	protected function _convert(array $convertedValues, array $originalvalues, $entityName)
	{
		if ($entityName=='listing') {
			switch ($originalvalues['nemovitost_typ']) {
				case self::BYTY:
				case self::DOMY_A_VILY:
					$convertedValues['category'] = 'bydlení';
					break;
				case self::POZEMKY:
					$convertedValues['category'] = 'pozemky';
					break;
				case self::KOMERCNI_OBJEKTY:
				case self::KOMERCNI_PROSTORY:
				case self::ZEMEDELSKE_OBJEKTY:
				case self::HOTELY:
				case self::NAJEMNI_DOMY:
					$convertedValues['category'] = 'komerční';
					break;
				default:
					$convertedValues['category'] = 'ostatní';
					break;
			}

			$convertedValues = $this->_convertPrice($convertedValues, $originalvalues);

			$hlavniPlocha = $this->_getHlavniPlocha($convertedValues, $originalvalues);
			if ($hlavniPlocha) {
				$convertedValues['measurement'] = $hlavniPlocha;
			}

			$convertedValues = $this->_convertAreas($convertedValues, $originalvalues);

			$convertedValues['ELEKTRINA'] = ($this->_isAnyChecked($originalvalues, 'nemovitost_elektrina') ? 'True' : 'False');
			$convertedValues['VODA'] = ($this->_isAnyChecked($originalvalues, 'nemovitost_voda') ? 'True' : 'False');



			return $convertedValues;
		}
		if ($entityName=='agent') {
			$jmenoCele = $originalvalues['uzivatel_os_jmeno'] . (!empty($originalvalues['uzivatel_os_prostredni_jmeno']) ?  ' ' . $originalvalues['uzivatel_os_prostredni_jmeno'] : '') . ' ' . $originalvalues['uzivatel_os_prijmeni'];

			if (!empty($originalvalues['uzivatel_tituly_pred_jmenem'])) {
				$jmenoCele = $originalvalues['uzivatel_tituly_pred_jmenem'] . ' ' . $jmenoCele;
			}

			if (!empty($originalvalues['uzivatel_tituly_za_jmenem'])) {
				$jmenoCele = $jmenoCele . ' ' . $originalvalues['uzivatel_tituly_za_jmenem'];
			}

			$convertedValues['MAKLER_JMENO'] = $jmenoCele;
		}
		return $convertedValues;
	}

	/**
	 * Vrací je-li v poli $data zaškrnuté cokoliv z multicheckboxu $fieldName.
	 *
	 * @param array  $data      Pole s daty.
	 * @param string $fieldName Název políčka.
	 *
	 * @return bool Je to tak?
	 */
	private function _isAnyChecked(array $data, $fieldName) {
		if (isset($data[$fieldName])) {
			if (strpos($data[$fieldName], '1')===false) {
				return false;
			}
			return true;
		}
		return false;
	}

}
