<?php
namespace Dalten\WebBundle\PhpParser\NodeVisitor;

use PhpParser\Node;
use PhpParser\Node\Expr\MethodCall;
use PhpParser\NodeVisitorAbstract;
use Symfony\Component\Translation\MessageCatalogue;

/**
 * Visitor pro PHP Parser, který z analyzovaného souboru vytahuje první argument volání metod
 * trans a transChoice, pokud je tento argument string.
 *
 * Vzhledem k nutnosti dodržování interface jsou nalezené zprávy ukládány interně a do katalogu jsou vloženy
 * až voláním metody fillCatalogue.
 */
class TranslationFinder extends NodeVisitorAbstract
{
	/**
	 * Nalezené řetězce k překladu.
	 *
	 * @var array
	 */
	private $_translations = array();

	/**
	 * Metoda, volaná při vstupu do nového nodu z AST.
	 *
	 * Zkoumá, zda je note voláním metody trans nebo transChoice. Pokud ano, pak se pokusí vyextrahovat první argument
	 * (je-li to string) k překladu.
	 *
	 * Vzhledem k nutnosti dodržování interface jsou nalezené zprávy ukládány interně a do katalogu jsou vloženy
	 * až voláním metody fillCatalogue.
	 *
	 * @param Node $node Uzel, do kterého právě Traverser vstoupil.
	 */
	public function enterNode(Node $node)
	{
		if ($node instanceof MethodCall) {
			if ($node->name === 'trans' || $node->name === 'transChoice') {
				$firstArg = $node->args[0];
				if ($firstArg->value instanceof Node\Scalar\String) {
					$this->_translations[] = $firstArg->value->value;
				}
			}
		}
	}

	/**
	 * Naplní předaný katalog nalezenými překlady.
	 *
	 * @param MessageCatalogue $catalogue Překladový katalog.
	 * @param string           $prefix    Prefix, přidaný před nově nalezené zprávy.
	 */
	public function fillCatalogue(MessageCatalogue $catalogue, $prefix = '')
	{
		$convertedMessages = array();
		foreach ($this->_translations as $message) {
			$convertedMessages[$message] = $prefix . $message;
		}

		$catalogue->add($convertedMessages);
	}
}
