<?php
namespace Dalten\WebBundle\Processor\CompanyList;

use Dalten\WebBundle\Filter\AddressFilter;
use Dalten\WebBundle\Filter\CompanyFilter;
use Doctrine\ORM\QueryBuilder;

/**
 * Jednoduchá komponenta, obstarávající přípravu Querysetu na výpis firem.
 */
class FilterToQueryBuilderConverter
{
    /**
     * Přidá do queryBuilderu order a základní where (společná pro všechny výpisy).
     *
     * @param QueryBuilder $queryBuilder Query Builder.
     * @param string $companyAlias Alias entity Company v QB.
     * @return QueryBuilder
     */
	public function prepareQueryBuilder(QueryBuilder $queryBuilder, $companyAlias)
	{
		$queryBuilder->andWhere($companyAlias . '.id > :min_company_id')->setParameter('min_company_id', 1);

		return $queryBuilder;
	}

	/**
	 * Přidá omezení z filtru do QueryBuilderu.
	 *
	 * @param QueryBuilder  $queryBuilder QB z nabídek.
	 * @param CompanyFilter $filter       Filtr firem.
	 * @param string        $companyAlias Alias entity Company v QB.
	 *
	 * @return QueryBuilder
	 */
	public function addFilterToQueryBuilder(QueryBuilder $queryBuilder, CompanyFilter $filter, $companyAlias)
	{
		$prefix = $companyAlias . '.';


		if ($filter->search_text) {
			$expr = $queryBuilder->expr();

			$condition = $expr->orX(
				$expr->like($prefix . '_nazev', ':search_text'),
				$expr->like($prefix . '_okres_text', ':search_text'),
				$expr->like($prefix . '_kraj_text', ':search_text'),
				$expr->like($prefix . '_mesto_text', ':search_text')
			);

			$queryBuilder->andWhere($condition)->setParameter('search_text', '%' . $filter->search_text . '%');
        }

		if ($filter->getAddressFilters()) {
			$this->_addAddressRules($filter->getAddressFilters(), $queryBuilder, $prefix);
		}

		return $queryBuilder;
	}

	/**
	 * Vytvoří omezení na adresu.
	 *
	 * @param AddressFilter[] $addressFilters Omezení na adresu.
	 * @param QueryBuilder    $queryBuilder   QB k doplnění podmínek.
	 * @param string          $prefix         Prefix entity nabídky (vč. separatoru).
	 *
	 * @return QueryBuilder Předaný QueryBuilder.
	 */
	private function _addAddressRules(array $addressFilters, QueryBuilder $queryBuilder, $prefix)
	{
		$expr = $queryBuilder->expr();
		$rules = array();
		$countyColumn = $prefix . '_kraj';

        foreach ($addressFilters as $index => $filter) {
            $regionColumn = $prefix . '_okres';
            $queryBuilder->setParameter('locality_county_' . $index, $filter->county_code);
            if ($filter->region_codes) {
                $queryBuilder->setParameter('locality_region_' . $index, $filter->region_codes);
                $rules[] = $expr->andX(
                    $expr->eq($countyColumn, ':locality_county_' . $index),
                    $expr->in($regionColumn, ':locality_region_' . $index)
                );
            } else {
                $rules[] = $expr->eq($countyColumn, ':locality_county_' . $index);
            }
        }

		if ($rules) {
			$queryBuilder->andWhere(call_user_func_array(array($expr, 'orX'), $rules));
		}

		return $queryBuilder;
	}
}
