<?php

namespace Dalten\WebBundle\Processor\Favourite;


use Dalten\Date\Date;
use Symfony\Component\HttpFoundation\Cookie;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Stará se o načtení a uložení oblíbených do cookies.
 *
 * POZOR uložení do cookies dojde pouze když se odešle předaný response object.
 *
 * @package Dalten\WebBundle\Processor\Favourite
 */
class CookieStorage implements StorageInterface
{
	/**
	 * @var Request
	 */
	private $_request;

	/**
	 * @var Response
	 */
	private $_response;

	private $_listingIds = array();

	private $_cookieName = 'favourites';

	/**
	 * Inicializuje objekt.
	 *
	 * @param \Symfony\Component\HttpFoundation\Request  $request
	 * @param \Symfony\Component\HttpFoundation\Response $response
	 */
	public function __construct(Request $request, Response $response)
	{
		$this->_request = $request;
		$this->_response = $response;

		$listingIds = array_filter(explode(',', rtrim($request->cookies->get($this->_cookieName, ''), ',')));
		$this->_listingIds = $listingIds;
	}

	/**
	 * Přidá nabídky do oblíbených.
	 *
	 * @param array $ids Pole identifikátorů nabídek.
	 */
	public function addListings(array $ids)
	{
		$this->_listingIds = array_merge($this->_listingIds, $ids);
		$this->_setCookie();
	}

	/**
	 * Smaže nabídky z oblíbených.
	 *
	 * @param array $ids Pole identifikátorů nabídek.
	 */
	public function delListings(array $ids)
	{
		$this->_listingIds = \array_diff($this->_listingIds, $ids);
		$this->_setCookie();
	}

	/**
	 * Vrátí seznam identifikátorů oblíbených nabídek.
	 *
	 * @return array
	 */
	public function getListingIds()
	{
		return $this->_listingIds;
	}

	/**
	 * Serializuje data do formátu pro vložení do cookie.
	 *
	 * @return string
	 */
	private function _getSerializeListingIds()
	{
		return sprintf('%s', implode(',', \array_unique($this->_listingIds)));
	}

	/**
	 * Nastaví název proměné pod kterou se mají oblíbené hledat v cookies.
	 *
	 * @param string $cookieName
	 *
	 * @return $this
	 */
	public function setCookieName($cookieName)
	{
		$this->_cookieName = (string) $cookieName;

		return $this;
	}

	/**
	 * Nastaví cookie a přidá ji do response objektu.
	 */
	private function _setCookie()
	{
		$expire = Date::now();
		$expire->add('90 days');
		$domain = sprintf('.%s', $this->_request->getHost());
		$path = $this->_request->getBasePath();
		$data = $this->_getSerializeListingIds();
		$cookie = new Cookie($this->_cookieName, $data, $expire->getDateTime(), $path, $domain);

		$this->_response->headers->setCookie($cookie);
	}
}
 