<?php
namespace Dalten\WebBundle\SeoRoute;

use Dalten\WebBundle\Filter\ListingFilter;
use Dalten\WebBundle\SeoRoute\Matcher\ListingFilter\FragmentMatcherInterface;

/**
 * Třída, reprezentující a spravující více matcherů fragmentů filtru.
 *
 * Dokáže najít routu na základě výstupu předaných macherů a konfiguračního pole.
 */
class FragmentCollection
{
	/**
	 * Pole definic rout dle fragmentů ve struktuře:
	 * [názevRouty] => array(pole, požadovaných, fragmentů).
	 *
	 * @var array
	 */
	private $_definitions = array();

	/**
	 * Pole matcherů.
	 *
	 * @var Matcher\ListingFilter\FragmentMatcherInterface[]
	 */
	private $_matchers = array();

	/**
	 * Nastavuje konfiguraci a předané matchery.
	 *
	 * $routeDefinitions je pole definic rout dle fragmentů ve struktuře:
	 * array('názevRouty' => array('flat_sub_type', 'listing_type', 'advert_type')).
	 *
	 * @param array                      $routeDefinitions Pole definic rout dle názvů parametrů, které obsahují.
	 * @param FragmentMatcherInterface[] $fragmentMatchers Pole jednotlivých matcherů.
	 */
	public function __construct(array $routeDefinitions, array $fragmentMatchers)
	{
		foreach ($routeDefinitions as $routeName => $allowedParams) {
			sort($allowedParams);
			$this->_definitions[$routeName] = $allowedParams;
		}
		$this->_matchers = $fragmentMatchers;
	}

	/**
	 * Zkusí získat konfiguraci pro routu.
	 *
	 * @param ListingFilter $filter Vyplněný filtr nabídek.
	 *
	 * @return array Pole s klíči 'parameters' a 'name' a nebo null, pokud se routu nepovede najít.
	 */
	public function getRouteConfig(ListingFilter $filter)
	{
		$matchedFilterParams = array('locality_country_code');
		$routeParams = array();

		foreach ($this->_matchers as $matcher) {
			$params = $matcher->getRouteParamsForFilter($filter);
			if ($params) {
				$routeParams = array_merge($routeParams, $params);
				$matchedFilterParams = array_merge(
					$matchedFilterParams, $matcher->getMatchedFilterParamsForFilter($filter)
				);
			}
		}

		$matchedRouteParams = array_unique(array_keys($routeParams));
		sort($matchedRouteParams);

		foreach ($this->_definitions as $routeName => $acceptedParams) {
			if ($acceptedParams == $matchedRouteParams
				&& $this->_doesFilterContainOnlyAllowedProperties($filter, $matchedFilterParams)) {
				return array('name' => $routeName, 'parameters' => $routeParams);
			}
		}

		return null;
	}

	/**
	 * Zkontroluje, zda jsou v předaném filtru vyplněny pouze požadované vlastnosti.
	 *
	 * @param ListingFilter $filter            Vyplněný filtr nabídek.
	 * @param array         $allowedProperties Pole názvu povolených vlastností filtru.
	 *
	 * @return bool Jsou ve filtru obsaženy pouze povolené vlastnosti?
	 */
	private function _doesFilterContainOnlyAllowedProperties(ListingFilter $filter, array $allowedProperties)
	{
		foreach ($filter->toArray() as $propertyName => $propertyValue) {
			if (!in_array($propertyName, $allowedProperties) && !empty($filter->$propertyName)) {
				return false;
			}
		}

		$regionCodesAreValid = $countyCodesAreValid = true;
		if (!in_array('county', $allowedProperties)) {
			foreach ($filter->getAddressFilters() as $addressFilter) {
				if ($addressFilter->county_code) {
					$countyCodesAreValid = false;
					break;
				}
			}
		}

		if (!in_array('region', $allowedProperties)) {
			foreach ($filter->getAddressFilters() as $addressFilter) {
				if ($addressFilter->region_codes) {
					$regionCodesAreValid = false;
					break;
				}
			}
		}

		return $regionCodesAreValid && $countyCodesAreValid;
	}
}
