<?php
namespace Dalten\WebBundle\SeoRoute\Matcher\ListingFilter;

use Dalten\WebBundle\Filter\ListingFilter;

/**
 * Jednoduchý matcher, který převádí části filtru na části konfigurace routy.
 */
class BasicFragmentMatcher implements FragmentMatcherInterface
{
	/**
	 * Název parametru routy, který doplňujeme.
	 *
	 * @var string
	 */
	private $_routeParamName;

	/**
	 * Konfigurační pole, obsahující jako klíče slug a jako hodnoty očekávané hodnoty dané položky filtru.
	 *
	 * @var array
	 */
	private $_config;

	/**
	 * Úložiště pro cache mezivýpočtů.
	 *
	 * @var array
	 */
	private $_cache = array();

	/**
	 * Pole vlastností filtru, které tento matcher sleduje.
	 *
	 * @var array
	 */
	private $_watchedProperties = array();

	/**
	 * Nastavuje konfiguraci matcheru.
	 *
	 * $config je pole, obsahující jako klíče slug a jako hodnoty očekávané hodnoty dané položky filtru.
	 *
	 * Např.: array(array('byty' => array(['listing_type'] => array(4))).
	 *
	 * @param string $routeParamName      Název fragmentu (parametru routy).
	 * @param array  $config              Konfigurační pole.
	 */
	public function __construct($routeParamName, array $config)
	{
		$this->_routeParamName = $routeParamName;
		$this->_config = $config;

		foreach ($config as $expectedParams) {
			$this->_watchedProperties = array_unique(
				array_merge($this->_watchedProperties, array_keys($expectedParams))
			);
		}
	}

	/**
	 * Vypočte klíč do cache pro daný filtr.
	 *
	 * @param ListingFilter $filter Vyplněný filtr nabídek.
	 *
	 * @return string Cache key pro daný filtr.
	 */
	protected function _computeCacheKey(ListingFilter $filter)
	{
		$out = array();
		foreach ($this->_watchedProperties as $propertyName) {
			$value = $filter->$propertyName;
			if (is_array($value)) {
				sort($value);
				$value = sprintf('[%s]', implode(',', $value));
			}
			$out[] = sprintf('%s=%s', $propertyName, $value);
		}

		return implode('||', $out);
	}

	/**
	 * Vrací nastavené konfigurační pole.
	 *
	 * @return array Nastavené konfigurační pole
	 */
	protected function _getConfig()
	{
		return $this->_config;
	}

	/**
	 * Vrací nastavený název parametru routy.
	 *
	 * @return string Nastavený název parametru routy.
	 */
	protected  function _getRouteParamName()
	{
		return $this->_routeParamName;
	}

	/**
	 * Vrací pole parametrů pro routu, obsahující matchnutou část filtru.
	 *
	 * @param ListingFilter $filter Vyplněný filtr nabídek.
	 *
	 * @return array Pole parametrů nalezené routy a nebo prázdné pole.
	 */
	public function getRouteParamsForFilter(ListingFilter $filter)
	{
		$slug = $this->_findSlug($filter);

		if ($slug) {
			return array($this->_routeParamName => $slug);
		}

		return array();
	}

	/**
	 * Vrátí pole názvů vlastností filtru, které byly zpracovány tímto fragmentem pro daný filtr.
	 *
	 * Vrací prázdné pole, pokud daný filtr není pokryt.
	 *
	 * @param ListingFilter $filter Vyplněný filtr nabídek.
	 *
	 * @return array Pole názvů vlastností filtru, byly zpracovány tímto fragmentem pro daný filtr.
	 */
	public function getMatchedFilterParamsForFilter(ListingFilter $filter)
	{
		$slug = $this->_findSlug($filter);
		if (isset($this->_config[$slug])) {
			return array_keys($this->_config[$slug]);
		}

		return array();
	}

	/**
	 * Projede pole $_config a hledá, zda některá z jeho hodnot odpovídá předanému filtru.
	 *
	 * @param ListingFilter $filter Vyplněný fitlr nabídek.
	 *
	 * @return null|string Název nalezeného slugu a nebo null.
	 */
	protected function _findSlug(ListingFilter $filter)
	{
		$cacheKey = $this->_computeCacheKey($filter);

		if (isset($this->_cache[$cacheKey])) {
			return $this->_cache[$cacheKey];
		}


		foreach ($this->_config as $slug => $expectedProperties) {
			$target = count($expectedProperties);
			$found = 0;
			foreach ($expectedProperties as $propertyName => $expectedPropertyValue) {
				$currentValue = $filter->{$propertyName};

				if (empty($currentValue)) {
					break;
				}

				if (is_array($currentValue) && is_array($expectedPropertyValue)) {
					sort($currentValue);
					sort($expectedPropertyValue);
				}

				if ($currentValue == $expectedPropertyValue) {
					++$found;
				}
			}

			if ($found == $target) {
				$this->_cache[$cacheKey] = $slug;
				return $slug;
			}
		}

		$this->_cache[$cacheKey] = null;
		return null;
	}
}

