<?php
namespace Dalten\WebBundle\Breadcrumbs\Listing\Builder;

use Dalten\WebBundle\Filter\ListingFilter;
use Dalten\WebBundle\Entity\Listing;
use Dalten\WebBundle\Breadcrumbs\Breadcrumb;
use Mockery as m;

class ListingAndAdvertTypeTest extends \PHPUnit_Framework_TestCase
{
	private $_defaultConfig = array(
		'routeName' => 'test',
		'advertTypeMap' => array(
			'prodej' => array('acceptedValues' => array(1), 'label' => 'Prodej', 'routeParams' => array('advert_type' => 'prodej')),
            'pronajem' => array('acceptedValues' => array(2), 'label' => 'Pronájem', 'routeParams' => array('advert_type' => 'pronajem'))
		)
	);

	private $_listingTypeBreadcrumb;

	public function setUp()
	{
		$this->_listingTypeBreadcrumb = new Breadcrumb('Test', 'test', array('listing_type' => 'bagr'));
	}

	/**
	 * @dataProvider listingAndAdvertTypeAndExpectedParamsAndLabelProvider
	 */
	public function testCreateBreadcrumbFromFilterReturnsSpecifiedBreadcrumb($advertType, $expectedRouteParams, $expectedLabel)
	{
		$filter = new ListingFilter(array('listing_type' => 1, 'advert_type' => $advertType));

		$listingTypeBuilder = m::mock('Dalten\WebBundle\Breadcrumbs\Listing\Builder\ListingType');
		$listingTypeBuilder->shouldReceive('getBreadcrumbFromFilter')->with($filter)->once()
			->andReturn($this->_listingTypeBreadcrumb);

		$builder = new ListingAndAdvertType($listingTypeBuilder, $this->_defaultConfig);
		$breadcrumb = $builder->getBreadcrumbFromFilter($filter);

		$this->assertSame($expectedLabel, $breadcrumb->getLabel());
		$this->assertSame($this->_defaultConfig['routeName'], $breadcrumb->getRouteName());
		$this->assertEquals($expectedRouteParams, $breadcrumb->getRouteParams(), '', 0, 10, true);
	}

	/**
	 * @dataProvider listingAndAdvertTypeAndExpectedParamsAndLabelProvider
	 */
	public function testCreateBreadcrumbFromListingReturnsSpecifiedBreadcrumb($advertType, $expectedRouteParams, $expectedLabel)
	{
		$listing = new Listing;
		$listing->advert_type = $advertType;

		$listingTypeBuilder = m::mock('Dalten\WebBundle\Breadcrumbs\Listing\Builder\ListingType');
		$listingTypeBuilder->shouldReceive('getBreadcrumbFromListing')->with($listing)->once()
			->andReturn($this->_listingTypeBreadcrumb);

		$builder = new ListingAndAdvertType($listingTypeBuilder, $this->_defaultConfig);
		$breadcrumb = $builder->getBreadcrumbFromListing($listing);

		$this->assertNotNull($breadcrumb);
		$this->assertSame($expectedLabel, $breadcrumb->getLabel());
		$this->assertSame($this->_defaultConfig['routeName'], $breadcrumb->getRouteName());
		$this->assertEquals($expectedRouteParams, $breadcrumb->getRouteParams(), '', 0, 10, true);
	}


	public function testCreateBreadcrumbFromFilterReturnsNullIfListingTypeBreadcrumbCannotBeCreated()
	{
		$filter = new ListingFilter(array('listing_type' => array(1), 'advert_type' => array(1)));

		$listingTypeBuilder = m::mock('Dalten\WebBundle\Breadcrumbs\Listing\Builder\ListingType');
		$listingTypeBuilder->shouldReceive('getBreadcrumbFromFilter')->with($filter)->once()->andReturn(null);

		$builder = new ListingAndAdvertType($listingTypeBuilder, $this->_defaultConfig);
		$breadcrumb = $builder->getBreadcrumbFromFilter($filter);

		$this->assertNull($breadcrumb);
	}

	public function testCreateBreadcrumbFromListingReturnsNullIfInvalidListingTypeIsSet()
	{
		$listing = new Listing;

		$listingTypeBuilder = m::mock('Dalten\WebBundle\Breadcrumbs\Listing\Builder\ListingType');
		$listingTypeBuilder->shouldReceive('getBreadcrumbFromListing')->with($listing)->once()
			->andReturn($this->_listingTypeBreadcrumb);

		$builder = new ListingAndAdvertType($listingTypeBuilder, $this->_defaultConfig);
		$breadcrumb = $builder->getBreadcrumbFromListing($listing);

		$this->assertNull($breadcrumb);
	}



	public function listingAndAdvertTypeAndExpectedParamsAndLabelProvider()
	{
		return array(
			'byty-prodej(4,1)' => array(1, array('advert_type' => 'prodej', 'listing_type' => 'bagr'), 'Prodej'),
			'byty-pronajem(4,2)' => array(2, array('advert_type' => 'pronajem', 'listing_type' => 'bagr'), 'Pronájem')
		);
	}

}
