<?php

namespace Dalten\Util;

/**
 * Utilitky pro práci s řetězcem.
 */
class StringHelper
{
	/**
	 * Converts a string from UTF-8 to an identifier form.
	 *
	 * @param string $string String to convert
	 * @return string
	 */
	public static function slugify($string)
	{
		// Convert to lowercase ASCII and than all non-alphanumeric characters to dashes
		$ident = \preg_replace('~[^a-z0-9]~', '-', \strtolower(self::utf2ascii($string)));
		// Remove multiple dashes and dashes on boundaries
		return \trim(\preg_replace('~-+~', '-', $ident), '-');
	}

	/**
	 * Converts a string from UTF-8 to ASCII.
	 *
	 * @param string $string String to convert
	 * @return string
	 */
	public static function utf2ascii($string)
	{
		static $replace = array(
			'á' => 'a', 'Á' => 'A', 'ä' => 'a', 'Ä' => 'A', 'â' => 'a', 'Â' => 'A', 'ă' => 'a', 'Ă' => 'A', 'ą' => 'a',
			'Ą' => 'A', 'č' => 'c', 'Č' => 'C', 'ç' => 'c', 'Ç' => 'C', 'ć' => 'c', 'Ć' => 'C', 'ď' => 'd', 'Ď' => 'D',
			'đ' => 'd', 'Đ' => 'D', 'é' => 'e', 'É' => 'E', 'ě' => 'e', 'Ě' => 'E', 'ë' => 'e', 'Ë' => 'E', 'ę' => 'e',
			'Ę' => 'E', 'í' => 'i', 'Í' => 'I', 'î' => 'i', 'Î' => 'I', 'ł' => 'l', 'Ł' => 'L', 'ľ' => 'l', 'Ľ' => 'L',
			'ĺ' => 'l', 'Ĺ' => 'L', 'ń' => 'n', 'Ń' => 'N', 'ň' => 'n', 'Ň' => 'N', 'ó' => 'o', 'Ó' => 'O', 'ô' => 'o',
			'Ô' => 'O', 'ö' => 'o', 'Ö' => 'O', 'ő' => 'o', 'Ő' => 'O', 'o' => 'o', 'O' => 'O', 'ř' => 'r', 'Ř' => 'R',
			'ŕ' => 'r', 'Ŕ' => 'R', 'š' => 's', 'Š' => 'S', 'ś' => 's', 'Ś' => 'S', 'ş' => 's', 'Ş' => 'S', 'ť' => 't',
			'Ť' => 'T', 'ţ' => 't', 'Ţ' => 'T', 'ú' => 'u', 'Ú' => 'U', 'ů' => 'u', 'Ů' => 'U', 'ü' => 'u', 'Ü' => 'U',
			'ű' => 'u', 'Ű' => 'U', 'ý' => 'y', 'Ý' => 'Y', 'ž' => 'z', 'Ž' => 'Z', 'ź' => 'z', 'Ź' => 'Z', 'ż' => 'z',
			'Ż' => 'Z', 'ß' => 'ss', 'å' => 'a', 'Å' => 'A', '²' => '2'
		);
		return \strtr($string, $replace);
	}

	/**
	 * Trims all words in a string longer than given length.
	 * String is delimited by whitespaces.
	 * If a word is trimmed, an "etc" is added at the end. Its length is also considered.
	 *
	 * @param string  $string Processed string
	 * @param integer $length Maximum word length
	 * @param string  $etc    "etc" definition
	 *
	 * @return string
	 */
	public static function cutWords($string, $length = 25, $etc = '...')
	{
		$length = (int)$length;

		return \preg_replace_callback('~[^\s]{' . $length . ',}~', function($matches) use ($length, $etc)
		{
			return StringHelper::cut($matches[0], $length, $etc);
		}, $string);
	}

	/**
	 * Trims a string to given length.
	 * Trims at word boundaries (all non-alphanumeric characters are considered delimiters).
	 * If the given string is trimmed, an "etc" is added at the end. Its length is also considered.
	 *
	 * @param string  $string Trimmed string
	 * @param integer $max    Maximum length
	 * @param string  $etc    "etc" definition
	 *
	 * @return string
	 */
	public static function cut($string, $max = 50, $etc = '...')
	{
		// Trim whitespace
		$string = \trim($string);

		// No trimming is needed
		if (\mb_strlen($string, 'utf-8') <= $max) {
			return $string;
		}

		// Find out "etc" length
		switch ($etc) {
			default:
				$etcLength = \mb_strlen(\html_entity_decode($etc, ENT_COMPAT, 'utf-8'), 'utf-8');
				break;
		}

		// Look for word boundaries
		$search = \mb_substr($string, 0, ($max - $etcLength) + 1, 'utf-8');
		if (\preg_match('~[^\w\pL\pN]~u', $search)) {
			// Boundary found
			$string = \preg_replace('~[^\w\pL\pN]*[\w\pL\pN]*$~uU', '', $search);
		} else {
			// No word boundary found, will trim in the middle of a word
			$string = \mb_substr($string, 0, $max - $etcLength, 'utf-8');
		}

		// Add "etc" at the end
		$string .= $etc;

		return $string;
	}

	/**
	 * Nahradí v textu předaná slova placeholderem.
	 *
	 * @param string $source       Text ve kterém chceme nahradit řetězce.
	 * @param array  $words        Pole slov, které budeme nahrazovat.
	 * @param string $placeholder  Řetězec který zaměníme za hledaná slova.
	 *
	 * @return mixed
	 */
	public static function replace($source, array $words, $placeholder)
	{
		return \str_ireplace($words, $placeholder, $source);
	}
}
