<?php
namespace Dalten\WebBundle\Processor\Codebooks;

use Dalten\WebBundle\Entity\Variable;
use Dalten\WebBundle\Processor\Codebooks\Exception\UnknownCodebookException;
use Dalten\WebBundle\Processor\Codebooks\Exception\UnknownLanguageException;
use Doctrine\Common\Cache\Cache;
use Doctrine\ORM\EntityRepository;

/**
 * Provider na získávání číselníků.
 */
class CodebookProvider
{
	/**
	 * Repozitář proměnných.
	 *
	 * @var  EntityRepository
	 */
	private $_variableRepository;

	/**
	 * Cache.
	 *
	 * @var Cache
	 */
	private $_cacheBackend;

	/**
	 * Povolené jazyky.
	 *
	 * Pole, kde klíč je dvouznakový kód a hodnota je suffix lang-* v DB u variables.
	 * Tj.: array('cs' => 'czech');
	 *
	 * @var array
	 */
	protected $_allowedLanguages;

	/**
	 * CodebookProvider constructor.
	 *
	 * @param EntityRepository $variableRepository Repozitář na proměnné.
	 * @param Cache            $cacheBackend       Cache.
	 * @param array            $allowedLanguages   Pole povolených jazyků. [['cs' => 'czech'], ...]
	 */
	public function __construct(EntityRepository $variableRepository, Cache $cacheBackend, array $allowedLanguages)
	{
		$this->_variableRepository = $variableRepository;
		$this->_cacheBackend = $cacheBackend;
		$this->_allowedLanguages = $allowedLanguages;
	}

	/**
	 * Získá číselník podle var_name (třeba nemovitost_typ) a jazyka (ve formátu locale - cs).
	 *
	 * @param string $varName Název (var_name) požadované proměnné. [nemovitost_typ].
	 * @param string $locale  Locale, ze které se získá jazyk. [cs]
	 *
	 * @return array Číselník k dané proměnné.
	 *
	 * @throws UnknownCodebookException Pokud codebook neexistuje.
	 * @throws UnknownLanguageException Pokud není jazyk (locale) podporován.
	 */
	public function getCodebook($varName, $locale)
	{
		$this->_checkLanguage($locale);

		$cacheKey = $varName . '_' . $locale;

		if ($this->_cacheBackend->contains($cacheKey)) {
			return $this->_cacheBackend->fetch($cacheKey);
		}

		$codebook = (array) $this->_retrieveCodebook($varName, $locale);

		$this->_cacheBackend->save($cacheKey, $codebook);

		return $codebook;
	}

	protected function _retrieveCodebook($varName, $locale)
    {
	    /** @var Variable $variable */
	    $qb = $this->_variableRepository->createQueryBuilder('v');
	    $qb->andWhere('v._var_name = :var_name')
		    ->setParameter('var_name', $varName)
	        ->andWhere('vlv._visible = :visible')
		    ->setParameter('visible', 1)
		    ->join('v.listValues', 'vlv')
	        ->orderBy('vlv._ord');
	    $variable = current($qb->getQuery()->getResult());
	    if ($variable) {
		    return $variable->getCodebook($this->_getVariablePropertyNameForLanguage($locale));
	    }

	    throw new UnknownCodebookException(sprintf('Neznámý číselník: "%s".', $varName));
    }

	/**
	 * Zkontroluje, zda je daný jazyk podporován. Pokud ne, vyhodí InvalidArgumentException.
	 *
	 * @param string $locale Kód jazyka (cs).
	 */
	private function _checkLanguage($locale)
	{
		if (!isset($this->_allowedLanguages[$locale])) {
			throw new UnknownLanguageException(
				sprintf(
					'Neznámý jazyk "%s". Podporované jazyky: "%s".',
					$locale,
					implode('", "', array_keys($this->_allowedLanguages))
				)
			);
		}
	}

	/**
	 * Vrátí název vlastnosti listValue pro daný jazyk.
	 *
	 * Data pro převod jsou brána z $this->_allowedLanguages.
	 * Před voáním se předpokládá ověření existence jazyka.
	 *
	 * Např. ('cs') => 'lang_czech'.
	 *
	 * @param string $language Kód jazyka (cs).
	 *
	 * @return string String lang_*.
	 */
	private function _getVariablePropertyNameForLanguage($language)
	{
		return 'label_' . $this->_allowedLanguages[$language];
	}
}
