<?php

namespace Dalten\WebBundle\Processor;

/**
 * Objekt starající se o výpočet a přepravu informací ohledně stránkování záznamů.
 */
class Paginator implements PaginatorIterface
{
	/**
	 * Číslo aktuální stránky.
	 *
	 * @var integer
	 */
	private $_current = 0;

	/**
	 * Celkový počet záznamů.
	 *
	 * @var integer
	 */
	private $_recordCount = 0;

	/**
	 * Počet záznamů na stránce.
	 *
	 * @var integer
	 */
	private $_recordsOnPage = 0;

	/**
	 * Načítat všechny záznamy bez stránkování.
	 *
	 * @var bool
	 */
	private $_loadAllRecords = false;

	/**
	 * Pole možných počtů záznamů na stránku.
	 *
	 * Pokud je pole prázdné, je povolené jakékoli číslo.
	 *
	 * @var array
	 */
	private $_allowedRecordsOnPage = array();

	/**
	 * Nastaví povolený počet záznamů na stránku.
	 *
	 * Pokud je nastaveno prázdné pole, není počet záznamů na stránku omezen.
	 *
	 * @param array $allowedRecordsOnPage Počet povolených záznamů na stránku (třeba [10, 20, 30]).
	 *
	 * @return \Dalten\Processor\Paginator Fluent interface.
	 */
	public function setAllowedRecordsOnPage(array $allowedRecordsOnPage)
	{
		$this->_allowedRecordsOnPage = $allowedRecordsOnPage;

		return $this;
	}

	/**
	 * Vrátí povolený počet záznamů na stránku.
	 *
	 * @return array Počet povolených záznamů na stránku (třeba [10, 20, 30]).
	 */
	public function getAllowedRecordsOnPage()
	{
		return $this->_allowedRecordsOnPage;
	}

	/**
	 * Určuje zda se mají načíst všechny položky bez ohledu na stránkování.
	 *
	 * @param bool $loadAll True pokud se mají načíst všechny položky, jinak false.
	 *
	 * @return self Fluent interface.
	 */
	public function setLoadAllRecords($loadAll)
	{
		$this->_loadAllRecords = (bool) $loadAll;

		return $this;
	}

	/**
	 * Vrátí aktuální stránku záznamu.
	 *
	 * @return integer
	 */
	public function getCurrent()
	{
		return $this->_current;
	}

	/**
	 * Nastaví aktuální stránku.
	 *
	 * @param integer $current Číslo aktuální stránky.
	 *
	 * @return self
	 */
	public function setCurrent($current)
	{
		$this->_current = (integer) $current;
		return $this;
	}

	/**
	 * Vrátí celkový počet záznamů v úložišti.
	 *
	 * @return integer
	 */
	public function getRecordCount()
	{
		return $this->_recordCount;
	}

	/**
	 * Nastaví počet záznamů v úložišti.
	 *
	 * @param integer $recordCount Číslo celkového počtu záznamů v úložišti.
	 *
	 * @return self
	 */
	public function setRecordCount($recordCount)
	{
		$this->_recordCount = (integer) $recordCount;
		return $this;
	}

	/**
	 * Vrátí požadovaný počet záznamů na stránku.
	 *
	 * @return integer
	 */
	public function getRecordsOnPage()
	{
		return $this->_recordsOnPage;
	}

	/**
	 * Nastaví požadovaný počet záznamů na stránku.
	 *
	 * Pokud předané číslo není v poli {@link $this->getAllowedRecordsOnPage()} a toto pole není prázdné,
	 * pak počet záznamů nastaven nebude.
	 *
	 * @param integer $recordsOnPage Požadovaný počet záznamů na stránku.
	 *
	 * @return self
	 */
	public function setRecordsOnPage($recordsOnPage)
	{
		$allowedRecordsOnPage = $this->getAllowedRecordsOnPage();
		if (empty($allowedRecordsOnPage) || in_array($recordsOnPage, $allowedRecordsOnPage)) {
			$this->_recordsOnPage = (integer) $recordsOnPage;
		}
		return $this;
	}

	/**
	 * Vrátí vypočtené číslo stránky v úložišti.
	 *
	 * @return integer
	 */
	public function getOffset()
	{
		$current = $this->_current ? : 1;
		return ($current -1) * $this->_recordsOnPage;
	}

	/**
	 * Vrátí vypočtený počet stránek v úložišti.
	 *
	 * @return integer
	 */
	public function getPageCount()
	{
		if ($this->_recordCount === 0 || $this->_recordsOnPage === 0) {
			return 0;
		}
		return (integer) ceil($this->_recordCount / $this->_recordsOnPage);
	}
}
