<?php
namespace Dalten\WebBundle\Translation\Extractor\Symfony;

use Symfony\Component\Translation\Extractor\ExtractorInterface;
use Symfony\Component\Translation\MessageCatalogue;

/**
 * Získává texty na překlad z víceroměrných polí.
 */
class RecursiveArrayValuesExtractor implements ExtractorInterface
{
	/**
	 * Překladový prefix.
	 *
	 * Dává se před nově zadané nepřeložené texty.
	 *
	 * @var string
	 */
	private $_prefix = '__';

	/**
	 * Nastavení polí na překlad.
	 *
	 * Jedná se o pole, jehož prvky jsou pole s klíči data a paths.
	 * * data = pole k překladu
	 * * paths = Pole cest v poli data k samotným textům. Pokud je prázdné, bude překládáno celé pole data.
	 *
	 * Příklad:
	 * { data: { test: { ahoj: bagr, nazdar: kuk } }, paths: [] } => přeloženo "bagr" a "kuk"
	 * { data: { test: { ahoj: bagr, nazdar: kuk } }, paths: [test.ahoj] } => přeloženo pouze "bagr"
	 *
	 * @var array
	 */
	private $_itemConfig;

	/**
	 * Nastavuje konfiguraci polí na překlad.
	 *
	 * $itemConfig je pole, jehož prvky jsou pole s klíči data a paths.
	 * * data = pole k překladu
	 * * paths = Pole cest v poli data k samotným textům. Pokud je prázdné, bude překládáno celé pole data.
	 *
	 * Příklad:
	 * { data: { test: { ahoj: bagr, nazdar: kuk } }, paths: [] } => přeloženo "bagr" a "kuk"
	 * { data: { test: { ahoj: bagr, nazdar: kuk } }, paths: [test.ahoj] } => přeloženo pouze "bagr"
	 *
	 * @param array $itemConfig Pole s konfigurací - viz popis metody.
	 */
	public function __construct(array $itemConfig)
	{
		$this->_itemConfig = $itemConfig;
	}

	/**
	 * Extracts translation messages from a template directory to the catalogue.
	 *
	 * @param string           $directory Adresář, odkud se berou překlad - v tomto případě je ignorován.
	 * @param MessageCatalogue $catalogue Katalog zpráv k překladu.
	 */
	public function extract($directory, MessageCatalogue $catalogue)
	{
		foreach ($this->_itemConfig as $config) {
			$catalogue->add($this->_extractTermsFromArray($config['data'], $config['paths']));
		}
	}

	/**
	 * Sets the prefix that should be used for new found messages.
	 *
	 * @param string $prefix The prefix.
	 */
	public function setPrefix($prefix)
	{
		$this->_prefix = $prefix;
	}

	/**
	 * Získá pole stringů na překlad ze zdrojového pole, které může filtrovat pomocí předaných cest.
	 *
	 * Popis funkcionality cest je v popisu konstruktoru.
	 *
	 * @param array|string $data  Data k překladu (obvykle pole, může být i string).
	 * @param array        $paths Seznam cest, na kterých se v poli nachází řetězce k překladu.
	 *
	 * @return array Texty k překladu.
	 */
	private function _extractTermsFromArray($data, array $paths = array())
	{
		$translations = array();
		$filteredData = $this->_filterDataByPaths($data, $paths);
		foreach ($filteredData as $item) {
			if (is_array($item)) {
				$translations = array_merge($translations, $this->_extractTermsFromArray($item, array()));
			} else {
				$translations[$item] = $this->_prefix . $item;
			}
		}

		return $translations;
	}

	/**
	 * Získá ze zdrojového pole texty na základě předaných cest.
	 *
	 * Popis funkcionality cest je v popisu konstruktoru.
	 *
	 * @param array|string $data  Data k překladu (obvykle pole, může být i string).
	 * @param array        $paths Seznam cest, na kterých se v poli nachází řetězce k překladu.
	 *
	 * @return array Texty k překladu.
	 *
	 * @throws \InvalidArgumentException Pokud se v poli nepodaří najít určenou cestu.
	 */
	private function _filterDataByPaths($data, $paths)
	{
		if (!is_array($data)) {
			$data = (array) $data;
		}

		if (empty($paths)) {
			return $data;
		}

		$filteredData = array();
		foreach ($paths as $path) {
			$item = $data;
			foreach(explode('.', $path) as $pathPart) {
				if (!isset($item[$pathPart])) {
					throw new \InvalidArgumentException(
						sprintf(
							'Nepodařilo se najít cestu "%s" (resp. její část "%s") v %s',
							$paths, $pathPart, print_r($data, true)
						)
					);
				}
				$item = $item[$pathPart];
			}
			$filteredData[] = $item;
		}

		return $filteredData;
	}
}
