<?php
namespace Dalten\WebBundle\Test\Breadcrumbs\Common\Builder;

use Dalten\WebBundle\Filter\AddressFilter;
use Dalten\WebBundle\Filter\ListingFilter;
use Dalten\WebBundle\Breadcrumbs\Breadcrumb;
use Dalten\WebBundle\Breadcrumbs\Common\Builder\CountyAndRegionHelper;

/**
 * Created by PhpStorm.
 * User: Karel
 * Date: 30. 10. 2014
 * Time: 14:43
 */
class CountyAndRegionHelperTest extends \PHPUnit_Framework_TestCase
{
	private $_countyIdToSlugMap = array(7 => 'bagr', 9 => 'dira', 11 => 'ahoj');
	private $_countyToRegionLabelMap = array(
		7 => array(1 => 'Bum Prsk', 2 => 'rum'),
		9 => array(4 => 'Cajicek', 1 => 'pivo'),
		11 => array(9 => 'ugh')
	);

	public function testBuilderReturnsNullIfRegionCannotBeMapped()
	{
		$countyCode = 7;
		$builder = new CountyAndRegionHelper($this->_countyIdToSlugMap, $this->_countyToRegionLabelMap);
		$previous = new Breadcrumb('Test', 'test', array('county' => 'bagr'));

		$this->assertNull($builder->getBreadcrumb($previous, array($countyCode), 3));
	}

	public function testBuilderReturnsNullIfPreviousBreadcrumbParamsDoNotContainCountyKey()
	{
		$countyCode = 7;
		$builder = new CountyAndRegionHelper($this->_countyIdToSlugMap, $this->_countyToRegionLabelMap);
		$previous = new Breadcrumb('Test', 'test', array());

		$this->assertNull($builder->getBreadcrumb($previous, array($countyCode), 3));
	}

	public function testBuilderCreatesBreadcrumbWithCorrectLabelAndSlug()
	{
		$countyCode = 7;
		$regionCode = 1;
		$routeName = 'test';
		$builder = new CountyAndRegionHelper($this->_countyIdToSlugMap, $this->_countyToRegionLabelMap);
		$previous = new Breadcrumb('Test', $routeName, array('county' => 'bagr'));

		$crumb = $builder->getBreadcrumb($previous, array($countyCode), $regionCode);
		$this->assertInstanceOf('Dalten\WebBundle\Breadcrumbs\Breadcrumb', $crumb);
		$this->assertSame(array('county' => 'bagr', 'region' => 'bum-prsk'), $crumb->getRouteParams());
		$this->assertSame($routeName . 'AndRegion', $crumb->getRouteName());
		$this->assertSame('Bum Prsk', $crumb->getLabel());
	}

	public function testBuilderAddsCountySlugToExistingRouteParams()
	{
		$countyCode = 7;
		$regionCode = 1;
		$routeName = 'test';
		$builder = new CountyAndRegionHelper($this->_countyIdToSlugMap, $this->_countyToRegionLabelMap);
		$previous = new Breadcrumb('Test', $routeName, array('county' => 'bagr', 'x' => 'y'));

		$crumb = $builder->getBreadcrumb($previous, array($countyCode), $regionCode);
		$this->assertInstanceOf('Dalten\WebBundle\Breadcrumbs\Breadcrumb', $crumb);
		$this->assertSame(array('county' => 'bagr', 'x' => 'y', 'region' => 'bum-prsk'), $crumb->getRouteParams());
	}

	public function testBuilderMatchesRegionToCountyFromPreviousBreadcrumb()
	{
		$countyCode = 7;
		$regionCode = 1;
		$routeName = 'test';
		$builder = new CountyAndRegionHelper($this->_countyIdToSlugMap, $this->_countyToRegionLabelMap);
		$previous = new Breadcrumb('Test', $routeName, array('county' => 'bagr'));

		$crumb = $builder->getBreadcrumb($previous, array(11, 9, 99, $countyCode), $regionCode);
		$this->assertInstanceOf('Dalten\WebBundle\Breadcrumbs\Breadcrumb', $crumb);
		$this->assertSame(array('county' => 'bagr', 'region' => 'bum-prsk'), $crumb->getRouteParams());
	}

	public function testBuilderReturnsNullIfRegionIsNotBelongingToCountyFromPreviousBreadcrumb()
	{
		$countyCode = 7;
		$regionCode = 9;
		$routeName = 'test';
		$builder = new CountyAndRegionHelper($this->_countyIdToSlugMap, $this->_countyToRegionLabelMap);
		$previous = new Breadcrumb('Test', $routeName, array('county' => 'bagr'));

		$crumb = $builder->getBreadcrumb($previous, array($countyCode), $regionCode);
		$this->assertNull($crumb);
	}

	public function testBuilderCanCreateBreadcrumbFromFilter()
	{
		$countyCode = 7;
		$regionCode = 1;
		$routeName = 'test';
		$filter = new ListingFilter(array(), array(new AddressFilter($countyCode, array($regionCode))));
		$builder = new CountyAndRegionHelper($this->_countyIdToSlugMap, $this->_countyToRegionLabelMap);
		$previous = new Breadcrumb('Test', $routeName, array('county' => 'bagr'));

		$crumb = $builder->getBreadcrumbFromFilter($filter, $previous);
		$this->assertInstanceOf('Dalten\WebBundle\Breadcrumbs\Breadcrumb', $crumb);
		$this->assertSame(array('county' => 'bagr', 'region' => 'bum-prsk'), $crumb->getRouteParams());
		$this->assertSame($routeName . 'AndRegion', $crumb->getRouteName());
		$this->assertSame('Bum Prsk', $crumb->getLabel());
	}

	public function testBuilderUsesFirstFoundCountyCodeFromFilter()
	{
		$countyCode = 7;
		$regionCode = 1;
		$routeName = 'test';
		$filter = new ListingFilter(
			array(),
			array(new AddressFilter($countyCode, array()), new AddressFilter($countyCode, array($regionCode, 99)))
		);
		$builder = new CountyAndRegionHelper($this->_countyIdToSlugMap, $this->_countyToRegionLabelMap);
		$previous = new Breadcrumb('Test', $routeName, array('county' => 'bagr'));

		$crumb = $builder->getBreadcrumbFromFilter($filter, $previous);
		$this->assertInstanceOf('Dalten\WebBundle\Breadcrumbs\Breadcrumb', $crumb);
		$this->assertSame(array('county' => 'bagr', 'region' => 'bum-prsk'), $crumb->getRouteParams());
		$this->assertSame($routeName . 'AndRegion', $crumb->getRouteName());
		$this->assertSame('Bum Prsk', $crumb->getLabel());
	}
}
