<?php
namespace Dalten\WebBundle\Entity;

use Dalten\WebBundle\Entity\Exception\InvalidLegacyPropertyException;
use Dalten\WebBundle\Entity\Exception\InvalidPropertyException;

/**
 * Trait, umožňující pracovat zároveň se starými (DB) a novými (kód) názvy proměnných.
 */
trait LegacyDataMethods
{
	public function __get($property)
	{
		if ($property === 'id' && property_exists($this, 'id')) {
			return $this->id;
		}

		if (isset(self::$_map[$property])) {
			return $this->{self::$_map[$property]};
		}

		throw new InvalidPropertyException(sprintf('No property "%s" to get', $property));
	}

	public function __set($property, $value)
	{
		if ($property !== 'id' && isset(self::$_map[$property]) && isset(self::$_map[$property])) {
			$this->{self::$_map[$property]} = $value;

			return $this;
		}

		throw new InvalidPropertyException(sprintf('No property "%s" to set', $property));
	}

	public function __isset($property)
	{
		if (isset(self::$_map[$property]) || (property_exists($this, $property) && $property === 'id')) {
			return true;
		}

		return false;
	}

	/**
	 * Zjistí, zda existuje daná "stara" (z DB) proměnná. Alternativa isset.
	 *
	 * @param string $property název "staré" proměnné.
	 *
	 * @return bool Existuje tato legacy proměnná?
	 */
	public function hasLegacyProperty($property)
	{
		$property = '_' . str_replace('-', '_', $property);

		return property_exists($this, $property);
	}

	/**
	 * Získá hodnotu "staré" (z DB) proměnné.
	 *
	 * @param string $property název "staré" proměnné.
	 *
	 * @return mixed Hodnota proměnné.
	 */
	public function getLegacyProperty($property)
	{
		if ($this->hasLegacyProperty($property)) {
			$prefixedProperty = '_' . str_replace('-', '_', $property);
			return $this->$prefixedProperty;
		}

		throw new InvalidLegacyPropertyException(sprintf('No legacy property "%s" to get', $property));
	}

	/**
	 * Nastaví hodnotu "staré" (z DB) proměnné.
	 *
	 * @param string $property Název "staré" proměnné.
	 * @param mixed  $value    Hodnota proměnné.
	 *
	 * @return $this
	 */
	public function setLegacyProperty($property, $value)
	{
		$prefixedProperty = '_' . str_replace('-', '_', $property);
		if ($this->hasLegacyProperty($property)) {
			$this->$prefixedProperty = $value;

			return $this;
		}

		throw new InvalidLegacyPropertyException(sprintf('No legacy property "%s" to set', $property));
	}
}
