<?php
namespace Dalten\WebBundle\Filter;

use Dalten\WebBundle\Filter\Exception\InvalidAddressFilter;

/**
 * Normalizovaný filtr na vyhledávání nabídek.
 *
 * @property int[]  $advert_type              Typ inzerátů.
 * @property int[]  $listing_type             Typ nemovitosti.
 * @property int[]  $ownership                Vlastnictví.
 * @property int    $equipment                Vybaveno?
 * @property int[]  $building_type            Typ budovy (cihlový, panelový, atp.).
 * @property int[]  $transport                Dopravní dostupnost.
 * @property int    $main_area_min            Plocha od.
 * @property int    $main_area_max            Plocha do.
 * @property int    $floor_min                Podlaží od.
 * @property int    $floor_max                Podlaží do.
 * @property float  $price_min                Cena min.
 * @property float  $price_max                Cena max.
 * @property int    $price_unit               Jednotka ceny.
 * @property int    $price_currency           Měna ceny.
 * @property int[]  $commercial_kind          Typ komerčního objektu.
 * @property int[]  $estate_kind              Typ pozemku.
 * @property int[]  $flat_kind                Typ bytu.
 * @property int[]  $object_kind_houses       Poloha objektu (domy).
 * @property int[]  $object_kind_cottages     Poloha objektu (chaty a chalupy).
 * @property int[]  $object_kind_smallobjects Poloha objektu (malé objekty).
 * @property int[]  $office_kind              Typ komerčního prostoru.
 * @property int[]  $hotel_kind               Typ HORECA zařízení.
 * @property string $locality_text            Název (nebo část) lokality.
 * @property int[]  $building_condition       Stav budovy.
 * @property int    $discount_duration        Počet dní od snížení ceny.
 * @property array  $locality_country_code    Omezení na státy.
 * @property bool   $only_projects            Omezení pouze na developerské projekty.
 * @property array  $company_id               Omezení na firmy.
 * @property array  $broker_id                Omezení na makléře.
 * @property bool   $has_open_day             Omezení pouze na nemovitosti s dny otevřených dveří.
 * @property int    $elevator                 Výtah.
 * @property int    $garage                   Garáž?
 * @property int    $terrace                  Terasa?
 * @property int    $balcony                  Balkon?
 * @property int    $loggia                   Lodzie?
 * @property int    $parking_lots             Parkovací stání?
 * @property int    $plot_area_min            Plocha pozemku od.
 * @property int    $plot_area_max            Plocha pozemku do.
 */
class ListingFilter extends Base implements FilterWithAddressInterface
{
	/**
	 * Asociativní pole názvů (klíče) a datových typů (hodnoty) prvků filtru.
	 *
	 * @var array
	 */
	private $_properties = array(
		'code' => 'int',
		'advert_type' => 'multiOption',
		'listing_type' => 'multiOption',
		'ownership' => 'multiOption',
		'equipment' => 'int',
		'building_type' => 'multiOption',
		'building_condition' => 'multiOption',
		'transport' => 'multiOption',
		'main_area_min' => 'int',
		'main_area_max' => 'int',
		'floor_min' => 'int',
		'floor_max' => 'int',
		'price_min' => 'float',
		'price_max' => 'float',
		'price_unit' => 'int',
		'price_currency' => 'int',
		'commercial_kind' => 'multiOption',
		'commercial_type' => 'multiOption',
		'estate_kind' => 'multiOption',
		'flat_kind' => 'multiOption',
		'object_kind_houses' => 'multiOption',
		'object_kind_cottages' => 'multiOption',
		'object_kind_smallobjects' => 'multiOption',
		'office_kind' => 'multiOption',
		'hotel_kind' => 'multiOption',
		'locality_text' => 'string',
		'discount_duration' => 'int',
		'locality_country_code' => 'array',
		'company_id' => 'int[]',
		'broker_id' => 'int[]',
		'only_projects' => 'bool',
		'has_open_day' => 'bool',
		'elevator' => 'int',
		'garage' => 'int',
		'terrace' => 'int',
		'balcony' => 'int',
		'loggia' => 'int',
		'parking_lots' => 'int',
		'plot_area_min' => 'int',
		'plot_area_max' => 'int',
	);

	/**
	 * Pole předaných adresních filtrů.
	 *
	 * @var AddressFilter[]
	 */
	private $_addressFilters = array();

	/**
	 * Inicializuje data filtru a uloží předané adresní filtry.
	 *
	 * @param array           $data           Pole (již přeložených) dat filtru.
	 * @param AddressFilter[] $addressFilters Pole naplněných adresních filtrů.
	 *
	 * @throws InvalidAddressFilter Pokud je v poli $addressFilters jiný prvek než třídy AddressFilter.
	 */
	public function __construct(array $data, array $addressFilters = array())
	{
		$normalizedData = $this->_normalizeData($this->_properties, $data);

		foreach ($addressFilters as $filter) {
			if (!($filter instanceof AddressFilter)) {
				throw new InvalidAddressFilter('Předané adresní filtry musí být instancemi AddressFilter');
			}
		}
		$this->_addressFilters = $addressFilters;

		parent::__construct($normalizedData);
	}

	/**
	 * Vrátí předané adresní filtry.
	 *
	 * @return AddressFilter[]
	 */
	public function getAddressFilters()
	{
		return $this->_addressFilters;
	}
}
