<?php
namespace Dalten\WebBundle\Processor\ListingListData\Components;

use Dalten\WebBundle\Entity\Listing;
use Dalten\WebBundle\Filter\AddressFilter;
use Dalten\WebBundle\Filter\ListingFilter;
use Dalten\WebBundle\Processor\ListingListData\Data;
use Doctrine\ORM\QueryBuilder;

/**
 * Jednoduchá komponenta, obstarávající přípravu Querysetu na základě předaných ListDat.
 */
class FilterToQueryBuilderConverter
{
	/**
	 * Mapa typů nemovitostí na názvy databázových polí s jejich hlavní plochou.
	 *
	 * @var array
	 */
	private static $databaseAreaMappings = array(
		'_nemovitost_plocha_uzitna' => array(1, 5, 6, 7, 8),
		'_nemovitost_plocha_celkova' => array(2, 3, 4),
		'_nemovitost_plocha_kancelari' => array(9),
		'_nemovitost_plocha_zastavena' => array(10, 11),
	);

	/**
	 * Přidá do queryBuilderu order a základní where (společná pro všechny výpisy).
	 *
	 * @param QueryBuilder $queryBuilder Query Builder.
	 * @param Data         $listData     ListData s definicí řazení.
	 * @param string       $listingAlias Alias entity Listing v QB.
	 *
	 * @return QueryBuilder
	 */
	public function prepareQueryBuilder(QueryBuilder $queryBuilder, Data $listData, $listingAlias)
	{
		$prefix = $listingAlias . '.';
		$queryBuilder->andWhere($prefix . '_zakazka_stav IN (:advert_status)');
		$queryBuilder->setParameter(
				'advert_status',
				[Listing::ADVERT_STATUS_ACTIVE, Listing::ADVERT_STATUS_RESERVED, Listing::ADVERT_STATUS_SOLD_OR_LEASED]
		);

		if ($listData->isFavorite()) {
			$queryBuilder->andWhere($prefix . 'id IN (:favorite_listings)');
			$queryBuilder->setParameter('favorite_listings', $listData->getFavoriteListingIds());
		}

		if ($listData->isOpenDay()) {
			$queryBuilder->orderBy($prefix . '_nemovitost_openhouse_datum', 'DESC');
		}

		$direction = $listData->isInAscendingOrder() ? 'ASC' : 'DESC';
		if ($listData->isOrderedByPrice()) {
			if ($listData->isInAscendingOrder()) {
				$queryBuilder->addOrderBy($prefix . '_nemovitost_cena_pronajem', $direction);
				$queryBuilder->addOrderBy($prefix . '_nemovitost_cena_prodej', $direction);
			} else {
				$queryBuilder->addOrderBy($prefix . '_nemovitost_cena_prodej', $direction);
				$queryBuilder->addOrderBy($prefix . '_nemovitost_cena_pronajem', $direction);
			}
		} elseif ($listData->isOrderedByPublishedDate()) {
			$queryBuilder->addOrderBy($prefix . '_zakazka_datum_vlozeno', $direction);
		} else {
			$queryBuilder->addOrderBy($prefix . 'id', $direction);
		}

		return $queryBuilder;
	}

	/**
	 * Přidá omezení z filtru do QueryBuilderu.
	 *
	 * @param QueryBuilder  $queryBuilder QB z nabídek.
	 * @param ListingFilter $filter       Filtr nabídek.
	 * @param string        $listingAlias Alias entity Listing v QB.
	 *
	 * @return QueryBuilder
	 */
	public function addFilterToQueryBuilder(QueryBuilder $queryBuilder, ListingFilter $filter, $listingAlias)
	{
		$prefix = $listingAlias . '.';

		if ($filter->advert_type) {
			$queryBuilder->andWhere($prefix . '_zakazka_typ IN (:advert_type)');
			$queryBuilder->setParameter('advert_type', $filter->advert_type);
		}

		if ($filter->listing_type) {
			$this->_addListingTypeRule($filter, $queryBuilder, $prefix);
		}

		if ($filter->ownership) {
			$queryBuilder->andWhere($prefix . '_nemovitost_vlastnictvi IN (:ownership)');
			$queryBuilder->setParameter('ownership', $filter->ownership);
		}

		if ($filter->equipment) {
			$queryBuilder->andWhere($prefix . '_nemovitost_vybaveno = :equipment');
			$queryBuilder->setParameter('equipment', $filter->equipment);
		}

		if ($filter->elevator) {
			$queryBuilder->andWhere($prefix . '_nemovitost_vytah = :elevator');
			$queryBuilder->setParameter('elevator', $filter->elevator);
		}

		if ($filter->garage) {
			$queryBuilder->andWhere($prefix . '_nemovitost_garaz = :garage');
			$queryBuilder->setParameter('garage', $filter->garage);
		}

		if ($filter->balcony) {
			$queryBuilder->andWhere($prefix . '_nemovitost_balkon = :balcony');
			$queryBuilder->setParameter('balcony', $filter->balcony);
		}

		if ($filter->terrace) {
			$queryBuilder->andWhere($prefix . '_nemovitost_terasa = :terrace');
			$queryBuilder->setParameter('terrace', $filter->terrace);
		}

        if ($filter->loggia) {
            $queryBuilder->andWhere($prefix . '_nemovitost_lodzie = :loggia');
            $queryBuilder->setParameter('loggia', $filter->loggia);
        }

		if ($filter->parking_lots) {
			$queryBuilder->andWhere($prefix . '_pocet_mist_k_parkovani >= :parking_lots');
			$queryBuilder->setParameter('parking_lots', $filter->parking_lots);
		}

		if ($filter->building_condition) {
			$queryBuilder->andWhere($prefix . '_nemovitost_stav_objektu IN (:building_condition)');
			$queryBuilder->setParameter('building_condition', $filter->building_condition);
		}

		if ($filter->building_type) {
			$queryBuilder->andWhere($prefix . '_nemovitost_druh_objektu IN (:building_type)');
			$queryBuilder->setParameter('building_type', $filter->building_type);
		}

		if ($filter->transport) {
			$queryBuilder->andWhere($prefix . '_nemovitost_doprava IN (:transport)');
			$queryBuilder->setParameter('transport', $filter->transport);
		}

		if ($filter->floor_min) {
			$queryBuilder->andWhere($prefix . '_nemovitost_podlazi_cislo >= :floor_min');
			$queryBuilder->setParameter('floor_min', $filter->floor_min);
		}

		if ($filter->floor_max) {
			$queryBuilder->andWhere($prefix . '_nemovitost_podlazi_cislo <= :floor_max');
			$queryBuilder->setParameter('floor_max', $filter->floor_max);
		}

		if ($filter->locality_country_code) {
			$queryBuilder->andWhere($prefix . '_nemovitost_uir_stat IN (:locality_country_code)');
			$queryBuilder->setParameter('locality_country_code', $filter->locality_country_code);
		}

        if ($filter->only_projects) {
			$queryBuilder->andWhere('_id_projekt > 0');
        }

        if ($filter->company_id) {
			$queryBuilder->andWhere($prefix . '_id_firma IN (:company_id)');
			$queryBuilder->setParameter('company_id', $filter->company_id);
        }

        if ($filter->broker_id) {
			$queryBuilder->andWhere($prefix . '_id_uzivatel IN (:broker_id)');
			$queryBuilder->setParameter('broker_id', $filter->broker_id);
        }

		$matches = array();

		if (preg_match('/^N(?P<listing_code>\d{1,5})$/', $filter->locality_text, $matches)) {
			$listingCode = (int) $matches['listing_code'];

			if ($listingCode) {
				$queryBuilder->andWhere($prefix . '_cislo <= :advert_code');
				$queryBuilder->setParameter('advert_code', $listingCode);
			}
		} elseif ($filter->locality_text) {
			$expr = $queryBuilder->expr();

			$condition = $expr->orX(
				$expr->like($prefix . '_nemovitost_uir_obec_text', ':locality_text'),
				$expr->like($prefix . '_nemovitost_uir_cobce_text', ':locality_text'),
				$expr->like($prefix . '_nemovitost_uir_mcast_text', ':locality_text'),
				$expr->like($prefix . '_nemovitost_uir_pobvod_text', ':locality_text'),
				$expr->like($prefix . '_nemovitost_uir_ulice_text', ':locality_text')
			);

			$queryBuilder->andWhere($condition)->setParameter('locality_text', '%' . $filter->locality_text . '%');
		}

		if ($filter->price_min > 0 || $filter->price_max > 0) {
			$this->_addPriceRule($filter->price_min, $filter->price_max, $queryBuilder, $prefix);
		}
		if ($filter->main_area_min > 0 || $filter->main_area_max > 0) {
			$this->_addAreaRule($filter->main_area_min, $filter->main_area_max, $queryBuilder, $prefix);
		}

		if ($filter->getAddressFilters()) {
			$this->_addAddressRules($filter->getAddressFilters(), $queryBuilder, $prefix);
		}

		return $queryBuilder;
	}

	/**
	 * Vytvoří pravidlo na omezení plochy.
	 *
	 * Pokud jsou obě předané plochy (min a max) menší nebo rovny 0, pak vyhodí výjimku.
	 *
	 * @param int          $minArea      Minimální plocha.
	 * @param int          $maxArea      Maximální plocha.
	 * @param QueryBuilder $queryBuilder QB k doplnění podmínek.
	 * @param string       $prefix       Prefix entity nabídky (vč. separatoru).
	 *
	 * @return QueryBuilder Předaný QueryBuilder.
	 */
	private function _addAreaRule($minArea, $maxArea, QueryBuilder $queryBuilder, $prefix)
	{
		if ($minArea <= 0 && $maxArea <= 0) {
			throw new \InvalidArgumentException('Musí být nastavena alespoň jedna plocha');
		}

		$expr = $queryBuilder->expr();

		$orWhere = $expr->orX();

		foreach (self::$databaseAreaMappings as $areaProperty => $typeIds) {
			$paramPrefix = 'area_' . $areaProperty;
			$areaProperty = $prefix . $areaProperty;
			$queryBuilder->setParameter($paramPrefix . '_listing_type', $typeIds);
			if ($minArea && $maxArea) {
				$orWhere->add(
					$expr->andX(
						$expr->in($prefix . '_nemovitost_typ', ':' . $paramPrefix . '_listing_type'),
						$expr->gte($areaProperty, ':' . $paramPrefix . '_min_area'),
						$expr->lte($areaProperty, ':' . $paramPrefix . '_max_area')
					)
				);
				$queryBuilder->setParameter($paramPrefix . '_min_area', $minArea);
				$queryBuilder->setParameter($paramPrefix . '_max_area', $maxArea);
			} elseif ($minArea) {
				$orWhere->add(
					$expr->andX(
						$expr->in($prefix . '_nemovitost_typ', ':' . $paramPrefix . '_listing_type'),
						$expr->gte($areaProperty, ':' . $paramPrefix . '_min_area')
					)
				);
				$queryBuilder->setParameter($paramPrefix . '_min_area', $minArea);
			} elseif ($maxArea) {
				$orWhere->add(
					$expr->andX(
						$expr->in($prefix . '_nemovitost_typ', ':' . $paramPrefix . '_listing_type'),
						$expr->lte($areaProperty, ':' . $paramPrefix . '_max_area')
					)
				);
				$queryBuilder->setParameter($paramPrefix . '_max_area', $maxArea);
			}
		}

		$queryBuilder->andWhere($orWhere);
	}

	/**
	 * Vytvoří pravidlo na omezení ceny (pokud není vybrán typ zakázky.
	 *
	 * Pokud je zvolen typ zakázky, pak duplikuje funkcionalitu _createAdvertTypeRule.
	 *
	 * Pokud jsou obě předané ceny (min a max) menší nebo rovny 0, pak vyhodí výjimku.
	 *
	 * @param int          $minPrice     Minimální cena.
	 * @param int          $maxPrice     Maximální cena.
	 * @param QueryBuilder $queryBuilder QB k doplnění podmínek.
	 * @param string       $prefix       Prefix entity nabídky (vč. separatoru).
	 *
	 * @return QueryBuilder Předaný QueryBuilder.
	 */
	private function _addPriceRule($minPrice, $maxPrice, QueryBuilder $queryBuilder, $prefix)
	{
		if ($minPrice <= 0 && $maxPrice <= 0) {
			throw new \InvalidArgumentException('Musí být nastavena alespoň jedna cena');
		}

		$expr = $queryBuilder->expr();

		if ($minPrice) {
			$queryBuilder->setParameter('advert_price_min', $minPrice);
			$queryBuilder->andWhere(
				$expr->orx(
					$expr->andX(
						$expr->eq($prefix . '_zakazka_typ', 1),
						$expr->gte($prefix . '_nemovitost_cena_prodej', ':advert_price_min')
					),
					$expr->andX(
						$expr->eq($prefix . '_zakazka_typ', 2),
						$expr->gte($prefix . '_nemovitost_cena_pronajem', ':advert_price_min')
					)
				)
			);
		}

		if ($maxPrice) {
			$queryBuilder->setParameter('advert_price_max', $maxPrice);
			$queryBuilder->andWhere(
				$expr->orX(
					$expr->andX(
						$expr->eq($prefix . '_zakazka_typ', 1),
						$expr->lte($prefix . '_nemovitost_cena_prodej', ':advert_price_max')
					),
					$expr->andX(
						$expr->eq($prefix . '_zakazka_typ', 2),
						$expr->lte($prefix . '_nemovitost_cena_pronajem', ':advert_price_max')
					)
				)
			);

		}

		return $queryBuilder;
	}

	/**
	 * Vytvoří omezení na adresu.
	 *
	 * @param AddressFilter[] $addressFilters Omezení na adresu.
	 * @param QueryBuilder    $queryBuilder   QB k doplnění podmínek.
	 * @param string          $prefix         Prefix entity nabídky (vč. separatoru).
	 *
	 * @return QueryBuilder Předaný QueryBuilder.
	 */
	private function _addAddressRules(array $addressFilters, QueryBuilder $queryBuilder, $prefix)
	{
		$expr = $queryBuilder->expr();
		$rules = array();
		$countyColumn = $prefix . '_nemovitost_uir_kraj_kod';

		foreach ($addressFilters as $index => $filter) {
			$regionColumn = $prefix . '_nemovitost_uir_okres_kod';
			if ($filter->county_code === 19) {
				$regionColumn = $prefix . '_nemovitost_uir_pobvod_kod';
			}
			$queryBuilder->setParameter('locality_county_' . $index, $filter->county_code);
			if ($filter->region_codes) {
			$queryBuilder->setParameter('locality_region_' . $index, $filter->region_codes);
				$rules[] = $expr->andX(
					$expr->eq($countyColumn, ':locality_county_' . $index),
					$expr->in($regionColumn, ':locality_region_' . $index)
				);
			} else {
				$rules[] = $expr->eq($countyColumn, ':locality_county_' . $index);
			}
		}

		if ($rules) {
			$queryBuilder->andWhere(call_user_func_array(array($expr, 'orX'), $rules));
		}

		return $queryBuilder;
	}

	/**
	 * Vytvoří omezení na typ a podtyp nemovitosti.
	 *
	 * @param ListingFilter $filter       Filtr nabídek.
	 * @param QueryBuilder  $queryBuilder QB k doplnění podmínek.
	 * @param string        $prefix       Prefix entity nabídky (vč. separatoru).
	 *
	 * @return QueryBuilder Předaný QueryBuilder.
	 */
	private function _addListingTypeRule(ListingFilter $filter, QueryBuilder $queryBuilder, $prefix)
	{
		$expr = $queryBuilder->expr();
		$rules = array();
		foreach ($filter->listing_type as $listingType) {
			$queryBuilder->setParameter('listing_type_' . $listingType, $listingType);
			switch ($listingType) {
				case 2: // Komerční objekty
					if ($filter->commercial_kind) {
						$queryBuilder->setParameter('listing_subtype_' . $listingType, $filter->commercial_kind);
						$rules[] = $expr->andX(
							$expr->eq($prefix . '_nemovitost_typ', ':listing_type_' . $listingType),
							$expr->in($prefix . '_nemovitost_ucel_budovy', ':listing_subtype_' . $listingType)
						);
					} else {
						$rules[] = $expr->eq($prefix . '_nemovitost_typ', ':listing_type_' . $listingType);
					}
					break;
				case 3: // Pozemky
					if ($filter->estate_kind) {
						$queryBuilder->setParameter('listing_subtype_' . $listingType, $filter->estate_kind);
						$rules[] = $expr->andX(
							$expr->eq($prefix . '_nemovitost_typ', ':listing_type_' . $listingType),
							$expr->in($prefix . '_nemovitost_druh_pozemku', ':listing_subtype_' . $listingType)
						);
					} else {
						$rules[] = $expr->eq($prefix . '_nemovitost_typ', ':listing_type_' . $listingType);
					}
					break;
				case 4: // byty
					if ($filter->flat_kind) {
						$queryBuilder->setParameter('listing_subtype_' . $listingType, $filter->flat_kind);
						$rules[] = $expr->andX(
							$expr->eq($prefix . '_nemovitost_typ', ':listing_type_' . $listingType),
							$expr->in($prefix . '_nemovitost_dispozice', ':listing_subtype_' . $listingType)
						);
					} else {
						$rules[] = $expr->eq($prefix . '_nemovitost_typ', ':listing_type_' . $listingType);
					}
					break;
				case 6: // Domy a vily - nemovitost_poloha_objektu
					if ($filter->object_kind_houses) {
						$queryBuilder->setParameter('listing_subtype_' . $listingType, $filter->object_kind_houses);
						$rules[] = $expr->andX(
							$expr->eq($prefix . '_nemovitost_typ', ':listing_type_' . $listingType),
							$expr->in($prefix . '_nemovitost_poloha_objektu', ':listing_subtype_' . $listingType)
						);
					} else {
						$rules[] = $expr->eq($prefix . '_nemovitost_typ', ':listing_type_' . $listingType);
					}
					break;
				case 7: // Hotely, penziony a restaurace
					if ($filter->hotel_kind) {
						$queryBuilder->setParameter('listing_subtype_' . $listingType, $filter->hotel_kind);
						$rules[] = $expr->andX(
							$expr->eq($prefix . '_nemovitost_typ', ':listing_type_' . $listingType),
							$expr->in($prefix . '_nemovitost_typ_zarizeni', ':listing_subtype_' . $listingType)
						);
					} else {
						$rules[] = $expr->eq($prefix . '_nemovitost_typ', ':listing_type_' . $listingType);
					}
					break;
				case 9: // Komerční prostory
					if ($filter->office_kind) {
						$queryBuilder->setParameter('listing_subtype_' . $listingType, $filter->office_kind);
						$rules[] = $expr->andX(
							$expr->eq($prefix . '_nemovitost_typ', ':listing_type_' . $listingType),
							$expr->in($prefix . '_nemovitost_druh_prostor', ':listing_subtype_' . $listingType)
						);
					} else {
						$rules[] = $expr->eq($prefix . '_nemovitost_typ', ':listing_type_' . $listingType);
					}
					break;
				case 10: // Chaty a rekreační objekty
					if ($filter->object_kind_cottages) {
						$queryBuilder->setParameter('listing_subtype_' . $listingType, $filter->object_kind_cottages);
						$rules[] = $expr->andX(
							$expr->eq($prefix . '_nemovitost_typ', ':listing_type_' . $listingType),
							$expr->in($prefix . '_nemovitost_poloha_objektu', ':listing_subtype_' . $listingType)
						);
					} else {
						$rules[] = $expr->eq($prefix . '_nemovitost_typ', ':listing_type_' . $listingType);
					}
					break;
				case 11: // Malé objekty, garáže
					if ($filter->object_kind_smallobjects) {
						$queryBuilder->setParameter(
							'listing_subtype_' . $listingType, $filter->object_kind_smallobjects
						);
						$rules[] = $expr->andX(
							$expr->eq($prefix . '_nemovitost_typ', ':listing_type_' . $listingType),
							$expr->in($prefix . '_nemovitost_poloha_objektu', ':listing_subtype_' . $listingType)
						);
					} else {
						$rules[] = $expr->eq($prefix . '_nemovitost_typ', ':listing_type_' . $listingType);
					}
					break;
				default: // Typy bez podtypů (
					// zemědělské objekty = 1
					// Historické objekty = 5
					// Nájemní domy = 8
					$rules[] = $expr->eq($prefix . '_nemovitost_typ', ':listing_type_' . $listingType);
					break;

			}
		}

		if ($rules) {
			$queryBuilder->andWhere(call_user_func_array(array($expr, 'orX'), $rules));
		}

		return $queryBuilder;
	}
}
